package jp.ill.photon.module.directory;

import java.io.File;
import java.io.FileFilter;
import java.math.BigDecimal;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.CheckUtil;
import jp.ill.photon.util.DateUtil;
import jp.ill.photon.util.StringUtil;

/**
 * 
 * 検索対象ディレクトリに存在するファイル、フォルダの一覧を取得します<br />
 * <br />
 * 入力: <br/>
 * dirPath: 【必須】検索対象ディレクトリ<br/>
 * dirFileFlg: 一覧取得対象 0:ファイル 1:ディレクトリ 左記以外:区別なし<br/>
 * page: ページ番号<br/>
 * pageMaxCount: ページ最大件数<br/>
 * fileName: 検索：ファイル名<br/>
 * fromDate: 検索：アップロード日(始)<br/>
 * toDate: 検索：アップロード日(至)<br/>
 * sort: ソート<br/>
 * <br/>
 * 戻り値: <br />
 * dirPath: 対象ディレクトリ(String)<br/>
 * dirFileFlg: 一覧取得対象(String)<br/>
 * absolutePathList: フルパスの一覧(ArrayList{@literal<Object>})<br/>
 * nameList: ファイル、フォルダの名前一覧(ArrayList{@literal<Object>})<br/>
 * fileList: ファイル、フォルダの名前、サイズ一覧(ArrayList{@literal<Object>})<br/>
 * contentCnt: 対象ディレクトリにある要素数(Integer)<br/>
 * 
 */
public class GetDirectoryContentsModule implements PhotonModule {

	/** 対象ディレクトリ */
	@ModuleParam(required = true)
	private String dirPath;

	/** 一覧取得対象 0:ファイル 1:ディレクトリ 左記以外:区別なし */
	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "static", transferVal = "")
	private String dirFileFlg;

	/** ページ番号 */
	@ModuleParam(required = false)
	private Integer page;

	/** ページ最大件数 */
	@ModuleParam(required = false)
	private Integer pageMaxCount;

	/** 検索：ファイル名 */
	@ModuleParam(required = false)
	private String fileName;

	/** 検索：アップロード日(始) */
	@ModuleParam(required = false)
	private String fromDate;

	/** 検索：アップロード日(至) */
	@ModuleParam(required = false)
	private String toDate;

	/** 検索：ソート */
	@ModuleParam(required = false)
	private String sort;

	/** 検索：ソート順マップ */
	@ModuleParam(required = false)
	private Map<String, Object> sortMap;

	/* dirFileFlgの値 */
	private final String DIR_FILE_FLG_FILE = "0";
	private final String DIR_FILE_FLG_DIRECTORY = "1";

	@SuppressWarnings({ "unchecked", "rawtypes" })
	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		// 返却用配列
		List<Object> absolutePath = new ArrayList<Object>();
		List<Object> contentsName = new ArrayList<Object>();
		List<Map<String, Object>> fileList = new ArrayList<Map<String, Object>>();

		// ページング判定
		Boolean pagingFlg = false;
		Integer skipCnt = 0;
		Integer contentCnt = 0;
		if (page != null || pageMaxCount != null) {
			// pageとpageMaxCountのどちらかが設定されている場合、ページング有りとする
			pagingFlg = true;
			if (page == null) {
				// 未設定の場合はデフォルトを設定
				page = 1;
			}
			if (pageMaxCount == null) {
				// 未設定の場合はデフォルトを設定
				pageMaxCount = 20;
			}
			// 読み飛ばし数設定
			skipCnt = (page - 1) * pageMaxCount;
		}

		/* フィルタ作成 */
		FileFilter filter = new FileFilter() {
			/* trueの場合、そのファイルは選択される */
			public boolean accept(File fileObject) {

				if (!CheckUtil.isEmpty(fileName)) {
					if (fileObject.getName().indexOf(fileName) < 0) {
						// 検索：ファイル名がファイル名に含まれない場合、選択しない
						return false;
					}
				}

				if (!CheckUtil.isEmpty(fromDate)) {
					java.sql.Date dateObject = StringUtil.toDate(fromDate);
					if (dateObject != null && fileObject.lastModified()
							- dateObject.getTime() < 0) {
						// ファイルの更新日付が検索：アップロード日(始)より過去であった場合、選択しない
						return false;
					}
				}

				if (!CheckUtil.isEmpty(toDate)) {
					Calendar cal = Calendar.getInstance();
					cal.setTime(StringUtil.toDate(toDate));
					cal.add(Calendar.DATE, 1);
					java.sql.Date dateObject = new java.sql.Date(
							cal.getTimeInMillis());
					if (dateObject != null && dateObject.getTime()
							- fileObject.lastModified() < 0) {
						// ファイルの更新日付が検索：アップロード日(始)より未来であった場合、選択しない
						return false;
					}
				}

				return true;
			}
		};

		// ディレクトリの内容を取得
		File[] dirContents = Paths.get(dirPath).toFile().listFiles(filter);
		if (dirContents != null) {
			// 並び替え
			if (!CheckUtil.isEmpty(sort)) {

				Map<String, Object> sortMap = getSortMap();
				if (sortMap != null) {
					sort = StringUtil.defaultString(
							sortMap.getOrDefault(sort, sort), sort);
				}

				switch (sort) {
				case "up_desc":
					Arrays.sort(dirContents, new Comparator() {
						public int compare(Object o1, Object o2) {
							File f1 = (File) o1;
							File f2 = (File) o2;
							BigDecimal bdVal1 = null;
							BigDecimal bdVal2 = null;
							try {
								bdVal1 = new BigDecimal(f1.lastModified());
								bdVal2 = new BigDecimal(f2.lastModified());
							} catch (Exception e) {
								return 0;
							}

							return bdVal2.compareTo(bdVal1);
						}
					});
					break;

				case "up_asc":
					Arrays.sort(dirContents, new Comparator() {
						public int compare(Object o1, Object o2) {
							File f1 = (File) o1;
							File f2 = (File) o2;
							BigDecimal bdVal1 = null;
							BigDecimal bdVal2 = null;
							try {
								bdVal1 = new BigDecimal(f1.lastModified());
								bdVal2 = new BigDecimal(f2.lastModified());
							} catch (Exception e) {
								return 0;
							}

							return bdVal1.compareTo(bdVal2);
						}
					});
					break;

				case "name_asc":
					Arrays.sort(dirContents, new Comparator() {
						public int compare(Object o1, Object o2) {
							File f1 = (File) o1;
							File f2 = (File) o2;

							return f1.getName().compareTo(f2.getName());
						}
					});
					break;

				case "name_desc":
					Arrays.sort(dirContents, new Comparator() {
						public int compare(Object o1, Object o2) {
							File f1 = (File) o1;
							File f2 = (File) o2;

							return f2.getName().compareTo(f1.getName());
						}
					});
					break;

				case "size_asc":
					Arrays.sort(dirContents, new Comparator() {
						public int compare(Object o1, Object o2) {
							File f1 = (File) o1;
							File f2 = (File) o2;
							BigDecimal bdVal1 = null;
							BigDecimal bdVal2 = null;
							try {
								bdVal1 = new BigDecimal(f1.length());
								bdVal2 = new BigDecimal(f2.length());
							} catch (Exception e) {
								return 0;
							}

							return bdVal1.compareTo(bdVal2);
						}
					});
					break;

				case "size_desc":
					Arrays.sort(dirContents, new Comparator() {
						public int compare(Object o1, Object o2) {
							File f1 = (File) o1;
							File f2 = (File) o2;
							BigDecimal bdVal1 = null;
							BigDecimal bdVal2 = null;
							try {
								bdVal1 = new BigDecimal(f1.length());
								bdVal2 = new BigDecimal(f2.length());
							} catch (Exception e) {
								return 0;
							}

							return bdVal2.compareTo(bdVal1);
						}
					});
					break;

				default:
					break;
				}

			}

			for (int i = 0; i < dirContents.length; i++) {

				File file = dirContents[i];

				if (DIR_FILE_FLG_FILE.equals(dirFileFlg)) {
					// ファイルのみ抽出
					if (file.isFile()) {
						addFileList(file, absolutePath, contentsName, fileList,
								pagingFlg, i, skipCnt);
						contentCnt++;
					}

				} else if (DIR_FILE_FLG_DIRECTORY.equals(dirFileFlg)) {
					// ディレクトリのみ抽出
					if (file.isDirectory()) {
						addFileList(file, absolutePath, contentsName, fileList,
								pagingFlg, i, skipCnt);
						contentCnt++;
					}

				} else {
					// 区別なし
					addFileList(file, absolutePath, contentsName, fileList,
							pagingFlg, i, skipCnt);
					contentCnt++;
				}
			}
		}

		// 結果を返す
		ModuleResult result = new ModuleResult();
		result.getReturnData().put("dirPath", dirPath);
		result.getReturnData().put("dirFileFlg", dirFileFlg);
		result.getReturnData().put("absolutePathList", absolutePath);
		result.getReturnData().put("nameList", contentsName);
		result.getReturnData().put("fileList", fileList);
		result.getReturnData().put("contentCnt", contentCnt);

		return result;
	}

	private Map<String, Object> filePropertyMap(File file) {
		Map<String, Object> rtnMap = new HashMap<String, Object>();
		rtnMap.put("name", file.getName());
		rtnMap.put("size_b", String.valueOf(file.length()) + "B");
		rtnMap.put("size_kb", new BigDecimal(file.length())
				.divide(new BigDecimal(1024)).intValue() + "KB");
		rtnMap.put("size_mb",
				new BigDecimal(file.length()).divide(new BigDecimal(1024))
						.divide(new BigDecimal(1024)).intValue() + "MB");
		rtnMap.put("date", DateUtil.formatDate(file.lastModified()));

		return rtnMap;
	}

	private void addFileList(	File file,
								List<Object> absolutePath,
								List<Object> contentsName,
								List<Map<String, Object>> fileList,
								Boolean pagingFlg,
								int i,
								Integer skipCnt) {
		if (!pagingFlg || (skipCnt <= i && i < (skipCnt + pageMaxCount))) {
			absolutePath.add(file.getAbsolutePath());
			contentsName.add(file.getName());
			fileList.add(filePropertyMap(file));

		}
	}

	/**
	 * 対象ディレクトリを取得します。
	 * 
	 * @return 対象ディレクトリ
	 */
	public String getDirPath() {
		return dirPath;
	}

	/**
	 * 対象ディレクトリを設定します。
	 * 
	 * @param dirPath 対象ディレクトリ
	 */
	public void setDirPath(String dirPath) {
		this.dirPath = dirPath;
	}

	/**
	 * 一覧取得対象 0:ファイル 1:ディレクトリ 左記以外:区別なしを取得します。
	 * 
	 * @return 一覧取得対象 0:ファイル 1:ディレクトリ 左記以外:区別なし
	 */
	public String getDirFileFlg() {
		return dirFileFlg;
	}

	/**
	 * 一覧取得対象 0:ファイル 1:ディレクトリ 左記以外:区別なしを設定します。
	 * 
	 * @param dirFileFlg 一覧取得対象 0:ファイル 1:ディレクトリ 左記以外:区別なし
	 */
	public void setDirFileFlg(String dirFileFlg) {
		this.dirFileFlg = dirFileFlg;
	}

	/**
	 * ページ番号を取得します。
	 * 
	 * @return ページ番号
	 */
	public Integer getPage() {
		return page;
	}

	/**
	 * ページ番号を設定します。
	 * 
	 * @param page ページ番号
	 */
	public void setPage(Integer page) {
		this.page = page;
	}

	/**
	 * ページ最大件数を取得します。
	 * 
	 * @return ページ最大件数
	 */
	public Integer getPageMaxCount() {
		return pageMaxCount;
	}

	/**
	 * ページ最大件数を設定します。
	 * 
	 * @param pageMaxCount ページ最大件数
	 */
	public void setPageMaxCount(Integer pageMaxCount) {
		this.pageMaxCount = pageMaxCount;
	}

	/**
	 * 検索：ファイル名を取得します。
	 * 
	 * @return 検索：ファイル名
	 */
	public String getFileName() {
		return fileName;
	}

	/**
	 * 検索：ファイル名を設定します。
	 * 
	 * @param fileName 検索：ファイル名
	 */
	public void setFileName(String fileName) {
		this.fileName = fileName;
	}

	/**
	 * 検索：アップロード日(始)を取得します。
	 * 
	 * @return 検索：アップロード日(始)
	 */
	public String getFromDate() {
		return fromDate;
	}

	/**
	 * 検索：アップロード日(始)を設定します。
	 * 
	 * @param fromDate 検索：アップロード日(始)
	 */
	public void setFromDate(String fromDate) {
		this.fromDate = fromDate;
	}

	/**
	 * 検索：アップロード日(至)を取得します。
	 * 
	 * @return 検索：アップロード日(至)
	 */
	public String getToDate() {
		return toDate;
	}

	/**
	 * 検索：アップロード日(至)を設定します。
	 * 
	 * @param toDate 検索：アップロード日(至)
	 */
	public void setToDate(String toDate) {
		this.toDate = toDate;
	}

	/**
	 * 検索：ソートを取得します。
	 * 
	 * @return 検索：ソート
	 */
	public String getSort() {
		return sort;
	}

	/**
	 * 検索：ソートを設定します。
	 * 
	 * @param sort 検索：ソート
	 */
	public void setSort(String sort) {
		this.sort = sort;
	}

	/**
	 * 検索：ソート順マップを取得します。
	 * 
	 * @return 検索：ソート順マップ
	 */
	public Map<String, Object> getSortMap() {
		return sortMap;
	}

	/**
	 * 検索：ソート順マップを設定します。
	 * 
	 * @param sort 検索：ソート順マップ
	 */
	public void setSortMap(Map<String, Object> sortMap) {
		this.sortMap = sortMap;
	}
}
