package jp.ill.photon.module.delivery;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.postgresql.util.PGobject;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.StringUtil;

/**
 * 納品先マスタデータをマージするモジュール
 *
 * @author m_fukukawa
 *
 */
public class MergeDeliveryInfoModule implements PhotonModule {

	@ModuleParam(required=true)
	private Map<String, Object> common;

	@ModuleParam(required=true)
	private String tenantId;

	@ModuleParam(required=true)
	private String userCd;

	@ModuleParam(required=true)
	private List<Map<String, Object>> columnInfo;

	@ModuleParam(required=true)
	private Map<String, Object> deliveryInfo;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "")
	private String userChargeName;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "success")
	private Object successRetValue;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "failed")
	private Object failedRetValue;

	private JsonDataDao dao;
	private String editTableCd;
	private List<Map<String, Object>> keyInfo;
	private boolean isNewRegist = true;
	private boolean isKeyValueGenerate = false;
	private Map<String, Object> workDeliveryInfo;
	private String generatedValue;

	@Override
	public ModuleResult execute(ModuleContext context) {

		editTableCd = "userdelivery";

		// ワークデータ作成
		workDeliveryInfo = new HashMap<String, Object>(deliveryInfo);

		// keyInfoにキーカラムをセット
		getKeyInfo();

		// isNewRegistに処理モードをセット
		setProcessMode();

		// isKeyValueGenerateに採番モードをセット
		setIsKeyValueGenerate();

		// お届け先名分割
		setSplittedName();

		// DB登録
		int retCd = registDeliveryInfo();

		ModuleResult result = new ModuleResult();
		if (retCd == 0) {
			result.setResultCode("success");
			result.getReturnData().put("result_code", getSuccessRetValue());
			result.getReturnData().put("generated_value", generatedValue);
		} else {
			result.setResultCode("failed");
			result.getReturnData().put("result_code", getFailedRetValue());
		}
		return result;

	}

	/***
	 *
	 * 複数階層からなるマップから、キーのパスを指定して値を取得する。<br />]
	 * パスのデリミタはドット（"."）。
	 *
	 * @param mp
	 * @param path
	 * @return 値
	 */
	private String getStrValueFromMapByKeyPath(Map<String, Object> mp, String path) {
		return (String)getValueFromMapByKeyPath(mp, path);
	}

	/***
	 *
	 * 複数階層からなるマップから、キーのパスを指定して値を取得する。<br />]
	 * パスのデリミタはドット（"."）。
	 *
	 * @param mp
	 * @param path
	 * @return 値
	 */
	@SuppressWarnings("unchecked")
	private Object getValueFromMapByKeyPath(Map<String, Object> mp, String path) {

		if (MapUtils.isEmpty(mp) || StringUtils.isEmpty(path) ) {
			return null;
		}

		Map<String, Object> workMap = new HashMap<String, Object>(mp);

		String[] items = StringUtils.split(path, ".");
		String lastKey = items[items.length - 1];
		Object value = null;

		for (String item : items) {
			if (workMap == null) {
				return null;
			}
			if (item.equals(lastKey)) {
				value = workMap.get(lastKey);
			} else {
				workMap = (Map<String, Object>) workMap.get(item);
			}
		}

		return value;

	}

	/**
	 *
	 * キー項目の選出
	 *
	 */
	private void getKeyInfo() {
		keyInfo = new ArrayList<Map<String, Object>>();
		for (Map<String, Object> mp : columnInfo) {
			Integer keyNo = (Integer)mp.get("key_no");
			if ( keyNo > 0 ) {
				keyInfo.add(mp);
			}
		}
	}

	/**
	 *
	 * 処理モードのセット<br />
	 * JSON文字列からキー項目の値を取り出し、<br />
	 * それらがすべて空白、またはnullだった場合は新規登録モードとして扱う。<br />
	 * そうでない場合は更新モードとして扱う。
	 *
	 */
	private void setProcessMode() {
		for (Map<String, Object> mp : keyInfo) {
			String val = (String)deliveryInfo.get( (String) mp.get("col_cd") );
			if (!StringUtils.isEmpty(val)) {
				isNewRegist = false;
				break;
			}
		}
	}

	/***
	 *
	 * 採番モードをセット<br />
	 * JSON文字列からキー項目の値を取り出し、<br />
	 * 自動採番項目が存在していれば採番モードをONにする。<br />
	 * そうでない場合は採番モードをOFFのままとする。
	 *
	 */
	private void setIsKeyValueGenerate() {
		for (Map<String, Object> mp : keyInfo) {
			if ( !StringUtils.isEmpty( getStrValueFromMapByKeyPath(mp, "soft_const.etcol_autonum_use_sequence") ) ) {
				isKeyValueGenerate = true;
				break;
			}
		}
	}

	/**
	 *
	 * 指定されたcol_cdを持つカラム定義をリストから取得
	 *
	 */
	private Map<String, Object> getSpecifiedColumnDefinition(String colCd) {
		// 指定されたcol_cdを持つカラム定義をリストから取得
		for (Map<String, Object> mp : columnInfo) {
			if ( colCd.equals((String)mp.get("col_cd")) ) {
				return mp;
			}
		}
		return null;
	}

	/***
	 *
	 * 納品先名切り分け<br />
	 * user_delivery_nameの値を<br />
	 * user_delivery_name1のカラム定義の長さで切り分ける<br />
	 *
	 */
	private void setSplittedName() {
		String userDeliveryName = (String)workDeliveryInfo.get("user_delivery_name");
		int userDeliveryNameLength = (Integer)getValueFromMapByKeyPath(
										getSpecifiedColumnDefinition("user_delivery_name1"),
										"soft_const.text_length_max"
									);
		if (!StringUtils.isEmpty(userDeliveryName)) {
			String[] name = StringUtil.lengthSplitBytes(
						userDeliveryName,
						userDeliveryNameLength,
						"Windows-31J"
			);
			workDeliveryInfo.put("user_delivery_name1", name[0]);
			workDeliveryInfo.put("user_delivery_name2", (name.length >= 2)? name[1] : "");
		}
	}

	/**
	 *
	 * DB更新処理<br />
	 * 新規登録時、採番モードがONの場合は、登録SPを切り替える。<br />
	 *
	 * @return 戻り値（0:成功、-1:失敗）
	 *
	 */
	private int registDeliveryInfo() {

		String data = JsonUtil.mapToJson(generateValueMap());
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		dao = new JsonDataDaoImpl();

		int returnCd = tm.required(() -> {

			Map<String, Object> ret = null;

			if ( isNewRegist ) {
				// 挿入
				// 不足分のデータをセット
				if ( isKeyValueGenerate ) {
					// 画面項目定義ファイルを使用し、納品先コードを0埋めで新規採番して登録
					// ※画面項目定義ファイルの値が取れなかった場合は7桁（AO1.2の仕様）
					ret = dao.insertEditTableDataUsingScreenItemDefinitionFile(tenantId, editTableCd, userChargeName, data, "納品先コード", 7);
				} else {
					ret = dao.insertEditTableData(tenantId, editTableCd, userChargeName, data);
				}
			} else {
				// 更新
				ret = dao.updateEditTableData(tenantId, editTableCd, userChargeName, data);
			}

			Integer retVal = (Integer)ret.get("ret");
			if (ret.get("generated_val") != null) {
				Object genVal = ret.get("generated_val");
				if (PGobject.class.equals(genVal.getClass()) && "jsonb".equals(((PGobject) genVal).getType().toLowerCase())) {
					PGobject pgVal = (PGobject) genVal;
					Map<String, Object> mp = JsonUtil.jsonToMap(pgVal.getValue());
					// ジェネレートされるカラムは、一つのみ
					for (Map<String, Object> key : keyInfo) {
						generatedValue = (String)mp.get((String) key.get("col_cd"));
						break;
					}
				}
			}

			if (retVal != 0) {
				return -1;
			}
			return 0;

		});

		return returnCd;

	}

	/***
	 *
	 * DB更新用の値マップを作成
	 *
	 * @return 値マップ
	 *
	 */
	private Map<String, Object> generateValueMap() {
		Map<String, Object> ret = new HashMap<String, Object>();
		// 不要なデータの削除
		for (Map<String, Object> mp : columnInfo) {
			String key = (String)mp.get("col_cd");
			if ( workDeliveryInfo.containsKey( key ) ) {
				ret.put(key, workDeliveryInfo.get(key));
			}
		}
		// 不足分の追加
		// 定義上は存在するが、画面にない項目
		ret.put("user_cd", userCd);
		ret.put("charge_name", "");
		ret.put("mail", "");
		ret.put("remarks1", "");
		ret.put("remarks2", "");
		ret.put("remarks3", "");
		ret.put("delete_flg", "0");
		ret.put("web_div", "1");

		if ( isNewRegist ) {
			ret.put("ao_flg", "0");
		}

		return ret;
	}

	/**
	 * commonを取得します。
	 * @return common
	 */
	public Map<String, Object> getCommon() {
		return common;
	}

	/**
	 * commonを設定します。
	 * @param common
	 */
	public void setCommon(Map<String, Object> common) {
		this.common = common;
	}

	/**
	 * tenantIdを取得します。
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	/**
	 * userCdを取得します。
	 * @return userCd
	 */
	public String getUserCd() {
		return userCd;
	}

	/**
	 * userCdを設定します。
	 * @param userCd
	 */
	public void setUserCd(String userCd) {
		this.userCd = userCd;
	}

	/**
	 * columnInfoを取得します。
	 * @return columnInfo
	 */
	public List<Map<String, Object>> getColumnInfo() {
		return columnInfo;
	}

	/**
	 * columnInfoを設定します。
	 * @param columnInfo
	 */
	public void setColumnInfo(List<Map<String, Object>> columnInfo) {
		this.columnInfo = columnInfo;
	}

	/**
	 * deliveryInfoを取得します。
	 * @return deliveryInfo
	 */
	public Map<String, Object> getDeliveryInfo() {
		return deliveryInfo;
	}

	/**
	 * deliveryInfoを設定します。
	 * @param deliveryInfo
	 */
	public void setDeliveryInfo(Map<String, Object> deliveryInfo) {
		this.deliveryInfo = deliveryInfo;
	}

	/**
	 * userChargeNameを取得します。
	 * @return userChargeName
	 */
	public String getUserChargeName() {
		return userChargeName;
	}

	/**
	 * userChargeNameを設定します。
	 * @param userChargeName
	 */
	public void setUserChargeName(String userChargeName) {
		this.userChargeName = userChargeName;
	}

	/**
	 * successRetValueを取得します。
	 * @return successRetValue
	 */
	public Object getSuccessRetValue() {
		return successRetValue;
	}

	/**
	 * successRetValueを設定します。
	 * @param successRetValue
	 */
	public void setSuccessRetValue(Object successRetValue) {
		this.successRetValue = successRetValue;
	}

	/**
	 * failedRetValueを取得します。
	 * @return failedRetValue
	 */
	public Object getFailedRetValue() {
		return failedRetValue;
	}

	/**
	 * failedRetValueを設定します。
	 * @param failedRetValue
	 */
	public void setFailedRetValue(Object failedRetValue) {
		this.failedRetValue = failedRetValue;
	}

}
