package jp.ill.photon.module.db;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.postgresql.jdbc4.Jdbc4Array;
import org.postgresql.util.PGobject;
import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.action.ActionDispatcher;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.doma.SqlSelectQueryFactory;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.SearchForm;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ArrayUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.LogUtil;

/**
 * [sql_text_select] 文字列をDomaテンプレートSQLとして解釈して実行し、実行結果を返すモジュール.
 *
 * <p>
 * Doma用SQLテキストと、そのSQL内で使用したいパラメータを渡すと、SQLを実行して結果を返す。</br>
 * 実行されるSQLはSelect文である必要はないが、必ず値を返す必要がある。<br/>
 * UPDATE文等を実行する時は、RETURNING句等で値を返す。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>cnt</dt>
 * <dd>limit前のデータ件数。(SQLから"count"という名称で返された値を使用する)</dd>
 * <dt>list</dt>
 * <dd>検索結果リスト。MapのListを返却する。</dd>
 * <dt>first</dt>
 * <dd>検索結果の最初の1データ。Mapを返却する。</dd>
 * </dl>
 * </p>
 * 
 * @see SqlFileSelectModule
 * @author h_tanaka
 *
 */
public class SqlTextSelectModule implements PhotonModule {

	/**
	 * SQLファイルで使用するパラメータマップ用モジュールパラメータ.
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 * 
	 * <pre>
	 * {
	 *   "tenantId": {
	 *     "type": "param",
	 *     "val": "_init.tenant_id"
	 *   },
	 *   "formCd": {
	 *     "type": "static",
	 *     "val": "test_val"
	 *   }
	 * }
	 * </pre>
	 * 
	 * 上記の内容を指定すると、SQLファイル内で「tenantId」「formCd」というパラメータを使用することができる。</dd>
	 * </dl>
	 * </p>
	 * 
	 */
	@ModuleParam(required = true, domainObject = true)
	private SearchForm searchForm;

	/**
	 * SQL文字列用モジュールパラメータ.
	 * 
	 * <p>
	 * 文字列形式でDoma用SQLテンプレートを受け取る。
	 * </p>
	 */
	@ModuleParam(required = true)
	private String sqlText;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(ActionDispatcher.class);

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		SqlSelectQuery selectQuery = SqlSelectQueryFactory.newInstance()
				.createSelectQuery(getSearchForm().getParamMap(),
						context.getDto(), getSqlText());

		selectQuery.prepare();

		// 検索処理を実行
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		List<Map<String, Object>> resultList = tm.required(() -> {
			SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
					selectQuery,
					new MapResultListHandler(MapKeyNamingType.NONE));
			return command.execute();
		});

		// 値のコンバート
		convertValues(resultList);

		// 結果オブジェクト生成
		ModuleResult result = new ModuleResult();
		result.getReturnData().put("list", resultList);
		if (resultList.size() > 0) {
			result.getReturnData().put("first", resultList.get(0));
		} else {
			result.getReturnData().put("first", null);
		}

		int cnt = 0;
		if (resultList != null && !resultList.isEmpty()
				&& resultList.get(0).get("count") != null) {
			cnt = Integer
					.parseInt(String.valueOf(resultList.get(0).get("count")));
		}

		logger.info(String.format("---->[count]:[%d]", ((cnt == 0)
				? ((resultList != null) ? resultList.size() : 0) : cnt)));

		result.getReturnData().put("cnt", cnt);

		return result;
	}

	/**
	 *
	 * 値のコンバート
	 *
	 * @param src
	 */
	protected void convertValues(List<Map<String, Object>> src) {

		// コンバート対象がないか、レコード内を走査
		if (!CollectionUtils.isEmpty(src)) {
			PGobject pgVal = null;
			Jdbc4Array arVal = null;
			for (Map<String, Object> mp : src) {
				for (Map.Entry<String, Object> e : mp.entrySet()) {
					if (e.getValue() != null) {

						if (PGobject.class.equals(e.getValue().getClass())
								&& "jsonb".equals(((PGobject) e.getValue())
										.getType().toLowerCase())) {
							pgVal = (PGobject) e.getValue();
							mp.put(e.getKey(),
									JsonUtil.jsonToMap(pgVal.getValue()));
						}

						if (Jdbc4Array.class.equals(e.getValue().getClass())) {
							arVal = (Jdbc4Array) e.getValue();
							try {
								mp.put(e.getKey(), ArrayUtil
										.asList((Object[]) arVal.getArray()));
							} catch (SQLException e1) {
								mp.put(e.getKey(), new ArrayList<Object>());
							}
						}
					}
				}
			}
		}
	}

	public SearchForm getSearchForm() {
		return searchForm;
	}

	public void setSearchForm(SearchForm searchForm) {
		this.searchForm = searchForm;
	}

	public String getSqlText() {
		return sqlText;
	}

	public void setSqlText(String sqlText) {
		this.sqlText = sqlText;
	}
}
