package jp.ill.photon.module.db;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.doma.SqlSelectQueryFactory;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.SearchForm;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.SQLUtil;

/**
 * [sql_file_select] 指定したDoma用SQLテンプレートファイルを読み込んでSELECT処理を実行するモジュール.
 *
 * <p>
 * Doma用SQLテンプレートファイルのパスと、そのSQL内で使用したいパラメータを渡すと、SQLを実行して結果を返す。</br>
 * 実行されるSQLはSelect文である必要はないが、必ず値を返す必要がある。<br/>
 * UPDATE文等を実行する時は、RETURNING句等で値を返す。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>cnt</dt>
 * <dd>limit前のデータ件数。(SQLから"count"という名称で返された値を使用する)</dd>
 * <dt>list</dt>
 * <dd>検索結果リスト。MapのListを返却する。</dd>
 * <dt>first</dt>
 * <dd>検索結果の最初の1データ。Mapを返却する。</dd>
 * </dl>
 * </p>
 *
 * @author h_tanaka
 *
 */
public class SqlFileSelectModule implements PhotonModule {

	/**
	 * SQLファイルで使用するパラメータマップ用モジュールパラメータ.
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 *
	 * <pre>
	 * {
	 *   "tenantId": {
	 *     "type": "param",
	 *     "val": "_init.tenant_id"
	 *   },
	 *   "formCd": {
	 *     "type": "static",
	 *     "val": "test_val"
	 *   },
	 *   "check_list": {
	 *     "type": "param",
	 *     "val": "checks",
	 *     "data_type": "object_list"
	 *   }
	 * }
	 * </pre>
	 *
	 * </dd>
	 * </dl>
	 * </p>
	 *
	 * <p>
	 * 上記の内容を指定すると、SQLファイル内で「tenantId」「formCd」「check_list」というパラメータを使用することができる。<br/>
	 * 特に指定しない場合、パラメータはString型として解釈されるが、「data_type」キーでデータ型を指定することでマップやリストとして<br/>
	 * 使用できるようになる。
	 * </p>
	 *
	 * <p>
	 * data_typeで指定可能な値：<br/>
	 * <dl>
	 * <dt>text</dt>
	 * <dd>文字列</dd>
	 * <dt>number</dt>
	 * <dd>数値</dd>
	 * <dt>text_list</dt>
	 * <dd>List&lt;String&gt;型</dd>
	 * <dt>object_list</dt>
	 * <dd>List&lt;Object&gt;型</dd>
	 * <dt>text_map</dt>
	 * <dd>Map&lt;String, String&gt;型</dd>
	 * <dt>object_map</dt>
	 * <dd>Map&lt;String, Object&gt;型</dd>
	 * </dl>
	 * </p>
	 *
	 */
	@ModuleParam(required = true, domainObject = true)
	private SearchForm searchForm;

	/**
	 * SQLファイルのフォルダパス指定用モジュールパラメータ.
	 *
	 * <p>
	 * 通常はDTOの「common.systemsetting.exSqlFileDir.note」を指定することで、<br/>
	 * システム設定のSQL保存ディレクトリからSQLファイルを読み込むことができる。
	 * </p>
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>dto</dd>
	 * <dt>transfer_val</dt>
	 * <dd>common.systemsetting.exSqlFileDir.note</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = true)
	private String sqlFileDirPath;

	/**
	 * SQLファイルパス指定用モジュールパラメータ.
	 *
	 * <p>
	 * sqlFileDirPathからの相対パスを指定する。
	 * </p>
	 */
	@ModuleParam(required = true)
	private String sqlFilePath;

	/**
	 * 新規トランザクション開始指定用モジュールパラメータ.
	 *
	 * <p>
	 * このSQLを実行する際にrequiredNewで別トランザクションを開始するかを指定する<br/>
	 * （1:開始する、1以外:開始しない）
	 * </p>
	 */
	@ModuleParam(required = false)
	private String requireNewTransaction;

	/**
	 * return_prefixのデータセット統合用モジュールパラメータ.
	 *
	 * <p>
	 * return_prefixでlist情報が保存されている場合、当モジュールで取得したレコード内の列情報を優先してレコード情報をマージする<br/>
	 * （1:マージする、1以外:マージしない）
	 * </p>
	 */
	@ModuleParam(required = false)
	private String mergeValue;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(SqlFileSelectModule.class);

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		ModuleResult result = new ModuleResult();

		if (getSqlFileDirPath() == null || getSqlFilePath() == null) {
			logger.error("SQLファイルパスが指定されていません");
			result.getReturnData().put("cnt", 0);
			result.getReturnData().put("list",
					new ArrayList<Map<String, Object>>());
			result.getReturnData().put("first", null);
			return result;
		}

		SqlSelectQuery selectQuery = SqlSelectQueryFactory.newInstance()
				.createSelectQueryFromFile(getSearchForm().getParamMap(),
						context.getDto(), getSqlFileDirPath(),
						getSqlFilePath());

		selectQuery.prepare();

		// 検索処理を実行
		List<Map<String, Object>> resultList = null;
		if (getRequireNewTransaction().equals("1")) {
			TransactionManager tm = DomaConfig.singleton()
					.getTransactionManager();
			resultList = tm.requiresNew(() -> {
				SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
						selectQuery,
						new MapResultListHandler(MapKeyNamingType.NONE));
				return command.execute();
			});
		} else {
			SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
					selectQuery,
					new MapResultListHandler(MapKeyNamingType.NONE));
			resultList = command.execute();
		}

		// 値のコンバート
		SQLUtil.convertDbValues(resultList);

		// 結果オブジェクトのマージを試みる
		if ("1".equals(getMergeValue())) {
			String return_key = (String) context.getActionModule().get("return_prefix");
			Object dtoVal = context.getDto().get(return_key);
			if (dtoVal instanceof Map) {
				Map<?, ?> base = (Map<?, ?>) dtoVal;
				if (!base.isEmpty()) {
					// listをマージ
					List<Map<String, Object>> blist = (List<Map<String, Object>>) base.get("list");
					if (blist.size() == resultList.size()) {
						for(int i = 0; i < blist.size(); ) {
							Map<String, Object> map = resultList.get(i);
							blist.get(i).forEach((key, val) -> {
								map.put(key, val);
							});
							resultList.set(i, map);
							i++;
						}
					}
				}
			}
		}

		// 結果オブジェクト生成
		result.getReturnData().put("list", resultList);
		if (resultList.size() > 0) {
			result.getReturnData().put("first", resultList.get(0));
		} else {
			result.getReturnData().put("first", null);
		}

		int cnt = 0;
		if (resultList != null && !resultList.isEmpty()
				&& resultList.get(0).get("count") != null) {
			cnt = Integer
					.parseInt(String.valueOf(resultList.get(0).get("count")));
		}

		logger.info(String.format("---->[count]:[%d]", ((cnt == 0)
				? ((resultList != null) ? resultList.size() : 0) : cnt)));

		result.getReturnData().put("cnt", cnt);

		return result;
	}

	public SearchForm getSearchForm() {
		return searchForm;
	}

	public void setSearchForm(SearchForm searchForm) {
		this.searchForm = searchForm;
	}

	public String getSqlFileDirPath() {
		if (sqlFileDirPath == null) {
			sqlFileDirPath = "";
		}
		return sqlFileDirPath;
	}

	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	public String getSqlFilePath() {
		return sqlFilePath;
	}

	public void setSqlFilePath(String sqlFilePath) {
		this.sqlFilePath = sqlFilePath;
	}

	public String getRequireNewTransaction() {
		if (requireNewTransaction == null) {
			requireNewTransaction = "";
		}
		return requireNewTransaction;
	}

	public void setRequireNewTransaction(String requireNewTransaction) {
		this.requireNewTransaction = requireNewTransaction;
	}

	public String getMergeValue() {
		return mergeValue;
	}

	public void setMergeValue(String mergeValue) {
		this.mergeValue = mergeValue;
	}

}
