package jp.ill.photon.module.db;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ParamUtil;

import org.apache.commons.collections.CollectionUtils;
import org.seasar.doma.FetchType;
import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.internal.jdbc.sql.SqlParser;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

/**
 * sqlFilePathで指定されたパスにあるSQLファイル（Doma2テンプレートSQL）
 * とdataListからSQLを組み立て実行する汎用検索用モジュール
 *
 * @author m_fukukawa
 *
 */
public class SqlFileSelectByListModule implements PhotonModule {

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.exSqlFileDir.note")
	private String sqlFileDirPath;

	@ModuleParam(required = true)
	private String sqlFilePath;

	@ModuleParam(required = true, domainObject = true)
	private Map<String, Object> commonParams;

	@ModuleParam(required = true, domainObject = true)
	private List<Map<String, Object>> dataList;

	private ActionDto dto;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		// 結果オブジェクト生成
		ModuleResult result = new ModuleResult();
		dto = context.getDto();

		if (!CollectionUtils.isEmpty(dataList)) {

			for (Map<String, Object> mp : dataList) {

				String key = (String)mp.get("key");

				// SQL読み込み
				SqlSelectQuery selectQuery = getSqlSelectQueryFromSqlFile();

				// パラメータセット
				addParamsToSelectQuery(selectQuery, mp);

				selectQuery.prepare();

				// 検索処理を実行
				TransactionManager tm = DomaConfig.singleton().getTransactionManager();
				List<Map<String, Object>> resultList = tm.required(() -> {
					SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
							selectQuery,
							new MapResultListHandler(MapKeyNamingType.NONE));
					return command.execute();
				});

				// 結果をマップに格納
				Map<String, Object> retMap = new HashMap<String, Object>();
				retMap.put("list", resultList);
				if (!CollectionUtils.isEmpty(resultList)) {
					retMap.put("first", resultList.get(0));
				} else {
					retMap.put("first", null);
				}

				int cnt = 0;
				if (!CollectionUtils.isEmpty(resultList) && resultList.get(0).get("count") != null) {
					cnt = Integer.parseInt(String.valueOf(resultList.get(0).get("count")));
				}
				retMap.put("cnt", cnt);

				result.getReturnData().put(key, retMap);

			}

		}

		return result;

	}

	/**
	 * SQLファイルからDomaのクエリオブジェクト生成
	 *
	 * @exception PhotonModuleException
	 *
	 */
	private SqlSelectQuery getSqlSelectQueryFromSqlFile() throws PhotonModuleException {

		// SQLファイル存在チェック
		Path path = Paths.get(sqlFileDirPath, sqlFilePath);
		if (!Files.exists(path)) {
			throw new PhotonModuleException("SQLファイルが存在しません:" + path.toString(),
					null);
		}

		// SQLを読み込んでDomaのSQLParserで解析
		SqlParser parser;
		try {
			parser = new SqlParser(
					Files.readAllLines(path, Charset.forName("UTF-8")).stream()
							.collect(Collectors.joining("\n")));
		} catch (IOException e) {
			throw new PhotonModuleException("SQLの解析に失敗しました:" + path.toString(),
					e);
		}

		// DomaのクエリオブジェクトにSQLをセットする
		SqlSelectQuery selectQuery = new SqlSelectQuery();
		selectQuery.setConfig(DomaConfig.singleton());
		selectQuery.setCallerClassName(getClass().getName());
		selectQuery.setCallerMethodName("execute");
		selectQuery.setFetchType(FetchType.LAZY);
		selectQuery.setSqlLogType(SqlLogType.FORMATTED);
		selectQuery.setSqlNode(parser.parse());

		return selectQuery;

	}

	/**
	 * SQLにパラメータをセット
	 *
	 * @param query クエリオブジェクト
	 * @param data 1行のデータ。中にSQL文のもとが入っている想定
	 *
	 * */
	private void addParamsToSelectQuery(SqlSelectQuery query, Map<String, Object> data) {

		// 共通パラメータがあった場合は、先にセットする
		if (commonParams != null) {
			for (Map.Entry<String, Object> e : commonParams.entrySet()) {
				Object value = ParamUtil.getParamObjectValueByType(e.getValue(), dto);
				query.addParameter(
						e.getKey(),
						value.getClass(),
						value
				);
			}
		}

		String valueSelectQuery = (String)data.get("value_select_query");
		String captionSelectQuery = (String)data.get("caption_select_query");
		String tableName = (String)data.get("table_name");
		String orderQuery = (String)data.get("order_query");

		query.addParameter(ParamUtil.snakeToCamel("value_select_query"), String.class, valueSelectQuery);
		query.addParameter(ParamUtil.snakeToCamel("caption_select_query"), String.class, captionSelectQuery);
		query.addParameter(ParamUtil.snakeToCamel("table_name"), String.class, tableName);
		query.addParameter(ParamUtil.snakeToCamel("order_query"), String.class, orderQuery);

	}

	/**
	 * sqlFileDirPathを取得します。
	 * @return sqlFileDirPath
	 */
	public String getSqlFileDirPath() {
		return sqlFileDirPath;
	}

	/**
	 * sqlFileDirPathを設定します。
	 * @param sqlFileDirPath
	 */
	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	/**
	 * sqlFilePathを取得します。
	 * @return sqlFilePath
	 */
	public String getSqlFilePath() {
		return sqlFilePath;
	}

	/**
	 * sqlFilePathを設定します。
	 * @param sqlFilePath
	 */
	public void setSqlFilePath(String sqlFilePath) {
		this.sqlFilePath = sqlFilePath;
	}

	/**
	 * commonParamsを取得します。
	 * @return commonParams
	 */
	public Map<String, Object> getCommonParams() {
		return commonParams;
	}

	/**
	 * commonParamsを設定します。
	 * @param commonParams
	 */
	public void setCommonParams(Map<String, Object> commonParams) {
		this.commonParams = commonParams;
	}

	/**
	 * dataListを取得します。
	 * @return dataList
	 */
	public List<Map<String, Object>> getDataList() {
		return dataList;
	}

	/**
	 * dataListを設定します。
	 * @param dataList
	 */
	public void setDataList(List<Map<String, Object>> dataList) {
		this.dataList = dataList;
	}

}
