package jp.ill.photon.module.db;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.stream.Collectors;

import org.seasar.doma.internal.jdbc.sql.SqlParser;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.command.DeleteCommand;
import org.seasar.doma.jdbc.command.InsertCommand;
import org.seasar.doma.jdbc.command.UpdateCommand;
import org.seasar.doma.jdbc.query.DeleteQuery;
import org.seasar.doma.jdbc.query.InsertQuery;
import org.seasar.doma.jdbc.query.SqlDeleteQuery;
import org.seasar.doma.jdbc.query.SqlInsertQuery;
import org.seasar.doma.jdbc.query.SqlModifyQuery;
import org.seasar.doma.jdbc.query.SqlUpdateQuery;
import org.seasar.doma.jdbc.query.UpdateQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ParamUtil;

public class SqlFileModifyModule implements PhotonModule {

	@ModuleParam(required = true)
	private Map<String, Object> paramMap;

	@ModuleParam(required = true)
	private String sqlFileDirPath;

	@ModuleParam(required = true)
	private String sqlFilePath;

	@ModuleParam(required = true)
	private String queryType;

	public String getQueryType() {
		return queryType;
	}

	public void setQueryType(String queryType) {
		this.queryType = queryType;
	}

	public Map<String, Object> getParamMap() {
		return paramMap;
	}

	public void setParamMap(Map<String, Object> paramMap) {
		this.paramMap = paramMap;
	}

	public String getSqlFileDirPath() {
		return sqlFileDirPath;
	}

	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	public String getSqlFilePath() {
		return sqlFilePath;
	}

	public void setSqlFilePath(String sqlFilePath) {
		this.sqlFilePath = sqlFilePath;
	}

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		ModuleResult result = new ModuleResult();

		Map<String, Object> params = new LinkedHashMap<String, Object>();
		for (Map.Entry<String, Object> param : paramMap.entrySet()) {
			params.put((String) param.getKey(),
					ParamUtil.getParamValueByType(
							String.valueOf(
									((Map) param.getValue()).get("type")),
							((Map) param.getValue()).get("val"),
							context.getDto()));
		}

		// SQLファイル存在チェック
		Path path = Paths.get(getSqlFileDirPath(), getSqlFilePath());
		if (!Files.exists(path)) {
			throw new PhotonModuleException("SQLファイルが存在しません", null);
		}

		// SQLを読み込んでDomaのSQLParserで解析
		SqlParser parser;
		try {
			parser = new SqlParser(
					Files.readAllLines(path, Charset.forName("UTF-8")).stream()
							.collect(Collectors.joining("\n")));
		} catch (IOException e) {
			throw new PhotonModuleException("SQLの解析に失敗しました", e);
		}

		SqlModifyQuery modifyQuery = null;

		if ("insert".equalsIgnoreCase(queryType)) {
			modifyQuery = new SqlInsertQuery();
		} else if ("update".equalsIgnoreCase(queryType)) {
			modifyQuery = new SqlUpdateQuery();
		} else if ("delete".equalsIgnoreCase(queryType)) {
			modifyQuery = new SqlDeleteQuery();
		} else {
			throw new PhotonModuleException("Query type is invalid.", null);
		}

		modifyQuery.setConfig(DomaConfig.singleton());
		modifyQuery.setCallerClassName(getClass().getName());
		modifyQuery.setCallerMethodName("execute");
		modifyQuery.setSqlLogType(SqlLogType.FORMATTED);
		modifyQuery.setSqlNode(parser.parse());

		// パラメータをクエリオブジェクトにセットする
		for (Map.Entry<String, Object> entry : params.entrySet()) {
			// TODO ootah: String.valueOfしているのは暫定 -> imakake: IN句で使用するため、パラメータのクラスを使用するように変更
			modifyQuery.addParameter(
					entry.getKey(),
					entry.getValue() == null ? String.class : entry.getValue().getClass(),
					entry.getValue() == null ? "" : entry.getValue());
		};

		modifyQuery.prepare();

		// 検索処理を実行
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		int sqlResult = -1;
		if ("insert".equalsIgnoreCase(queryType)) {
			InsertQuery query = (InsertQuery)modifyQuery;
			sqlResult = tm.required(() -> {
				InsertCommand command = new InsertCommand(query);
				return command.execute();
			});
		} else if ("update".equalsIgnoreCase(queryType)) {
			UpdateQuery query = (UpdateQuery)modifyQuery;
			sqlResult = tm.required(() -> {
				UpdateCommand command = new UpdateCommand(query);
				return command.execute();
			});
		} else {
			DeleteQuery query = (DeleteQuery)modifyQuery;
			sqlResult = tm.required(() -> {
				DeleteCommand command = new DeleteCommand(query);
				return command.execute();
			});
		}

		result.getReturnData().put("sql-result", sqlResult);
		result.setResultCode("success");

		return result;
	}

}
