package jp.ill.photon.module.db;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.map.HashedMap;
import org.apache.commons.lang3.StringUtils;
import org.postgresql.jdbc4.Jdbc4Array;
import org.postgresql.util.PGobject;
import org.seasar.doma.FetchType;
import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.internal.jdbc.sql.SqlParser;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.SqlNode;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.doma.SqlSelectQueryFactory;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ArrayUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;

/**
 * [sql_file_bulk_select] 複数のDoma用SQLテンプレートファイルを読み込んでまとめてSELECT処理を実行するモジュール.
 * 
 * <p>
 * SqlFileSelectModuleの複数クエリ対応版。<br/>
 * Doma用SQLテンプレートファイルのパスと、そのSQL内で使用したいパラメータを渡すと、SQLを実行して結果を返す。</br>
 * 実行されるSQLはSelect文である必要はないが、必ず値を返す必要がある。<br/>
 * UPDATE文等を実行する時は、RETURNING句等で値を返す。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * 
 * <p>
 * このモジュールはsql_filesパラメータで指定されたkeyをDTOのreturn_prefix下に生成し、<br/>
 * その中に下記の値をセットして返す。
 * </p>
 * 
 * <dl>
 * <dt>cnt</dt>
 * <dd>limit前のデータ件数。(SQLから"count"という名称で返された値を使用する)</dd>
 * <dt>list</dt>
 * <dd>検索結果リスト。MapのListを返却する。</dd>
 * <dt>first</dt>
 * <dd>検索結果の最初の1データ。Mapを返却する。</dd>
 * </dl>
 * </p>
 * 
 * @see SqlFileSelectModule
 * @author m_fukukawa
 *
 */
public class SqlFileBulkSelectModule implements PhotonModule {

	/**
	 * SQLファイルのフォルダパス指定用モジュールパラメータ.
	 * 
	 * <p>
	 * 通常はDTOの「common.systemsetting.exSqlFileDir.note」を指定することで、<br/>
	 * システム設定のSQL保存ディレクトリからSQLファイルを読み込むことができる。
	 * </p>
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>dto</dd>
	 * <dt>transfer_val</dt>
	 * <dd>common.systemsetting.exSqlFileDir.note</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.exSqlFileDir.note")
	private String sqlFileDirPath;

	/**
	 * SQLファイルパスリスト指定用モジュールパラメータ.
	 * 
	 * <p>
	 * 一意のキーとファイルパスを持つファイルパスマップのリストを受け取る。<br/>
	 * ファイルパスはsqlFileDirPathからの相対パスを指定する。
	 * </p>
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 * 
	 * <pre>
	 * [
	 *   {
	 *     "key": "form_setting",
	 *     "file": "aec20/table/selectDispFieldsByFormCdForSearch.sql",
	 *     "remarks": "フォーム設定取得"
	 *   },
	 *   {
	 *     "key": "info_messages",
	 *     "file": "aec20/table/selectInfoMessagesByFormCd.sql",
	 *     "remarks": "メッセージ取得"
	 *   }
	 * ]
	 * </pre>
	 * 
	 * </dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = true)
	private List<Map<String, Object>> sqlFiles;

	/**
	 * SQLファイルで使用するパラメータマップ用モジュールパラメータ.
	 * 
	 * <p>
	 * 各SQLで共通で使用するSQLパラメータをまとめた「common_params」と<br/>
	 * 個別に使用するSQLパラメータをまとめた「individual_params」を設定する。（どちらも必須）<br/>
	 * individual_paramsはリスト形式で受取り、モジュールパラメータsql_filesの対応するインデックスのSQLファイルで使用される。
	 * </p>
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 * 
	 * <pre>
	 * {
	 *   "common_params": {
	 *     "tenantId": {
	 *       "type": "param",
	 *       "val": "_init.tenant_id",
	 *       "remarks": "共通パラメータ1"
	 *     },
	 *     "formCd": {
	 *       "type": "static",
	 *       "val": "test_val",
	 *       "remarks": "共通パラメータ2"
	 *     }
	 *   },
	 *   "individual_params": [
	 *     {
	 *       "param1": {
	 *         "type": "param",
	 *         "val": "param1"
	 *         "remarks": "SQL1用パラメータ"
	 *       }
	 *     },
	 *     {
	 *       "param2": {
	 *         "type": "static",
	 *         "val": "param2",
	 *         "remarks": "SQL2用パラメータ"
	 *       }
	 *     }
	 *   ]
	 * }
	 * </pre>
	 * 
	 * </dd>
	 * </dl>
	 * </p>
	 * 
	 */
	@ModuleParam(required = true, domainObject = true)
	private Map<String, Object> searchForms;

	/** データタイプ */
	public static final class DataTypes {
		public static final String TEXT = "text";
		public static final String NUMBER = "number";
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		// 結果オブジェクト生成
		ModuleResult result = new ModuleResult();

		Map<String, Object> common_params = (Map<String, Object>) searchForms
				.get("common_params");

		for (int i = 0; i < sqlFiles.size(); i++) {

			String key = (String) sqlFiles.get(i).get("key");
			String sqlFilePath = (String) sqlFiles.get(i).get("file");

			List<Map<String, Object>> individual_params = (List<Map<String, Object>>) searchForms
					.get("individual_params");
			// SearchForm searchForm = searchForms.get(i);

			// パラメータデータを取得
			// Map<String, Object> srcParams = searchForm.getParamMap();
			Map<String, Object> srcParams = new HashMap<String, Object>();
			if (!CollectionUtils.isEmpty(individual_params)) {
				srcParams = individual_params.get(i);
			}
			Map<String, Object> params = new LinkedHashMap<String, Object>();
			Map<String, Object> paramTypes = new LinkedHashMap<String, Object>();

			generateParams(common_params, params, paramTypes, context.getDto());
			generateParams(srcParams, params, paramTypes, context.getDto());

			// SQLファイル存在チェック
			Path path = Paths.get(sqlFileDirPath, sqlFilePath);
			if (!Files.exists(path)) {
				throw new PhotonModuleException(
						"SQLファイルが存在しません:" + path.toString(), null);
			}

			// SQLを読み込んでDomaのSQLParserで解析
			SqlParser parser;
			try {
				parser = new SqlParser(
						Files.readAllLines(path, Charset.forName("UTF-8"))
								.stream().collect(Collectors.joining("\n")));
			} catch (IOException e) {
				throw new PhotonModuleException(
						"SQLの解析に失敗しました:" + path.toString(), e);
			}

			// DomaのクエリオブジェクトにSQLをセットする
			SqlSelectQuery selectQuery = new SqlSelectQuery();
			selectQuery.setConfig(DomaConfig.singleton());
			selectQuery.setCallerClassName(getClass().getName());
			selectQuery.setCallerMethodName("execute");
			selectQuery.setFetchType(FetchType.LAZY);
			selectQuery.setSqlLogType(SqlLogType.FORMATTED);
			// selectQuery.setSqlNode(parser.parse());
			SqlNode sn = parser.parse();
			selectQuery.setSqlNode(sn);

			SqlSelectQueryFactory sqf = SqlSelectQueryFactory.newInstance();
			// パラメータをクエリオブジェクトにセットする // SQL文の中からパラメータを取得
			List<String> paramFromSqlFile = new ArrayList<String>();
			sqf.getVariableName(sn, paramFromSqlFile);

			// paramに不足分をマージする
			for (String p : paramFromSqlFile) {
				if (!params.containsKey(p)) {
					params.put(p, null);
					paramTypes.put(p, new HashedMap() {
						{
							put("type", "");
							put("data_type", ""); // データタイプは、強制的に単一テキストとなる
						}
					});
				}
			}

			for (Map.Entry<String, Object> entry : params.entrySet()) {

				Map<String, String> typeDataTypeMap = (Map<String, String>) paramTypes
						.get(entry.getKey());
				String type = typeDataTypeMap.get("type");
				String dataType = typeDataTypeMap.get("data_type");

				Class clazz = null;
				Object defValue = null;
				Object setValue = null;
				if ("param_multi".equals(type)) {

					clazz = List.class;
					String[] strArr = (String[]) entry.getValue();

					if (DataTypes.NUMBER.equals(dataType)) {
						List<Integer> list = new ArrayList<Integer>();
						for (String item : strArr) {
							list.add(Integer.parseInt(item));
						}
						setValue = list;
					} else {
						List<String> list = new ArrayList<String>();
						for (String item : strArr) {
							list.add(item);
						}
						setValue = list;
					}

				} else {

					if (DataTypes.NUMBER.equals(dataType)) {
						clazz = Integer.class;
						setValue = (Integer) entry.getValue();
					} else {
						defValue = "";
						if (entry.getValue() == null) {
							clazz = String.class;
						} else {
							clazz = entry.getValue().getClass();
							setValue = entry.getValue();
						}
					}

				}

				selectQuery.addParameter(entry.getKey(), clazz,
						(entry.getValue() == null ? defValue : setValue));

			}

			selectQuery.prepare();

			// 検索処理を実行
			TransactionManager tm = DomaConfig.singleton()
					.getTransactionManager();
			List<Map<String, Object>> resultList = tm.required(() -> {
				SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
						selectQuery,
						new MapResultListHandler(MapKeyNamingType.NONE));
				return command.execute();
			});

			// 値のコンバート
			convertValues(resultList);

			Map<String, Object> retMap = new HashMap<String, Object>();
			retMap.put("list", resultList);
			if (!CollectionUtils.isEmpty(resultList)) {
				retMap.put("first", resultList.get(0));
			} else {
				retMap.put("first", null);
			}

			int cnt = 0;
			if (!CollectionUtils.isEmpty(resultList)
					&& resultList.get(0).get("count") != null) {
				cnt = Integer.parseInt(
						String.valueOf(resultList.get(0).get("count")));
			}
			retMap.put("cnt", cnt);

			result.getReturnData().put(key, retMap);

		}

		return result;

	}

	/**
	 *
	 * 値のコンバート
	 *
	 * @param src
	 */
	private void convertValues(List<Map<String, Object>> src) {

		// コンバート対象がないか、レコード内を走査
		if (!CollectionUtils.isEmpty(src)) {

			PGobject pgVal = null;
			Jdbc4Array arVal = null;

			for (Map<String, Object> mp : src) {

				for (Map.Entry<String, Object> e : mp.entrySet()) {

					if (e.getValue() != null) {

						if (PGobject.class.equals(e.getValue().getClass())
								&& "jsonb".equals(((PGobject) e.getValue())
										.getType().toLowerCase())) {
							pgVal = (PGobject) e.getValue();
							if (JsonUtil
									.isAbleToConverToMap(pgVal.getValue())) {
								mp.put(e.getKey(),
										JsonUtil.jsonToMap(pgVal.getValue()));
							} else if (JsonUtil
									.isAbleToConverToList(pgVal.getValue())) {
								mp.put(e.getKey(),
										JsonUtil.jsonToList(pgVal.getValue()));
							}
						}

						if (Jdbc4Array.class.equals(e.getValue().getClass())) {
							arVal = (Jdbc4Array) e.getValue();
							try {
								mp.put(e.getKey(), ArrayUtil
										.asList((Object[]) arVal.getArray()));
							} catch (SQLException e1) {
								mp.put(e.getKey(), new ArrayList<Object>());
							}
						}

					}

				}

			}

		}

	}

	/**
	 * srcからパラメータを生成します。
	 * 
	 * @param src
	 * @param params
	 * @param paramTypes
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private void generateParams(Map<String, Object> src,
								Map<String, Object> params,
								Map<String, Object> paramTypes,
								ActionDto dto) {

		for (Map.Entry<String, Object> param : src.entrySet()) {

			String key = (String) param.getKey();

			Map mp = (Map) param.getValue();
			String type = String.valueOf(mp.get("type"));
			String dataType = (String) mp.getOrDefault("data_type", "");
			if (StringUtils.isEmpty(dataType)) {
				dataType = DataTypes.TEXT;
			}

			params.put(key,
					ParamUtil.getParamValueByType(type, mp.get("val"), dto));

			Map<String, String> typeDataTypeMap = new HashMap<String, String>();
			typeDataTypeMap.put("type", type);
			typeDataTypeMap.put("data_type", dataType);
			paramTypes.put(key, typeDataTypeMap);

		}

	}

	/**
	 * sqlFileDirPathを取得します。
	 * 
	 * @return sqlFileDirPath
	 */
	public String getSqlFileDirPath() {
		if (sqlFileDirPath == null) {
			sqlFileDirPath = "";
		}
		return sqlFileDirPath;
	}

	/**
	 * sqlFileDirPathを設定します。
	 * 
	 * @param sqlFileDirPath
	 */
	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	/**
	 * sqlFilesを取得します。
	 * 
	 * @return sqlFiles
	 */
	public List<Map<String, Object>> getSqlFiles() {
		return sqlFiles;
	}

	/**
	 * sqlFilesを設定します。
	 * 
	 * @param sqlFiles
	 */
	public void setSqlFiles(List<Map<String, Object>> sqlFiles) {
		this.sqlFiles = sqlFiles;
	}

	/**
	 * searchFormsを取得します。
	 * 
	 * @return searchForms
	 */
	public Map<String, Object> getSearchForms() {
		return searchForms;
	}

	/**
	 * searchFormsを設定します。
	 * 
	 * @param searchForms
	 */
	public void setSearchForms(Map<String, Object> searchForms) {
		this.searchForms = searchForms;
	}

}
