package jp.ill.photon.module.db;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.doma.SqlSelectQueryFactory;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.MapParam;
import jp.ill.photon.model.SearchForm;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;

/**
 * 指定したファイルリストのファイル名毎にSQLを実行するモジュール
 * 
 * @author h_tanaka
 *
 */
public class SqlByFileListModule implements PhotonModule {

	@ModuleParam(required = true)
	private String srcDirPath;

	@ModuleParam(required = true)
	private String srcFileNamePattern;

	@ModuleParam(required = false)
	private MapParam srcFileNameParts;

	@ModuleParam(required = true)
	private String destDirPath;

	@ModuleParam(required = true, domainObject = true)
	private SearchForm searchForm;

	@ModuleParam(required = true)
	private String sqlFileDirPath;

	@ModuleParam(required = true)
	private String sqlFilePath;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(SqlByFileListModule.class);

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		SqlSelectQuery selectQuery = SqlSelectQueryFactory.newInstance()
				.createSelectQueryFromFile(getSearchForm().getParamMap(),
						context.getDto(), getSqlFileDirPath(),
						getSqlFilePath());

		// 検索処理を実行
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		// ソースフォルダ取得
		Path srcPath = Paths.get(getSrcDirPath());
		if (!Files.exists(srcPath)) {
			logger.info("コピー元フォルダが存在しません：" + srcPath.toString());
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_nosrc");
			return result;
		}

		// ファイルごとにクエリを実行
		long count = 0;
		Pattern pattern = Pattern.compile(getSrcFileNamePattern());
		Map<String, Object> paramMap = createParams(
				getSrcFileNameParts().getParamMap(), context);
		try {
			count = Files.list(srcPath).count();
			Files.list(srcPath).forEach(new Consumer<Path>() {
				@Override
				public void accept(Path t) {
					Map<String, String> filePartMap = createFilePartMap(t,
							paramMap, pattern);
					selectQuery.addParameter("fields", Map.class, filePartMap);
					selectQuery.prepare();
					tm.required(() -> {
						SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
								selectQuery, new MapResultListHandler(
										MapKeyNamingType.NONE));
						return command.execute();
					});
				}
			});
		} catch (IOException e) {
			logger.info("ファイル取得失敗しました");
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setResultCode("error_io");
		}

		// 結果オブジェクト生成
		result.getReturnData().put("cnt", count);

		return result;
	}

	/**
	 * ファイルパス部品をファイル名パターンパラメータでマップに変換
	 * 
	 * @param path
	 * @param paramMap
	 * @param fileNamePattern
	 * @return
	 */
	protected Map<String, String> createFilePartMap(Path path,
													Map<String, Object> paramMap,
													Pattern pattern) {

		Map<String, String> partMap = new HashMap<>();
		partMap.put("_file_name_", path.getFileName().toString());

		Matcher m = pattern.matcher(path.getFileName().toString());
		if (m.matches()) {
			String partName = null;
			for (int i = 1; i <= m.groupCount(); i++) {
				partName = String
						.valueOf(paramMap.getOrDefault(String.valueOf(i), ""));
				if (partName != "") {
					partMap.put(partName, m.group(i));
				}
			}
		}

		return partMap;
	}

	/**
	 * パラメータデータを生成する
	 * 
	 * @param srcParams
	 * @param context
	 * @return
	 */
	@SuppressWarnings({ "rawtypes" })
	protected Map<String, Object> createParams(	Map<String, Object> srcParams,
												ModuleContext context) {
		// パラメータデータを取得
		Map<String, Object> params = new HashMap<String, Object>();
		for (Map.Entry<String, Object> param : srcParams.entrySet()) {

			String type = String.valueOf(((Map) param.getValue()).get("type"));
			params.put((String) param.getKey(),
					ParamUtil.getParamValueByType(type,
							((Map) param.getValue()).get("val"),
							context.getDto()));
		}

		return params;
	}

	public SearchForm getSearchForm() {
		return searchForm;
	}

	public void setSearchForm(SearchForm searchForm) {
		this.searchForm = searchForm;
	}

	public String getSqlFileDirPath() {
		if (sqlFileDirPath == null) {
			sqlFileDirPath = "";
		}
		return sqlFileDirPath;
	}

	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	public String getSqlFilePath() {
		return sqlFilePath;
	}

	public void setSqlFilePath(String sqlFilePath) {
		this.sqlFilePath = sqlFilePath;
	}

	public String getSrcDirPath() {
		return srcDirPath;
	}

	public void setSrcDirPath(String srcDirPath) {
		this.srcDirPath = srcDirPath;
	}

	public String getSrcFileNamePattern() {
		return srcFileNamePattern;
	}

	public void setSrcFileNamePattern(String srcFileNamePattern) {
		this.srcFileNamePattern = srcFileNamePattern;
	}

	public String getDestDirPath() {
		return destDirPath;
	}

	public void setDestDirPath(String destDirPath) {
		this.destDirPath = destDirPath;
	}

	public MapParam getSrcFileNameParts() {
		return srcFileNameParts;
	}

	public void setSrcFileNameParts(MapParam srcFileNameParts) {
		this.srcFileNameParts = srcFileNameParts;
	}
}
