package jp.ill.photon.module.db;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.SearchForm;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;

import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;



/**
 * カラム設定を読んで、データ取得処理を実施する基底クラス
 *
 * @author m_fukukawa
 *
 */
public abstract class SettingDataToSearchProcessBaseModule extends SettingDataToManipulateQueryBaseModule implements PhotonModule {

	/* クラス内の複数メソッドで使用する変数 */
	protected String selPopupCd;

	/* 定数 */
	protected static class TextNumberDigitDiv {
		public static final String NONE = "0";
		public static final String FIX = "1";
		public static final String AO = "2";
	}

	/**
	 * 基本的なパラメータを取得
	 *
	 * @return 基本的なパラメータ
	 *
	 * */
	protected void setBasicParams() {
		SearchForm searchForm = getSearchForm();
		if (searchForm != null) {
			Map<String, Object> mp = searchForm.getParamMap();
			page = ParamUtil.getParamStrValueByType(mp.get("page"), dto);
			limit = ParamUtil.getParamStrValueByType(mp.get("limit"), dto);
			tenantId = ParamUtil.getParamStrValueByType(mp.get("tenantId"), dto);
			editTableCd = ParamUtil.getParamStrValueByType(mp.get("editTableCd"), dto);
			selPopupCd = ParamUtil.getParamStrValueByType(mp.get("selPopupCd"), dto);
			sort = ParamUtil.getParamStrValueByType(mp.get("sort"), dto);
		}
	}

	/**
	 * 検索結果データ取得
	 *
	 * @return 検索結果データ
	 * @throws PhotonModuleException
	 *
	 * */
	protected List<Map<String, Object>> getRows() throws PhotonModuleException {

		// 検索処理を実行
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();

		List<Map<String, Object>> resultList = tm.required(() -> {
			return dao.selectEditTableDataList(tenantId, editTableCd, columns, where, sort, page, limit);
		});

		return resultList;

	}

	/**
	 * 検索結果データ取得
	 *
	 * @param 行データ
	 *
	 * */
	protected Map<String, Object> formatRow(Map<String, Object> row) {

		Map<String, Object> convertedRow = row;

		// マップ内容の要素からカラム名を取得
		List<Map<String, Object>> columnMap = getColumnMap();
		for (Map<String, Object> column : columnMap) {

			if (column != null) {

				String dbColumnName = (String)column.get("col_cd");

				if (!StringUtils.isEmpty(dbColumnName)) {

					// 桁数の変換
					convertedRow = convValueDigit(convertedRow, column);
					// 日付フォーマット変換
					convertedRow = convValueDateFormat(convertedRow, column);
					// 0埋め
					// convertedRow = convValueDateFormat(convertedRow, column);

				}

			}

		}

		return convertedRow;

	}

	/**
	 * 桁数の変換
	 *
	 * @param 行データ
	 * @param 行の設定情報
	 *
	 * @return 変換後の行データ
	 *
	 * */
	@SuppressWarnings("unchecked")
	protected Map<String, Object> convValueDigit(Map<String, Object> row, Map<String, Object> column) {

		Map<String, Object> result = row;

		String dbColumnName = (String)column.get("col_cd");

		//String camelName = StringUtil.snakeToCamel(dbColumnName);
		String dataValue = StringUtils.defaultString((String)row.get(dbColumnName), "");
		String resultValue = dataValue;

		// 桁数設定があるかチェック
		String textNumberDigitDiv = StringUtils.defaultString((String)column.get("text_number_digit_div"), "");

		if (TextNumberDigitDiv.FIX.equals(textNumberDigitDiv)) {

			// 固定桁数
			String digitNum = StringUtils.defaultString((String)column.get("text_number_digit_integral"), "");
			String decimalPointNum = StringUtils.defaultString((String)column.get("text_number_digit_decimal"), "");
			resultValue = AoUtil.convAoRounding(dataValue, "1", digitNum, decimalPointNum, "0", AoUtil.DispPattern.PATTERN_NO_COMMA);

		} else if (TextNumberDigitDiv.AO.equals(textNumberDigitDiv)) {

			// アラジンシステム設定の桁数
			String digitNumAoName = StringUtils.defaultString((String)column.get("ao_set_name_digit_integral"), "");
			String decimalPointNumAoName = StringUtils.defaultString((String)column.get("ao_set_name_digit_decimal"), "");
			Map<String, Object> common = getCommon();
			Map<String, Object> aladdinSetting = (Map<String, Object>)common.get("aladdinsetting");
			String digitNum = getValueFromSystemSetting(aladdinSetting, digitNumAoName, "value");
			String decimalPointNum = getValueFromSystemSetting(aladdinSetting, decimalPointNumAoName, "value");
			resultValue = AoUtil.convAoRounding(dataValue, "1", digitNum, decimalPointNum, "0", AoUtil.DispPattern.PATTERN_NO_COMMA);

		} else {

			// それ以外の値はそのまま設定
			resultValue = dataValue;

		}

		// 取得した値を新しいマップに設定
		result.put(dbColumnName, resultValue);

		return result;

	}

	/**
	 * 日付フォーマット変換
	 *
	 * @param 行データ
	 * @param 行の設定情報
	 *
	 * @return 変換後の行データ
	 *
	 * */
	protected Map<String, Object> convValueDateFormat(Map<String, Object> row, Map<String, Object> column) {

		Map<String, Object> result = row;

		String dbColumnName = (String)column.get("col_cd");

		// String camelName = StringUtil.snakeToCamel(dbColumnName);
		String dataValue = StringUtils.defaultString((String)row.get(dbColumnName), "");
		String dataInputType = StringUtils.defaultString((String)column.get("data_input_type"), "");
		String resultValue = dataValue;

		if (DataInputType.DATE.equals(dataInputType)) {

			// 日付の場合
			if (!StringUtils.isEmpty(dataValue)) {
				dataValue = dataValue.replace("-", "/");
				resultValue = dataValue.split(" ")[0];
			}

		} else if (DataInputType.DATETIME.equals(dataInputType)) {

			// 日時の場合
			if (!StringUtils.isEmpty(dataValue)) {

				dataValue = dataValue.replace("-", "/");
				String[] dateSplit = dataValue.split(" ");
				String time = "";

				if (dateSplit.length >= 2) {
					time = dateSplit[1];
					if (time.length() > 8) {
						time = time.substring(0, 8);
					}

					resultValue = dateSplit[0] + " " + time;

				}

			}

		}

		// 取得した値を新しいマップに設定
		result.put(dbColumnName, resultValue);

		return result;
	}

	/**
	 * 0埋め
	 *
	 * @param row
	 * @param column
	 * @return
	 */
	protected Map<String, Object> zeroPad(Map<String, Object> row, Map<String, Object> column) {

		Map<String, Object> result = row;

		String dbColumnName = (String)column.get("col_cd");

		// String camelName = StringUtil.snakeToCamel(dbColumnName);
		String dataValue = StringUtils.defaultString((String)row.get(dbColumnName), "");
		String resultValue = dataValue;


		// 取得した値を新しいマップに設定
		result.put(dbColumnName, resultValue);

		return result;

	}

	/***
	 *
	 * 添付ファイルデータを取得
	 *
	 * @param guid
	 * @param col_cd
	 * @return
	 */
	protected Map<String, Object> getAttachment(String guid, String col_cd) {

		// 検索処理を実行
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();

		String cond = JsonUtil.mapToJson(new HashMap<String, Object>(){{
			put("edit_table_cd",editTableCd);
			put("col_cd", col_cd);
			put("ref_guid", guid);
		}});

		Map<String, Object> result = tm.required(() -> {
			return dao.getEditTableRow(tenantId, "attachment", "val->>$$file_name$$ AS file_name", cond);
		});

		return result;

	}

	/**
	 * columnMapを取得します。
	 * @return columnMap
	 */
	public abstract List<Map<String, Object>> getColumnMap();

	/**
	 * searchMapを取得します。
	 * @return searchMap
	 */
	public abstract List<Map<String, Object>> getSearchMap();

	/**
	 * sortDataを取得します。
	 * @return sortData
	 */
	public abstract List<Map<String, Object>> getSortData();

	/**
	 * keyColumnsを取得します。
	 * @return keyColumns
	 */
	public abstract List<Map<String, Object>> getKeyColumns();

	/**
	 * commonを取得します。
	 * @return common
	 */
	public abstract Map<String, Object> getCommon();

	/**
	 * searchFormを取得します。
	 * @return searchForm
	 */
	public abstract SearchForm getSearchForm();

}
