package jp.ill.photon.module.db;

import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.SearchForm;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ActionUtil;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.AoUtil.DispPattern;
import jp.ill.photon.util.CheckUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.SQLUtil;
import jp.ill.photon.util.StringUtil;

/**
 * カラム設定を読んで、データ取得処理を実施するクラス（一覧画面用）
 *
 * @author m_fukukawa
 *
 */
public class SettingDataToSearchListProcessModule
		extends SettingDataToSearchProcessBaseModule implements PhotonModule {

	/* モジュールパラメータ */
	@ModuleParam(required = true)
	protected List<Map<String, Object>> columnMap;

	@ModuleParam(required = false)
	protected List<Map<String, Object>> searchMap;

	@ModuleParam(required = false)
	protected List<Map<String, Object>> sortData;

	@ModuleParam(required = false)
	protected List<Map<String, Object>> keyColumns;

	@ModuleParam(required = false)
	protected Map<String, Object> common;

	@ModuleParam(required = true)
	protected SearchForm searchForm;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "static", transferVal = "0")
	private String deleteType;

	/* private定数 */
	private static final int ITEM_CATEGORY_CNT = 20;

	private static final String REPLACE_MARKER = "{1}";

	private static class SpecialSearchInputType {
		private static final String NOT = "0";
	}

	private static class SearchInputType {
		private static final String CHECKBOX = "2";
		private static final String RANGETEXT = "31";
		private static final String RANGENUM = "32";
		private static final String RANGEDAY = "33";
		private static final String FOREIGN_MULTI = "26";
		private static final String HIDDEN = "99";
	}

	private static class TextDataFormat {
		public static final String NUMERIC = "1";
	}

	private static class FractionDiv {
		public static final String DOWN = "0";
	}

	@Override
	public ModuleResult executeCustom(ModuleContext context)
			throws PhotonModuleException {

		ModuleResult result = new ModuleResult();

		setBasicParams();

		// クエリ作成
		buildSelectFiedldStr(columnMap, keyColumns);
		buildColumnsBySearchMap();
		buildSortBySortNo();

		// 検索データ取得
		List<Map<String, Object>> columnData = getRows();

		// 検索データが取得できた場合は特殊なデータを作成
		if (!CollectionUtils.isEmpty(columnData)) {
			for (Map<String, Object> mp : columnData) {
				mp = formatRow(mp);
			}
		}

		for (Map<String, Object> column : columnMap) {

			for (int count = 0; count < columnData.size(); count++) {

				String dbColumnName = (String) column.get("col_cd");
				Map<String, Object> data = columnData.get(count);

				// 画像の表示
				data = generateImpPath(data, column);

				// 値
				String value = String.valueOf(
						data.get(StringUtil.snakeToCamel(dbColumnName)));

				if (!StringUtils.isEmpty(value)) {
					data = formatNumber(data, value, column);
				}

				// リンクフラグとキーのプロパティをセット
				data.put("data_link", column.get("select_query"));

			}

		}

		// フォームデータのセット
		// 結果オブジェクト生成
		int cnt = 0;
		if (!CollectionUtils.isEmpty(columnData)) {

			result.getReturnData().put("first", columnData.get(0));

			if (columnData.get(0).get("count") != null) {
				cnt = Integer.parseInt(
						String.valueOf(columnData.get(0).get("count")));
			}

		} else {

			result.getReturnData().put("first", null);

		}

		result.getReturnData().put("cnt", cnt);
		result.getReturnData().put("list", columnData);

		return result;

	}

	/**
	 * 条件部を作成
	 *
	 * @return 条件部
	 *
	 */
	private void buildColumnsBySearchMap() {

		String result = null;
		StringBuffer sbWhere = new StringBuffer();

		// フラグが立っているものは見ないようにする
		sbWhere.append("is_deleted = 0");

		if (!CollectionUtils.isEmpty(searchMap)) {

			for (Map<String, Object> mp : searchMap) {

				String key = (String) mp.get("cond_no");
				String searchInputType = (String) mp
						.getOrDefault("data_input_type", "0");

				String referDto = (String) mp.getOrDefault("refer_dto", "");

				String whereQuery = (String) mp.get("where_query");
				if (!StringUtils.isEmpty(whereQuery)) {

					// whereQueryの中に"{1}"が見つからない場合は、そのまま条件に加える
					if (whereQuery.indexOf(REPLACE_MARKER) == -1) {
						if (!StringUtils.isEmpty(sbWhere.toString())) {
							sbWhere.append(" AND ");
						}
						sbWhere.append(whereQuery);
						continue;
					}

					// 範囲検索の場合
					if (SearchInputType.RANGETEXT.equals(searchInputType)
							|| SearchInputType.RANGENUM.equals(searchInputType)
							|| SearchInputType.RANGEDAY
									.equals(searchInputType)) {

						// t_bd_edit_search_table.where_queryの値を取得
						// 「"s"_ + アクションパラメータ名」、「"e_" +
						// アクションパラメータ名」をキーにそれぞれreqestオブジェクトから値を取得
						// "{1}"の記述がある位置にセット
						// 頭に" AND "の文字列を追加し、Where句文字列に追加

						String whereQueryEnd = (String) mp
								.get("where_query_end");

						String searchDataFirst = ActionUtil
								.getActionParamString(request,
										ColumnName.COLUMN_NAME + key + "_from",
										"");
						if (!StringUtils.isEmpty(referDto)) {
							searchDataFirst = (String) ParamUtil
									.getParamValueByType("dto",
											referDto + "_from", dto);
						}
						String searchDataEnd = ActionUtil.getActionParamString(
								request, ColumnName.COLUMN_NAME + key + "_to",
								"");
						if (!StringUtils.isEmpty(referDto)) {
							searchDataEnd = (String) ParamUtil
									.getParamValueByType("dto",
											referDto + "_to", dto);
						}

						if (!StringUtils.isEmpty(searchDataFirst)) {
							String whereQueryFormat = whereQuery.replace(
									REPLACE_MARKER,
									SQLUtil.escapeSqlString(searchDataFirst));
							if (!StringUtils.isEmpty(sbWhere.toString())) {
								sbWhere.append(" AND ");
							}
							sbWhere.append(whereQueryFormat);
						}

						if (!StringUtils.isEmpty(searchDataEnd)) {
							String whereQueryFormat = whereQueryEnd.replace(
									REPLACE_MARKER,
									SQLUtil.escapeSqlString(searchDataEnd));
							if (!StringUtils.isEmpty(sbWhere.toString())) {
								sbWhere.append(" AND ");
							}
							sbWhere.append(whereQueryFormat);
						}

						// チェックボックスの場合
					} else if (SearchInputType.CHECKBOX
							.equals(searchInputType)) {

						// t_bd_edit_search_table.where_queryの値を取得
						// アクションパラメータをrequestオブジェクトから値を取得する
						// 値の分だけwhere_queryの"{1}"の記述がある位置に値を入れ替えた文字列を生成し、
						// それらを" OR "でつなげ、
						// 頭に" AND "の文字列を追加し、Where句文字列に追加

						// 例）
						// チェックボックス1(name="c_001" value="1" checked)
						// チェックボックス2(name="c_001" value="2")
						// チェックボックス3(name="c_001" value="3" checked)
						// チェックボックス4(name="c_001" value="4")
						// があるとき、

						// 生成されるSQLは、
						// AND ((c_001 = '1') OR (c_001 = '3'))

						StringBuffer ret = new StringBuffer();

						String[] arr = (String[]) request
								.get(ColumnName.COLUMN_NAME + key);
						if (!StringUtils.isEmpty(referDto)) {
							arr = (String[]) ParamUtil
									.getParamValueByType("dto", referDto, dto);
						}
						if (arr != null) {

							for (int i = 0; i < arr.length; i++) {

								String searchData = arr[i];

								if (!StringUtils.isEmpty(ret.toString())) {
									ret.append(" OR ");
								}
								if (!StringUtils.isEmpty(searchData)) {
									ret.append(" ( ");
									ret.append(
											whereQuery.replace(REPLACE_MARKER,
													SQLUtil.escapeSqlString(
															searchData)));
									ret.append(" ) ");
								}

							}

							if (!StringUtils.isEmpty(ret.toString())) {
								ret.insert(0, " ( ");
								ret.append(" ) ");
								if (!StringUtils.isEmpty(sbWhere.toString())) {
									sbWhere.append(" AND ");
								}
								sbWhere.append(ret.toString());
							}

						}

						// それ以外の場合
					} else {

						// t_bd_edit_search_table.where_queryの値を取得
						// アクションパラメータをrequestオブジェクトから値を取得し、
						// "{1}"の記述がある位置にセット
						// 頭に" AND "の文字列を追加し、Where句文字列に追加

						String searchData = "";
						if (SearchInputType.FOREIGN_MULTI
								.equals(searchInputType)) {
							searchData = ActionUtil.getActionParamString(
									request, ColumnName.COLUMN_NAME + key, ", ",
									Integer.parseInt(searchInputType), true);
							if (!StringUtils.isEmpty(referDto)) {
								searchData = (String) ParamUtil
										.getParamValueByType("dto", referDto,
												dto);
							}
						} else {
							searchData = ActionUtil.getActionParamString(
									request, ColumnName.COLUMN_NAME + key, "");
							if (!StringUtils.isEmpty(referDto)) {
								searchData = (String) ParamUtil
										.getParamValueByType("dto", referDto,
												dto);
							}
						}

						if (!CheckUtil.isEmpty(searchData)) {
							String whereQueryFormat = whereQuery.replace(
									REPLACE_MARKER,
									SQLUtil.escapeSqlString(searchData));
							if (!CheckUtil.isEmpty(sbWhere.toString())) {
								sbWhere.append(" AND ");
							}
							sbWhere.append(whereQueryFormat);
						} else {
							if (SearchInputType.HIDDEN.equals(searchInputType)
									&& !StringUtils.isEmpty(whereQuery)) {
								if (!CheckUtil.isEmpty(sbWhere.toString())) {
									sbWhere.append(" AND ");
								}
								sbWhere.append(whereQuery);
							}
						}

					}

				}
			}

		}

		result = sbWhere.toString();

		// 条件が無い場合はSQLで使用しないようにNULLを設定
		if ("".equals(result)) {
			result = null;
		}

		where = result;

	}

	/**
	 * ソート部を作成
	 *
	 * @return ソート部
	 *
	 */
	private void buildSortBySortNo() {

		String result = "";

		Map<String, Object> data = null;

		if (!CollectionUtils.isEmpty(sortData)) {

			if (!StringUtils.isEmpty(sort)) {
				for (Map<String, Object> mp : sortData) {
					if (sort.equals(StringUtil
							.defaultString(mp.get("record_no"), ""))) {
						data = mp;
						break;
					}
				}
			}

			if (data == null) {
				// 一致するものがなければ1件目
				data = sortData.get(0);
			}

			result = StringUtil.defaultString(data.get("order_query"), "");

		}

		sort = result;

	}

	/**
	 * 画像表示設定
	 *
	 * @param 行データ
	 * @param 行の設定情報
	 *
	 * @return 変換後の行データ
	 * @throws PhotonModuleException
	 *
	 */
	private Map<String, Object> generateImpPath(Map<String, Object> row,
												Map<String, Object> column)
			throws PhotonModuleException {

		Map<String, Object> result = row;

		String key = (String) column.get("col_cd");

		// file_dir
		String fileDir = StringUtils.defaultString(
				(String) column.get("file_context_path_system_setting_id"), "");
		if (!StringUtils.isEmpty(fileDir)) {
			String rootPath = getValueFromSystemSetting(
					common.get("systemsetting"), fileDir, "note");
			result.put(key + "_file_dir", rootPath);
		}

		// file_sub_dir
		String fileSubDir = "";

		String select = StringUtils.defaultString(
				(String) column.get("sub_dir_name_select_query"), "");
		if (!StringUtils.isEmpty(select)) {
			fileSubDir = getSubDir(select,
					makeKeyJsonStr(keyColumns, result, false));
		}
		if (!StringUtils.isEmpty(fileSubDir)) {
			result.put(key + "_file_sub_dir", fileSubDir);
		}

		// file_name
		result.put(key + "_file_name", result.get(key));

		return result;

	}

	/**
	 * 画像格納サブディレクトリを抽出
	 * 
	 * @throws PhotonModuleException
	 *
	 */
	private String getSubDir(String columns, String where)
			throws PhotonModuleException {

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();

		List<Map<String, Object>> resultList = tm.required(() -> {
			return dao.selectRowForDownload(tenantId, editTableCd, columns,
					where);
		});

		String ret = "";
		if (!CollectionUtils.isEmpty(resultList)) {
			ret = (String) resultList.get(0).get("file_dir");
		}

		return ret;

	}

	/**
	 * 数値フォーマット設定
	 *
	 * @param 行データ
	 * @param 行の設定情報
	 *
	 * @return 変換後の行データ
	 *
	 */
	@SuppressWarnings("unchecked")
	private Map<String, Object> formatNumber(	Map<String, Object> row,
												String value,
												Map<String, Object> column) {

		Map<String, Object> result = row;

		String textDataFormat = (String) column.get("text_data_format");
		String textNumberDigitDiv = (String) column
				.get("text_number_digit_div");
		String dbColumnName = (String) column.get("col_cd");
		// 小数点フォーマット
		if (TextDataFormat.NUMERIC.equals(textDataFormat)
				&& TextNumberDigitDiv.FIX.equals(textNumberDigitDiv)) {

			// 整数桁
			String digitNum = String
					.valueOf(column.get("text_number_digit_integral"));
			// 小数桁
			String decimalPointNum = String
					.valueOf(column.get("text_number_digit_decimal"));
			// 端数処理
			String roundingDiv = FractionDiv.DOWN;

			// フォーマット変更
			String strResult = AoUtil.convAoRounding(value, "1", digitNum,
					decimalPointNum, roundingDiv, DispPattern.PATTERN_1);
			// columnData[count].put(StringUtil.snakeToCamel(col.getDbColumnName()),
			// strResult);
			result.put(dbColumnName, strResult);

		}

		// 数値チェック（桁数あり AOシステム設定値でチェック）
		if (TextDataFormat.NUMERIC.equals(textDataFormat)
				&& TextNumberDigitDiv.AO.equals(textNumberDigitDiv)) {

			String digitNumAoName = StringUtils.defaultString(
					(String) column.get("ao_set_name_digit_integral"), "");
			String decimalPointNumAoName = StringUtils.defaultString(
					(String) column.get("ao_set_name_digit_decimal"), "");
			Map<String, Object> aladdinSetting = (Map<String, Object>) common
					.get("aladdinsetting");

			// 整数桁
			String digitNum = getValueFromSystemSetting(aladdinSetting,
					digitNumAoName, "value");
			// 小数桁
			String decimalPointNum = getValueFromSystemSetting(aladdinSetting,
					decimalPointNumAoName, "value");
			// 端数処理
			String roundingDiv = FractionDiv.DOWN;

			// フォーマット変更
			String strResult = AoUtil.convAoRounding(value, "1", digitNum,
					decimalPointNum, roundingDiv, DispPattern.PATTERN_1);
			// columnData[count].put(StringUtil.snakeToCamel(col.getDbColumnName()),
			// strResult);
			result.put(dbColumnName, strResult);

		}

		return result;

	}

	/**
	 * columnMapを取得します。
	 * 
	 * @return columnMap
	 */
	public List<Map<String, Object>> getColumnMap() {
		return columnMap;
	}

	/**
	 * columnMapを設定します。
	 * 
	 * @param columnMap
	 */
	public void setColumnMap(List<Map<String, Object>> columnMap) {
		this.columnMap = columnMap;
	}

	/**
	 * searchMapを取得します。
	 * 
	 * @return searchMap
	 */
	public List<Map<String, Object>> getSearchMap() {
		return searchMap;
	}

	/**
	 * searchMapを設定します。
	 * 
	 * @param searchMap
	 */
	public void setSearchMap(List<Map<String, Object>> searchMap) {
		this.searchMap = searchMap;
	}

	/**
	 * sortDataを取得します。
	 * 
	 * @return sortData
	 */
	public List<Map<String, Object>> getSortData() {
		return sortData;
	}

	/**
	 * sortDataを設定します。
	 * 
	 * @param sortData
	 */
	public void setSortData(List<Map<String, Object>> sortData) {
		this.sortData = sortData;
	}

	/**
	 * keyColumnsを取得します。
	 * 
	 * @return keyColumns
	 */
	public List<Map<String, Object>> getKeyColumns() {
		return keyColumns;
	}

	/**
	 * keyColumnsを設定します。
	 * 
	 * @param keyColumns
	 */
	public void setKeyColumns(List<Map<String, Object>> keyColumns) {
		this.keyColumns = keyColumns;
	}

	/**
	 * commonを取得します。
	 * 
	 * @return common
	 */
	public Map<String, Object> getCommon() {
		return common;
	}

	/**
	 * commonを設定します。
	 * 
	 * @param common
	 */
	public void setCommon(Map<String, Object> common) {
		this.common = common;
	}

	/**
	 * searchFormを取得します。
	 * 
	 * @return searchForm
	 */
	public SearchForm getSearchForm() {
		return searchForm;
	}

	/**
	 * searchFormを設定します。
	 * 
	 * @param searchForm
	 */
	public void setSearchForm(SearchForm searchForm) {
		this.searchForm = searchForm;
	}

	/**
	 * deleteTypeを取得します。
	 * 
	 * @return deleteType
	 */
	public String getDeleteType() {
		return deleteType;
	}

	/**
	 * deleteTypeを設定します。
	 * 
	 * @param deleteType
	 */
	public void setDeleteType(String deleteType) {
		this.deleteType = deleteType;
	}

}
