package jp.ill.photon.module.db;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ActionUtil;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;



/**
 * 設定テーブルから、バリデーションの設定を組み立てる
 *
 * @author m_fukukawa
 * @see jp.ill.photon.module.validation.ParamValidateModule
 *
 */
public class SettingDataToParamValidationModule extends SettingDataToManipulateQueryBaseModule implements PhotonModule {

	/* モジュールパラメータ */
	@ModuleParam(required=true)
	private String tenantId;

	@ModuleParam(required=false)
	private Map<String, Object> common;

	@ModuleParam(required=true)
	private List<Map<String, Object>> settingData;

	/* クラス内の複数メソッドで使用する変数 */
	private boolean isUpdateProcess = false;
	private List<Map<String, Object>> dispColumns;
	private List<Map<String, Object>> keyColumns;
	private List<String> keyColumnCd;
	private List<Map<String, Object>> exListColumns;

	@Override
	public ModuleResult executeCustom(ModuleContext context) {

		// 設定情報から、各種リストを取得
		setBasicValuesFromRequestParam();
		setSettingItems();

		Map<String, Object> validationSetting = new HashMap<String, Object>();
		validationSetting.put("params", makeParams());
		//validationSetting.put("after", makeAfter());

		// フォームデータのセット
		ModuleResult result = new ModuleResult();
		result.getReturnData().put("validation_setting", validationSetting);
		result.getReturnData().put("list_for_check", makeListForCheck());
		result.getReturnData().put("compare_strings", makeCompareStrings());
		result.getReturnData().put("exec_validation", makeExecSetting());
		result.getReturnData().put("map_for_check", makeMapForCheck());
		List<Map<String, Object>> list = makeKeyNameList();
		result.getReturnData().put("key_name_list", list);
		result.getReturnData().put("key_name_string", makeKeyNameString(list));

		return result;

	}

	/**
	 * 基本的なパラメータを取得
	 *
	 * @return 基本的なパラメータ
	 *
	 * */
	protected void setBasicValuesFromRequestParam() {
		if (StringUtils.isEmpty(tenantId)) {
			tenantId = ActionUtil.getActionParamString(request, "tenant_id", "");
		}
		editTableCd = ActionUtil.getActionParamString(request, "edit_table_cd", "");
		String upFlg = ActionUtil.getActionParamString(request, "up_flg", "");
		isUpdateProcess = ( "1".equals(upFlg) );
	}

	/**
	 * 設定情報から、<br />
	 * 画面に表示されていた項目のリスト、<br />
	 * キーの項目のリスト、<br />
	 * 別途リストも取得する必要のある項目のリスト<br />
	 * を取得
	 */
	private void setSettingItems() {

		if (!CollectionUtils.isEmpty(settingData)) {

			for (Map<String, Object> mp : settingData) {

				// 画面に表示されていた項目のリスト
				if (dispColumns == null) {
					dispColumns = new ArrayList<Map<String, Object>>();
				}
				dispColumns.add(mp);

				// キーの項目のリスト
				if (!"0".equals((String)mp.getOrDefault("key_no", "0"))) {
					if (keyColumns == null) {
						keyColumns = new ArrayList<Map<String, Object>>();
					}
					keyColumns.add(mp);
					if (keyColumnCd == null) {
						keyColumnCd = new ArrayList<String>();
					}
					keyColumnCd.add(getKeyName(mp));
				}

				// 別途リストも取得する必要のある項目のリスト
				String dataInputtype = (String)mp.get("data_input_type");
				if(DataInputType.RADIO.equals(dataInputtype) ||
					DataInputType.COMBOBOX.equals(dataInputtype) ||
					DataInputType.POPUP.equals(dataInputtype)) {
					if (exListColumns == null) {
						exListColumns = new ArrayList<Map<String, Object>>();
					}
					exListColumns.add(mp);
				}

			}

		}

	}

	/***
	 *
	 * キー項目のcol_cdとカラム論理名の組み合わせ作成
	 *
	 * @return
	 */
	private List<Map<String, Object>> makeKeyNameList() {

		List<Map<String, Object>> ret = new ArrayList<Map<String, Object>>();

		if (!CollectionUtils.isEmpty(keyColumns)) {

			for (Map<String, Object> mp : keyColumns) {

				ret.add(new HashMap<String, Object>(){{
					put("col_cd", getKeyName(mp));
					put("name", getCaption(mp, common));
				}});

			}

		}

		return ret;

	}

	/***
	 *
	 * キー項目論理名を"、"で結合した文字列作成
	 *
	 * @return
	 */
	private String makeKeyNameString(List<Map<String, Object>> list) {

		StringJoiner sj = new StringJoiner("、");

		if (!CollectionUtils.isEmpty(list)) {

			for (Map<String, Object> mp : list) {

				sj.add((String) mp.get("name"));


			}

		}

		return sj.toString();

	}

	/***
	 * バリデーションモジュールパラメータの"params"部分作成
	 *
	 * @return
	 */
	private Map<String, Object> makeParams() {

		Map<String, Object> ret = new HashMap<String, Object>();

		Map<String, Object> paramJsonVal = new HashMap<String, Object>();
		Map<String, Object> paramNamesVal = new HashMap<String, Object>();

		if (!CollectionUtils.isEmpty(dispColumns)) {

			for (Map<String, Object> mp : dispColumns) {

				String colCd = getKeyName(mp);

				// 新規登録時：入力項目は全項目チェックの対象
				// 更新時：入力項目はキー項目以外がチェックの対象
				if (isAppendable(colCd)) {

					// String validationSet = (String)mp.get("validation_set");
					String validationSet = getValidationSet(mp);
					String dataInputType = (String)mp.get("data_input_type");
					String typeStr = "";
					if (DataInputType.FILE.equals(dataInputType)) {
						typeStr = "param_file";
					} else if (DataInputType.CHECKBOX.equals(dataInputType) || DataInputType.LISTBOX.equals(dataInputType)) {
						typeStr = "param_multi";
					} else {
						typeStr = "param";
					}
					String valStr = "";
					if (DataInputType.FILE.equals(dataInputType)) {
						valStr = colCd + ".file_name";
					} else {
						valStr = colCd;
					}
					String type= typeStr;
					String val = valStr;

					// "validation_set"が設定されていない項目は、バリデーションを実行しないものとして扱う
					if (!StringUtils.isEmpty(validationSet)) {

						paramJsonVal.put(colCd, new HashMap<String, Object>(){{
							put("type", type);
							put("val", val);
							put("validation_set", validationSet);
						}});

						paramNamesVal.put(colCd, new HashMap<String, Object>(){{
							put("type", "static");
							put("val", getCaption(mp, common));
						}});

					}

				}

			}

		}

		ret.put("param_json", paramJsonVal);
		ret.put("param_names", paramNamesVal);

		return ret;

	}

	/***
	 * バリデーションセットを取得
	 *
	 * @param mp
	 * @return
	 */
	private String getValidationSet(Map<String, Object> mp) {

		String ret = "";
		String targetValue = (String)mp.get("target_value");
		String validationSetByValue = (String)mp.get("validation_set_by_value");

		if (StringUtils.isEmpty(targetValue)) {

			// "target_value"が設定されていないときは、"validation_set"の値を返却
			ret = (String)mp.get("validation_set");

		} else {

			if (!StringUtils.isEmpty(validationSetByValue)) {

				// "target_value"が設定されているときは、
				// "validation_set_by_value"のマップの中に"target_value"のキーがないかを探す。
				// あればキーに対応するバリデーションセットを返却する
				// なければ"*"というキーを探す。

				String value = ParamUtil.getParamStrValueByType(JsonUtil.jsonToMap( targetValue ) , dto);

				if (!StringUtils.isEmpty(value)) {

					Map<String, Object> validationSetByValueMap = JsonUtil.jsonToMap( validationSetByValue );
					if (!MapUtils.isEmpty(validationSetByValueMap)) {
						if (validationSetByValueMap.containsKey(value)) {
							ret = (String)validationSetByValueMap.get(value);
						} else {
							ret = (String)validationSetByValueMap.get("*");
						}
					}

				}

			}

		}

		return ret;

	}

	/**
	 * 追加可能かを返却
	 *
	 * @param cd
	 * @return
	 */
	private boolean isAppendable(String cd) {
		return (
				!isUpdateProcess ||
				(
					isUpdateProcess &&
					!CollectionUtils.isEmpty(keyColumnCd) &&
					!keyColumnCd.contains(cd)
				)
		);
	}

	/***
	 * バリデーションモジュールの存在チェックで使うリストの設定作成
	 *
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private Map<String, Object> makeMapForCheck() {

		Map<String, Object> ret = new HashMap<String, Object>();

		if (!CollectionUtils.isEmpty(dispColumns)) {

			for (Map<String, Object> mp : dispColumns) {

				String key = getKeyName(mp);
				String textNumberDigitIntegral = StringUtils.defaultIfEmpty(StringUtil.defaultString(mp.get("text_number_digit_integral"), ""), "0");
				String textNumberDigitDecimal = StringUtils.defaultIfEmpty(StringUtil.defaultString(mp.get("text_number_digit_decimal"), ""), "0");
				String aoSetNameDigitIntegral = StringUtils.defaultIfEmpty(StringUtil.defaultString(mp.get("ao_set_name_digit_integral"), ""), "0");
				String aoSetNameDigitDecimal = StringUtils.defaultIfEmpty(StringUtil.defaultString(mp.get("ao_set_name_digit_decimal"), ""), "0");
				String digitIntegral = "";
				String digitDecimal = "";

				if (!"0".equals(textNumberDigitIntegral) || !"0".equals(textNumberDigitDecimal)) {

					digitIntegral = textNumberDigitIntegral;
					digitDecimal = textNumberDigitDecimal;

				} else if (!"0".equals(aoSetNameDigitIntegral) || !"0".equals(aoSetNameDigitDecimal)) {

					Map<String, Object> aladdinSetting = (Map<String, Object>)common.get("aladdinsetting");

					// 整数桁
					String digitNum = getValueFromSystemSetting(aladdinSetting, aoSetNameDigitIntegral, "value");
					// 小数桁
					String decimalPointNum = getValueFromSystemSetting(aladdinSetting, aoSetNameDigitDecimal, "value");

					digitIntegral = digitNum;
					digitDecimal = decimalPointNum;

				}

				StringBuffer sbValue = new StringBuffer();
				StringBuffer sbPattern = new StringBuffer();

				if (!StringUtils.isEmpty(digitIntegral) || !StringUtils.isEmpty(digitDecimal)) {

					int integral = Integer.parseInt(digitIntegral);
					int decimal = Integer.parseInt(digitDecimal);

					sbValue.append(StringUtils.repeat("9", integral));

					// sbPattern.append("^\\-?(0|[1-9][0-9]{0," + (integral - 1) + "})");
					sbPattern.append("^\\-?(0|[1-9]");
					int integralDigit = integral - 1;
					if (integralDigit > 0) {
						sbPattern.append("[0-9]{0," + integralDigit + "}");
					}
					sbPattern.append(")");

					if (decimal > 0) {

						sbValue.append(".");
						sbValue.append(StringUtils.repeat("9", decimal));

						sbPattern.append("(\\.[0-9]{1," + digitDecimal + "})?");

					}

					sbPattern.append("$");

				}

				if (!StringUtils.isEmpty(sbValue.toString())) {
					Map<String, Object> maxDigitsMap = new HashMap<String, Object>();
					maxDigitsMap.put("max_digits", sbValue.toString());
					maxDigitsMap.put("max_digits_formatted", AoUtil.convDecimalFormat(sbValue.toString()));
					maxDigitsMap.put("min_digits", "-" + sbValue.toString());
					maxDigitsMap.put("min_digits_formatted", "-" + AoUtil.convDecimalFormat(sbValue.toString()));
					maxDigitsMap.put("regex_pattern", sbPattern.toString());
					ret.put(key, maxDigitsMap);
				}

			}

		}

		return ret;

	}

	/***
	 * バリデーションモジュールの存在チェックで使うリストの設定作成
	 *
	 * @return
	 */
	private Map<String, Object> makeListForCheck() {
		Map<String, Object> ret = new HashMap<String, Object>();

		// 入力タイプが
		//      コンボボックス
		//      ラジオボタン
		//      外部参照
		// のものがあれば、存在チェックのためにリストを保持する
		if (!CollectionUtils.isEmpty(exListColumns)) {

			Map<String, Map<String, Object>> queryList = new HashMap<String, Map<String, Object>>();
			for (Map<String, Object> mp : exListColumns) {
				// コンボボックス、ラジオボタン：
				// 外部参照：クエリリストを作成し、あとで実行する
				String dataInputtype = (String)mp.get("data_input_type");
				if (DataInputType.RADIO.equals(dataInputtype) ||
					DataInputType.COMBOBOX.equals(dataInputtype)) {
					String key = getKeyName(mp);
					String val = ActionUtil.getActionParamString(request, key, "");
					// "choice_list"に定義された値を"choice_list_separator"をセパレータとして分割
					// 奇数番目の要素を値、偶数番目の要素をキャプションにしてリストを作成
					List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
					String[] choiceList = getStringArrayFromChoiceList(mp);
					if (choiceList != null) {
						for (int i = 0; i < choiceList.length; i = i + 2) {
							String choiceVal = StringUtils.defaultString(choiceList[i], "");
							String choiceDispVal = ( (choiceList.length > (i + 1))? choiceList[i + 1] : "" );
							if (!StringUtils.isEmpty(choiceDispVal)) {
								if (!StringUtils.isEmpty(val) && val.equals(choiceVal)) {
									list.add(new HashMap<String, Object>(){{
										put(choiceVal, choiceDispVal);
									}});
								}
							}
						}
					}
					ret.put(key, list);
				}
				else if (DataInputType.POPUP.equals(dataInputtype)) {
					String key = getKeyName(mp);
					String val = ActionUtil.getActionParamString(request, key, "");
					// 値が未入力の場合は、チェック対象のリストは取得しない
					// 必須入力の場合は、必須チェックで引っかかるし、必須でない場合は、入力されたときのみチェックすればよいため
					if (!StringUtils.isEmpty(val)) {
						queryList.put(key, mp);
					}
				}

			}

			if (!queryList.isEmpty()) {

				// リスト取得
				// 検索処理を実行
				TransactionManager tm = DomaConfig.singleton().getTransactionManager();
				JsonDataDao dao = new JsonDataDaoImpl();
				tm.required(() -> {

					for (Map.Entry<String, Map<String, Object>> e : queryList.entrySet()) {
						Map<String, Object> vals = e.getValue();
						// sel_popup_edit_table_cdに応じたentity_fieldの情報取得
						String selPopupEditTableCd = (String)vals.get("sel_popup_edit_table_cd");
						if (!StringUtils.isEmpty(selPopupEditTableCd)) {
							List<Map<String, Object>> list = dao.getSelpopupColumns(tenantId, selPopupEditTableCd);
							String columns = makeSelectFieldStrFromColumnMap(list);
							String value = ActionUtil.getActionParamString(request, e.getKey(), "");
							String where = buildWhere((String)vals.get("foreign_exist_check_query"), value);
							ret.put(getKeyName(vals), dao.selectEditTableDataList(
														(String)vals.get("tenant_id"),
														(String)vals.get("sel_popup_edit_table_cd"),
														columns,
														where,
														null, null, null
													)
										);
						}

					}

				});

			}

		}

		return ret;
	}

	/***
	 * バリデーションモジュールの文字列比較チェックで使うリストの設定作成
	 *
	 * @return
	 */
	private Map<String, Object> makeCompareStrings() {
		Map<String, Object> ret = new HashMap<String, Object>();

		// 入力タイプがパスワード
		// のものがあれば、文字列比較チェックのためにペアを保持する
		if (!CollectionUtils.isEmpty(dispColumns)) {

			for (Map<String, Object> mp : dispColumns) {

				String dataInputtype = (String)mp.get("data_input_type");

				if (DataInputType.PASSWORD.equals(dataInputtype)) {

					// パスワード：
					// [keyの名前]のリクエストの値と、[keyの名前]_conのリクエストの値をペアにする
					String key = getKeyName(mp);
					String val = ActionUtil.getActionParamString(request, key, "");
					String keyCon = getKeyName(mp) + "_con";
					String valCon = ActionUtil.getActionParamString(request, keyCon, "");
					ret.put(key, new HashMap<String, Object>(){{
						put("master_string", val);
						put("re_enter_string", valCon);
					}});

				}

			}


		}

		return ret;
	}

	/***
	 * バリデーションの実行可否設定作成
	 *
	 * @return
	 */
	private Map<String, Object> makeExecSetting() {
		Map<String, Object> ret = new HashMap<String, Object>();

		// 入力タイプがパスワード
		// のものは、チェックボックスの値を見て判定する
		if (!CollectionUtils.isEmpty(dispColumns)) {

			for (Map<String, Object> mp : dispColumns) {

				String dataInputtype = (String)mp.get("data_input_type");

				if (DataInputType.PASSWORD.equals(dataInputtype)) {

					// パスワード：
					// チェックボックスの値を見て判定
					String key = getKeyName(mp);
					String keyChk = getKeyName(mp) + "_chk";
					String valChk = ActionUtil.getActionParamString(request, keyChk, "");

					ret.put(key, (!StringUtils.isEmpty(valChk))? "1" : "0");

				} else {

					// パスワード以外：
					// ここの値は参照しないので、必要なし

				}

			}


		}

		return ret;
	}

	/**
	 * 条件部を作成
	 *
	 * @return 条件部
	 *
	 * */
	private String buildWhere(String query, String... values) {

		// tenant_id + edit_table_cd + キー項目として設定されているカラム名
		// 上記項目名をrequestから取得

		String result = null;
		String work = query;
		if (values != null) {
			for (int i = 0; i < values.length; i++) {
				work = work.replace("$$%%%" + (i+1) + "%%%$$", escapeValStr( values[i] ));
				work = work.replace("%%%" + (i+1) + "%%%", escapeValStr( values[i] ));
			}
		}

		result = work;

		// 条件が無い場合はSQLで使用しないようにNULLを設定
		if ("".equals(result)) {
			result = null;
		}

		return result;

	}

	/**
	 * columnMapを取得します。
	 * @return columnMap
	 */
	@Override
	public List<Map<String, Object>> getColumnMap() {
		return settingData;
	}

	/**
	 * keyColumnsを取得します。
	 * @return keyColumns
	 */
	@Override
	public List<Map<String, Object>> getKeyColumns() {
		return keyColumns;
	}

	/**
	 * tenantIdを取得します。
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	/**
	 * commonを取得します。
	 * @return common
	 */
	public Map<String, Object> getCommon() {
		return common;
	}

	/**
	 * commonを設定します。
	 * @param common
	 */
	public void setCommon(Map<String, Object> common) {
		this.common = common;
	}

	/**
	 * settingDataを取得します。
	 * @return settingData
	 */
	public List<Map<String, Object>> getSettingData() {
		return settingData;
	}

	/**
	 * settingDataを設定します。
	 * @param settingData
	 */
	public void setSettingData(List<Map<String, Object>> settingData) {
		this.settingData = settingData;
	}

}
