package jp.ill.photon.module.db;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ActionUtil;
import jp.ill.photon.util.CheckUtil;
import jp.ill.photon.util.FileUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;



/**
 * カラム設定を読んで、DB更新処理を実施するクラス
 *
 * @author m_fukukawa
 *
 */
public class SettingDataToModifyProcessModule extends SettingDataToManipulateQueryBaseModule implements PhotonModule {

	/* モジュールパラメータ */
	@ModuleParam(required=true)
	private String tenantId;

	@ModuleParam(required=false)
	private Map<String, Object> common;

	@ModuleParam(required=true)
	private List<Map<String, Object>> settingData;

	@ModuleParam(required=false)
	private Map<String, Object> user;

	@ModuleParam(required=false)
	private Map<String, Object> autoGeneratedItems;

	@ModuleParam(required=false)
	private Map<String, Object> after;

	/* クラス内の複数メソッドで使用する変数 */
	private boolean isUpdateProcess = false;
	private boolean isInputFileItems = false;
	private List<Map<String, Object>> dispColumns;
	private List<Map<String, Object>> keyColumns;
	private List<String> keyColumnCd;
	private List<Map<String, Object>> fileColumns;
	private List<Map<String, Object>> notKeyColumns;

	private Map<String, Object> systemSetting;
	private Map<String, Object> aladdinSetting;

	private int sqlResult = -1;
	private JsonDataDao dao;
	private String userName;
	private Map<String, List<String>> errorMessageListMap;
	private String guid;

	@SuppressWarnings("unchecked")
	@Override
	public ModuleResult executeCustom(ModuleContext context)
			throws PhotonModuleException {

		// 設定情報から、各種リストを取得
		setBasicValuesFromRequestParam(context);
		setSettingItems();
		setSystemSettingFromCommon();

		// UPDATE/INSERT処理実施
		doModify();

		ModuleResult result = new ModuleResult();

		result.getReturnData().put("sql-result", sqlResult);
		String resultCd = (sqlResult == -1)? "failed" : "success";
		result.getReturnData().put("result_code", resultCd);
		if (sqlResult == 0) {
			result.setResultCode("success");
		} else {
			result.setResultCode("failed");
			result.getReturnData().put("messages", errorMessageListMap);
		}
		// モジュールのパラメータ"after"に値がセットされていたとき
		if (after != null) {
			Map<String, Object> mp = (Map<String, Object>)after.get(result.getResultCode());
			if (mp != null) {
				result.getReturnData().put("after_params", mp.get("params"));
				result.setNextPath((String)mp.get("next_path"));
				result.setResultType((String)mp.get("result_type"));
			}
		}

		return result;

	}

	/**
	 *
	 * 基本的なパラメータを取得
	 *
	 * */
	private void setBasicValuesFromRequestParam(ModuleContext context) {
		if (StringUtils.isEmpty(tenantId)) {
			tenantId = ActionUtil.getActionParamString(request, "tenant_id", "");
		}
		editTableCd = ActionUtil.getActionParamString(request, "edit_table_cd", "");
		String upFlg = ActionUtil.getActionParamString(request, "up_flg", "");
		isUpdateProcess = ( "1".equals(upFlg) );

		if (user != null) {
			Map<String, Object> params = new LinkedHashMap<String, Object>();
			for (Map.Entry<String, Object> param : user.entrySet()) {
				params.put((String) param.getKey(),
						ParamUtil.getParamValueByType(
								String.valueOf(
										((Map) param.getValue()).get("type")),
								((Map) param.getValue()).get("val"),
								context.getDto()));
			}
			userName = (String)params.get("name");
		}
	}

	/**
	 *
	 * 設定情報から、<br />
	 * 画面に表示されていた項目のリスト、<br />
	 * キーの項目のリスト、<br />
	 * 別途リストも取得する必要のある項目のリスト<br />
	 * を取得
	 *
	 */
	private void setSettingItems() {

		if (!CollectionUtils.isEmpty(settingData)) {

			for (Map<String, Object> mp : settingData) {

				// 画面に表示されていた項目のリスト
				if (dispColumns == null) {
					dispColumns = new ArrayList<Map<String, Object>>();
				}
				dispColumns.add(mp);

				// キーの項目のリスト
				if (!"0".equals((String)mp.getOrDefault("key_no", "0"))) {
					if (keyColumns == null) {
						keyColumns = new ArrayList<Map<String, Object>>();
					}
					keyColumns.add(mp);
					if (keyColumnCd == null) {
						keyColumnCd = new ArrayList<String>();
					}
					keyColumnCd.add(getKeyName(mp));
				} else {
					// キーではない項目のリスト
					// 更新時に使用
					if (notKeyColumns == null) {
						notKeyColumns = new ArrayList<Map<String, Object>>();
					}
					notKeyColumns.add(mp);
				}

				// ファイル項目のリスト
				String dataInputtype = (String)mp.get("data_input_type");
				if(DataInputType.FILE.equals(dataInputtype)) {
					if (fileColumns == null) {
						fileColumns = new ArrayList<Map<String, Object>>();
					}
					fileColumns.add(mp);
					// ファイル項目が入力されているかの判定
					String name = getKeyName(mp);
					Path p = getFileFromRequestParameter(name);
					if (p != null && !StringUtils.isEmpty(p.toString())) {
						isInputFileItems = true;
					}
					// ファイルが削除されるのかの判定
					String delFlg = ActionUtil.getActionParamString(request, name + ".delFlg", "");
					if (DeleteFlg.DELETE_FLG_ON.equals(delFlg)) {
						isInputFileItems = true;
					}
				}

			}

		}

	}

	/**
	 *
	 * 共通設定データ取得
	 *
	 */
	@SuppressWarnings("unchecked")
	private void setSystemSettingFromCommon() {
		if (common != null && !common.isEmpty()) {
			systemSetting =(Map<String, Object>) common.get("systemsetting");
			aladdinSetting = (Map<String, Object>) common.get("aladdinsetting");
		}
	}


	/***
	 *
	 * 対象テーブルマスタ更新処理
	 *
	 */
	private void doModify() {

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		dao = new JsonDataDaoImpl();

		String cryptKey = getValueFromSystemSetting(systemSetting, "projectCryptKey", "note");

		// トランザクション開始
		sqlResult = tm.required(() -> {

			// ◆INSERT、またはUPDATEで使用するentity_data.valの部分を作成する
			String data = makeDataJsonStr( dispColumns, autoGeneratedItems, cryptKey);

			Map<String, Object> tableInfoDataMap = dao.getTableInfo(tenantId, editTableCd);

			// ◆insertFlgが1ならINSERT文、1以外ならUPDATE文を実行
			Map<String, Object> ret = null;
			if (!isUpdateProcess) {
				where = (autoGeneratedItems != null && !autoGeneratedItems.isEmpty())? makeKeyJsonStr(keyColumns, autoGeneratedItems, false) : makeKeyJsonStr(keyColumns, request, true);
				String funcName = (String)tableInfoDataMap.get("insert_proc");
				// INSERT文実行
				// "insert_proc"に名前が指定されていないときは、通常のSP実行用のDaoメソッドを呼ぶ
				// "insert_proc"に名前が指定されていたときは、その名前のSP実行用のDaoメソッドを呼ぶ
				if (StringUtils.isEmpty(funcName)) {
					ret = dao.insertEditTableData(tenantId, editTableCd, userName, data);
				} else {
					ret = dao.insertEditTableDataBySpecifiedFunction(funcName, tenantId, editTableCd, userName, data);
				}
			} else {
				// ◆keyからUPDATE文のWHERE句部分を生成
				where = makeKeyJsonStr(keyColumns, request, true);
				String funcName = (String)tableInfoDataMap.get("update_proc");
				// UPDATE文実行
				// "update_proc"に名前が指定されていないときは、通常のSP実行用のDaoメソッドを呼ぶ
				// "update_proc"に名前が指定されていたときは、その名前のSP実行用のDaoメソッドを呼ぶ
				if (StringUtils.isEmpty(funcName)) {
					ret = dao.updateEditTableData(tenantId, editTableCd, userName, data);
				} else {
					ret = dao.updateEditTableDataBySpecifiedFunction(funcName, tenantId, editTableCd, userName, data);
				}
			}

			// 登録、更新したデータのguidを取得
			Map<String, Object> modifiedData = dao.getEditTableRow(tenantId, editTableCd, "guid", makeKeyJsonStr(keyColumns, request, true));
			if (!MapUtils.isEmpty(modifiedData)) {
				guid = StringUtil.defaultString(modifiedData.get("guid"));
			}

			if ( ret != null) {
				Integer retVal = (Integer)ret.get("ret");
				if (retVal != 0) {
					return -1;
				}
			}

			// ◆ファイル処理
			if (isInputFileItems) {

				// ◆SELECT文のフィールド部分を生成
				String select = makeSelectFieldStrFromColumnMap(dispColumns);

				// 情報取得
				if (CheckUtil.isEmpty(where)) {
					where = null;
				}

				// ◆対象テーブルの既存データをDBから取得
				Map<String, Object> orgData = null;
				if (isUpdateProcess) {
					orgData = dao.getEditTableRow(tenantId, editTableCd, select, where);
				}

				// ファイル削除・更新
				try {
					deleteAndSaveFiles(".delFlg", orgData);
				} catch (Exception e) {
					e.printStackTrace();
					tm.setRollbackOnly();
					return -1;
				}

			}

			return 0;

		});

	}

	/**
	 * ファイル削除＆登録
	 *
	 * @param paramNameSuffix 削除フラグの接尾辞
	 * @param orgData 対象テーブルの元データ
	 *
	 * @throws Exception
	 *
	 */
	private void deleteAndSaveFiles(String paramNameSuffix, Map<String, Object> orgData) throws Exception {

		if (!CollectionUtils.isEmpty(fileColumns)) {

			for (Map<String, Object> col : fileColumns) {

				String name = getKeyName(col);
				String fileNameFromDB = null;

				if (orgData != null) {
					fileNameFromDB = (String)orgData.get(name);

					// ファイルの削除フラグが設定されている場合、ファイルを削除
					String delFlg = ActionUtil.getActionParamString(request, name + paramNameSuffix, "");
					if (DeleteFlg.DELETE_FLG_ON.equals(delFlg)) {
						deleteAttachedFile(col, fileNameFromDB);
					}

				}

				Path p = getFileFromRequestParameter(name);
				if (p != null && !StringUtils.isEmpty(p.toString())) {
					// アップロードされたファイルを保存する
					saveAttachedFile(col, fileNameFromDB);
				}

			}

		}

	}

	/**
	 *
	 * データ1件の指定列に添付されているファイルを削除する
	 *
	 * @param col カラム定義
	 * @param fileNameFromDB ファイル名
	 *
	 * @throws Exception
	 *
	 */
	private void deleteAttachedFile(Map<String, Object> col, String fileNameFromDB) throws Exception {

		// ファイルユーティリティクラスのインスタンス生成
		FileUtil fileUtil = new FileUtil();

		// ルートパスを取得
		String rootPath = getRootPathFromColumnInfo(col);

		// ファイル名updateのクエリ作成
		// dataの文字列作成
		// 削除なので、項目名の値をクリアする固定のSQLを実行する
		Map<String, Object> mp = createKeyColumnMap(keyColumns, request, true);
		mp.put(getKeyName(col), "");
		String data = JsonUtil.mapToJson(mp);
		dao.updateEditTableData(tenantId, editTableCd, userName, data);

		if (!StringUtils.isEmpty(rootPath) && !StringUtils.isEmpty(fileNameFromDB)) {

			String delFileName = fileNameFromDB;
			// 実ファイル削除
			// 拡張項目の時にはattachmentテーブルのレコードも削除する
			if ("1".equals((String)col.get("ex_field_flg")) ) {

				String colCd = getKeyName(col);
				// --------------------------------
				// ファイル保存テーブルからデータ削除
				// --------------------------------
				Map<String, Object> attachmentMap = new HashMap<String, Object>();

				attachmentMap.put("edit_table_cd", editTableCd);
				attachmentMap.put("col_cd", colCd);
				attachmentMap.put("ref_guid", guid);
				String mapStr = JsonUtil.mapToJson(attachmentMap);
				dao.deleteEditTableData(tenantId, "attachment", mapStr);

				// delFileName = String.format("%s_%s_%s", guid, editTableCd, colCd) + "." + getSuffix(delFileName);
				delFileName = FileUtil.getAttachmentFileName(guid, editTableCd, colCd, getSuffix(delFileName));

			}

			fileUtil.delFile(rootPath, delFileName);

		}

	}

	/**
	 *
	 * アップロードされたファイルを添付する
	 *
	 * @param col カラム定義
	 * @param fileNameFromDB ファイル名
	 *
	 * @throws Exception
	 *
	 */
	private void saveAttachedFile(Map<String, Object> col, String fileNameFromDB) throws Exception {

		// ファイルユーティリティクラスのインスタンス生成
		FileUtil fileUtil = new FileUtil();

		// ルートパスを取得
		String rootPath = getRootPathFromColumnInfo(col);

		String fileNameDir = "";
		String fileName = (String)col.get("file_name_select_query");
		String reqKey = getKeyName(col);
		String colCd = getKeyName(col);

		String file = (String)ParamUtil.getParamValueByType(
					ParamUtil.TransferTypes.PARAM_FILE,
					reqKey + ".file_name",
					dto);

		String updateFileName = file;
		if (!CheckUtil.isEmpty(file)) {

			if (!CheckUtil.isEmpty(fileName)) {
				fileNameDir = getFileNameByColumnInfo(fileName, colCd);
				updateFileName = fileNameDir + "." + getSuffix(file);
			}

			Map<String, Object> mp = (!isUpdateProcess && (autoGeneratedItems != null && !autoGeneratedItems.isEmpty()) )? createKeyColumnMap(keyColumns, autoGeneratedItems, false) : createKeyColumnMap(keyColumns, request, true);
			mp.put(colCd, updateFileName);
			String data = JsonUtil.mapToJson(mp);

			dao.updateEditTableData(tenantId, editTableCd, userName, data);

		}

		// ファイルが設定されていれば実施する
		Path srcFilePath = getFileFromRequestParameter(reqKey);
		if (srcFilePath != null) {

			if (fileNameFromDB != null) {
				if (!StringUtils.isEmpty(rootPath) && !StringUtils.isEmpty(fileNameFromDB)) {
					fileUtil.delFile(rootPath, fileNameFromDB);
				}
			}

			String dataFileName = file;
//			if (!CheckUtil.isEmpty(fileName)) {
//				dataFileName = fileNameDir + "." + getSuffix(file);
//			}
			if ("1".equals((String)col.get("ex_field_flg")) ) {
				// dataFileName = String.format("%s_%s_%s", guid, editTableCd, colCd) + "." + getSuffix(file);
				dataFileName = FileUtil.getAttachmentFileName(guid, editTableCd, colCd, getSuffix(file));
			} else {
				if (!CheckUtil.isEmpty(fileName)) {
					dataFileName = fileNameDir + "." + getSuffix(file);
				}
			}

			// ファイルをコピーする
			Path dstFilePath = Paths.get(rootPath + File.separator + dataFileName);
			Files.copy(srcFilePath, dstFilePath, StandardCopyOption.REPLACE_EXISTING);

			if ("1".equals((String)col.get("ex_field_flg")) ) {
				// --------------------------------
				// ファイル保存テーブルにデータ登録
				// --------------------------------
				Map<String, Object> attachmentMap = new HashMap<String, Object>();

				String saveBaseDirSysId = (String) col.get("file_store_path_system_setting_id");
				String refBaseDirSysId = (String) col.get("file_context_path_system_setting_id");
				attachmentMap.put("edit_table_cd", editTableCd);
				attachmentMap.put("col_cd", colCd);
				attachmentMap.put("ref_guid", guid);
				attachmentMap.put("save_base_dir_sys_id",saveBaseDirSysId);
				attachmentMap.put("ref_base_dir_sys_id", refBaseDirSysId);
				attachmentMap.put("file_name", dataFileName);
				attachmentMap.put("org_file_name", updateFileName);
				String mapStr = JsonUtil.mapToJson(attachmentMap);
				dao.deleteEditTableData(tenantId, "attachment", mapStr);
				dao.insertEditTableData(tenantId, "attachment", userName, mapStr);
			}

		}

	}

	/**
	 *
	 * カラム設定情報から、ファイルパスを取得
	 *
	 * @param col カラム定義
	 *
	 * @return ファイルパス
	 *
	 */
	private String getRootPathFromColumnInfo(Map<String, Object> col) {

		String dir = (String)col.get("file_store_path_system_setting_id");
		String select = (String)col.get("sub_dir_name_select_query");
		String fileDir = "";

		if (!CheckUtil.isEmpty(select)) {
			fileDir = getFileNameByColumnInfo(select, getKeyName(col));
		}

		StringJoiner sj = new StringJoiner(File.separator);
		sj.add(getValueFromSystemSetting(systemSetting, dir, "note"));
		if (!StringUtils.isEmpty(fileDir)) {
			sj.add(fileDir);
		}

		return sj.toString();

	}

	/**
	 * 対象データのcolNameに格納されているファイル名を取得する
	 *
	 * @param colName
	 * @return
	 */
	private String getFileNameByColumnInfo(String valName, String dispName) {
		String selectQuery = valName + " AS " + dispName;
		Map<String, Object> mp = dao.getEditTableRow(tenantId, editTableCd, selectQuery, where);
		if (mp != null && !mp.isEmpty()) {
			return (String)mp.get(dispName);
		}
		return null;
	}

	/**
	 *
	 * ファイル名から拡張子を返します。
	 *
	 * @param fileName ファイル名
	 *
	 * @return ファイルの拡張子
	 *
	 */
	private String getSuffix(String fileName) {
		if (fileName == null) {
			return null;
		}
		int point = fileName.lastIndexOf(".");
		if (point != -1) {
			return fileName.substring(point + 1);
		}
		return fileName;
	}

	/**
	 *
	 * リスエスト情報からファイルを取り出す
	 *
	 * @param key リクエスト情報のキー名
	 *
	 * @return ファイルパス
	 *
	 */
	private Path getFileFromRequestParameter(String key) {
		String file = (String)ParamUtil.getParamValueByType(
				ParamUtil.TransferTypes.PARAM_FILE,
				key + ".file.absolute_path",
				dto);
		if (StringUtils.isEmpty(file)) {
			return null;
		}
		Path srcPath = Paths.get(file);
		if (!Files.exists(srcPath)) {
			return null;
		}
		return srcPath;
	}

	/**
	 * columnMapを取得します。
	 * @return columnMap
	 */
	@Override
	public List<Map<String, Object>> getColumnMap() {
		return dispColumns;
	}

	/**
	 * keyColumnsを取得します。
	 * @return keyColumns
	 */
	@Override
	public List<Map<String, Object>> getKeyColumns() {
		return keyColumns;
	}

	/**
	 * commonを取得します。
	 * @return common
	 */
	public Map<String, Object> getCommon() {
		return common;
	}

	/**
	 * commonを設定します。
	 * @param common
	 */
	public void setCommon(Map<String, Object> common) {
		this.common = common;
	}

	/**
	 * settingDataを取得します。
	 * @return settingData
	 */
	public List<Map<String, Object>> getSettingData() {
		return settingData;
	}

	/**
	 * settingDataを設定します。
	 * @param settingData
	 */
	public void setSettingData(List<Map<String, Object>> settingData) {
		this.settingData = settingData;
	}

	/**
	 * userを取得します。
	 * @return user
	 */
	public Map<String, Object> getUser() {
		return user;
	}

	/**
	 * userを設定します。
	 * @param user
	 */
	public void setUser(Map<String, Object> user) {
		this.user = user;
	}

	/**
	 * autoGeneratedItemsを取得します。
	 * @return autoGeneratedItems
	 */
	public Map<String, Object> getAutoGeneratedItems() {
		return autoGeneratedItems;
	}

	/**
	 * autoGeneratedItemsを設定します。
	 * @param autoGeneratedItems
	 */
	public void setAutoGeneratedItems(Map<String, Object> autoGeneratedItems) {
		this.autoGeneratedItems = autoGeneratedItems;
	}

	/**
	 * afterを取得します。
	 * @return after
	 */
	public Map<String, Object> getAfter() {
		return after;
	}

	/**
	 * afterを設定します。
	 * @param after
	 */
	public void setAfter(Map<String, Object> after) {
		this.after = after;
	}

	/**
	 * tenantIdを取得します。
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}


}
