package jp.ill.photon.module.db;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ActionUtil;
import jp.ill.photon.util.CryptUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.StringUtil;

/**
 * カラム設定を読んで、他のモジュールで使用できるデータを出力するためのクエリを作成する基底クラス
 *
 * @author m_fukukawa
 *
 */
public abstract class SettingDataToManipulateQueryBaseModule
		extends SettingDataToManipulateBaseModule implements PhotonModule {

	/* クラス内の複数メソッドで使用する変数 */
	protected String page;
	protected String limit;
	protected String tenantId;
	protected String editTableCd;
	protected String sort;
	protected String columns;
	protected String where;

	protected static class DeleteType {
		public static final String PHYSICALLY = "0";
		public static final String LOGICALLY = "1";
	}

	/**
	 *
	 * キー名を取得
	 *
	 * @param src カラム定義
	 *
	 * @return キー名
	 *
	 */
	protected String getKeyName(Map<String, Object> src) {
		return (String) src.get("col_cd");
	}

	/**
	 *
	 * SELECTフィールド部を作成
	 *
	 * @param data カラム定義のリスト
	 * @param keyList キーのリスト
	 *
	 */
	protected void buildSelectFiedldStr(List<Map<String, Object>> data,
										List<Map<String, Object>> keyList) {

		columns = makeSelectFieldStrFromColumnMap(data, keyList);

	}

	/**
	 *
	 * SELECTフィールド部を作成
	 *
	 * @param data カラム定義のリスト
	 *
	 * @return SELECTフィールドの文字列
	 *
	 */
	protected String makeSelectFieldStrFromColumnMap(List<Map<String, Object>> data) {
		return makeSelectFieldStrFromColumnMap(data, null);
	}

	/**
	 *
	 * SELECTフィールド部を作成
	 *
	 * @param data カラム定義のリスト
	 *
	 * @return SELECTフィールドの文字列
	 *
	 */
	protected String makeSelectFieldStrFromColumnMap(	List<Map<String, Object>> data,
														List<Map<String, Object>> keyList) {

		StringBuffer result = new StringBuffer();
		int cnt = 0;
		List<String> outputList = new ArrayList<String>();
		if (!CollectionUtils.isEmpty(data)) {
			for (Map<String, Object> mp : data) {
				if (mp != null) {
					String colCd = (String) mp.get("col_cd");
					String val = String.format("val->>$$%s$$ AS %s ", colCd,
							colCd);
					if (!StringUtils.isEmpty(val)) {
						if (cnt > 0) {
							result.append(", ");
						}
						result.append(val);
						outputList.add(colCd);
						cnt++;
					}
				}
			}
		}
		if (!CollectionUtils.isEmpty(keyList)) {
			for (Map<String, Object> mp : keyList) {
				String colCd = (String) mp.get("col_cd");
				if (!outputList.contains(colCd)) {
					String val = String.format("val->>$$%s$$ AS %s ", colCd,
							colCd);
					if (cnt > 0) {
						result.append(", ");
					}
					result.append(val);
					cnt++;
				}
			}
		}

		StringJoiner sj = new StringJoiner(" || ");
		if (!CollectionUtils.isEmpty(outputList)) {

			// jsonb_build_object関数の制限で、引数は100個までと決まっているので、
			// 表示するカラム数が50を超えるときには、50個ごとにjsonb_build_object関数を作成し、
			// それぞれを || 演算子で結合するようにする

			int functionArgsCnt = (100 / 2);

			int listSize = outputList.size();
			int set = listSize / functionArgsCnt;
			if ((listSize % functionArgsCnt) > 0) {
				// 剰余があれば、セット数を1追加
				set++;
			}

			for (int i = 0; i < set; i++) {

				StringJoiner sjFields = new StringJoiner(", ");
				StringBuffer sbFunction = new StringBuffer();

				int startIdx = i * functionArgsCnt;
				int endIdx = (i + 1) * functionArgsCnt;
				if (endIdx > listSize) {
					endIdx = listSize;
				}
				List<String> subList = outputList.subList(startIdx, endIdx);

				for (String output : subList) {
					sjFields.add(
							"$$" + output + "$$, val->>$$" + output + "$$");
				}

				if (sjFields.length() > 0) {
					sbFunction.append("jsonb_build_object(");
					sbFunction.append(sjFields.toString());
					sbFunction.append(")");
					sj.add(sbFunction.toString());
				}

			}

			if (sj.length() > 0) {
				result.append(",(");
				result.append(sj.toString());
				result.append(")::TEXT AS row_json_str ");
			}

		}

		return result.toString();

	}

	/**
	 *
	 * 更新対象JSON文字列を作成
	 *
	 * @param data カラム定義のリスト
	 * @param seqDataInfo 自動採番情報
	 * @param cryprKey 暗号化キー
	 *
	 * @return 更新対象JSON文字列
	 *
	 */
	protected String makeDataJsonStr(	List<Map<String, Object>> data,
										Map<String, Object> seqDataInfo,
										String cryprKey) {

		Map<String, Object> ret = new Hashtable<String, Object>();
		if (!CollectionUtils.isEmpty(data)) {

			for (Map<String, Object> mp : data) {

				String dataInputType = (String) mp.get("data_input_type");
				String textHalfZeroFillFlg =  (String) mp.get("text_half_zero_fill_flg");
				String textLengthMaxString = (String) mp.get("text_length_max");


				if (DataInputType.FILE.equals(dataInputType)) {
					// データタイプがファイルの項目は、ここではなく、後続の処理でする
					continue;
				}

				if (DataInputType.PASSWORD.equals(dataInputType)) {
					// データタイプがパスワード、かつ、パスワードを変更しない設定になっている項目は、更新対象にしない
					String colChk = (String) mp.get("col_cd") + "_chk";
					Object valCon = ActionUtil.getActionParamString(request,
							colChk, "");
					if (StringUtils.isEmpty(StringUtil.defaultString(valCon))) {
						continue;
					}
				}

				String col = (String) mp.get("col_cd");
				Object val = ActionUtil.getActionParamString(request,
						getKeyName(mp), "");

				// 自動採番で値を作成した項目であれば、その値を優先
				if (seqDataInfo != null && seqDataInfo.containsKey(col)) {
					val = (String) seqDataInfo.get(col);
				}

				if (!StringUtils.isEmpty(col)) {

					if (DataInputType.CHECKBOX.equals(dataInputType)
							|| DataInputType.LISTBOX.equals(dataInputType)) {

						String[] params = ActionUtil.getActionParams(request,
								getKeyName(mp));

						if (params != null && params.length >= 2) {

							// ex_field_flgが1のデータは、選択された値をchoice_list_separatorで結合した文字列として保存する
							if ("1".equals((String) mp.get("ex_field_flg"))) {

								StringJoiner sj = new StringJoiner(
										(String) mp.getOrDefault(
												"choice_list_separator", ","));
								for (int i = 0; i < params.length; i++) {
									sj.add(params[i]);
								}
								val = sj.toString();

							} else {

								// データタイプがチェックボックス、およびリストボックスの場合は、配列として登録
								// 値部分は配列でわたってきている可能性があるため、取得し直し

								List<String> paramVal = new ArrayList<String>();
								for (int i = 0; i < params.length; i++) {
									paramVal.add(params[i]);
								}
								val = paramVal;

							}

						} else {

							val = ((params == null) ? null : params[0]);

						}

						if (val == null) {
							val = "";
						}

					} else if (DataInputType.PASSWORD.equals(dataInputType)) {

						// データタイプがパスワードの場合は、エンコードして登録
						val = CryptUtil.digestPassword(
								StringUtil.defaultString(val), cryprKey);

					} else if (DataInputType.TEXT.equals(dataInputType) && TextHalfZeroFillFlgType.TEXT_HALF_ZERO_FILL_FLG.equals(textHalfZeroFillFlg) && textLengthMaxString != null) {
						int textLengthMax = Integer.parseInt(textLengthMaxString);
						val = StringUtil.zeroPadding(val.toString(), textLengthMax);
					}

					if (val != null) {
						ret.put(col, val);
					}

				}

			}

		}

		return JsonUtil.mapToJson(ret);

	}

	/***
	 *
	 * キーカラム定義のリストとデータの取得元マップからマップ生成
	 *
	 * @param keyColumns キーカラム定義のリスト
	 * @param valueSourceMap データの取得元
	 * @param getValueFromRequest valueSourceMapをリクエスト情報として扱うか(true:扱う /
	 *            false:扱わない)
	 *
	 * @return マップ<br />
	 *         キー：keyColumnsの要素の"col_cd"の値 値：上記キーでデータの取得元から取得した値
	 *
	 */
	protected Map<String, Object> createKeyColumnMap(	List<Map<String, Object>> keyColumns,
														Map<String, Object> valueSourceMap,
														boolean getValueFromRequest) {
		Map<String, Object> retMap = new LinkedHashMap<String, Object>();
		if (!CollectionUtils.isEmpty(keyColumns)) {
			for (Map<String, Object> mp : keyColumns) {
				String keyColumnName = getKeyName(mp);
				String val = "";
				if (getValueFromRequest) {
					val = ActionUtil.getActionParamString(valueSourceMap,
							keyColumnName, "");
				} else {
					val = String.valueOf(valueSourceMap.get(keyColumnName));
				}
				retMap.put(keyColumnName, val);
			}
		}
		return retMap;
	}

	/***
	 *
	 * キーカラム定義のリストとデータの取得元マップからJSON文字列生成
	 *
	 * @param keyColumns キーカラム定義のリスト
	 * @param valueSourceMap データの取得元
	 * @param getValueFromRequest valueSourceMapをリクエスト情報として扱うか(true:扱う /
	 *            false:扱わない)
	 *
	 * @return JSON文字列
	 *
	 */
	protected String makeKeyJsonStr(List<Map<String, Object>> keyColumns,
									Map<String, Object> valueSourceMap,
									boolean getValueFromRequest) {
		return JsonUtil.mapToJson(createKeyColumnMap(keyColumns, valueSourceMap,
				getValueFromRequest));
	}

	/***
	 *
	 * キーカラム定義のリストとデータの取得元マップからWHERE句で使用する文字列生成
	 *
	 * @param keyColumns キーカラム定義のリスト
	 * @param valueSourceMap データの取得元
	 * @param getValueFromRequest valueSourceMapをリクエスト情報として扱うか(true:扱う /
	 *            false:扱わない)
	 *
	 * @return WHERE文字列
	 *
	 */
	protected String makeWhereStrFromKeyColumn(	List<Map<String, Object>> keyColumns,
												Map<String, Object> valueSourceMap,
												boolean getValueFromRequest) {
		StringJoiner sj = new StringJoiner(" AND ");
		Map<String, Object> keyMap = createKeyColumnMap(keyColumns,
				valueSourceMap, getValueFromRequest);
		if (!keyMap.isEmpty()) {
			for (Map.Entry<String, Object> e : keyMap.entrySet()) {
				sj.add(String.format("val->>$$%s$$ = %s", e.getKey(),
						escapeValStr((String) e.getValue())));
			}
		}
		return sj.toString();
	}

	/***
	 *
	 * keyColumnとvalueSourceMapからWHERE句で使用する文字列生成
	 *
	 * @param valueSourceMap データの取得元
	 * @param getValueFromRequest valueSourceMapをリクエスト情報として扱うか(true:扱う /
	 *            false:扱わない)
	 *
	 * @return WHERE文字列
	 *
	 */
	protected String makeWhereStrFromKeyColumn(	Map<String, Object> valueSourceMap,
												boolean getValueFromRequest) {
		return makeWhereStrFromKeyColumn(getKeyColumns(), valueSourceMap,
				getValueFromRequest);
	}

	/***
	 *
	 * Domaの埋め込み文字列として使用する為、<br />
	 * 引き渡された値をエスケープする
	 *
	 * @param val 入力値
	 *
	 * @return エスケープされた文字列
	 *
	 */
	protected String escapeValStr(String val) {
		if (StringUtils.isEmpty(val)) {
			return "$$$$";
		}
		StringJoiner sj = new StringJoiner(",");
		// Domaの埋め込み文字列内で使えない文字は"chr(XX)"にしてしまう
		List<String> needToConvert = new ArrayList<String>() {
			{
				add("/");
				add("*");
				add("'");
				add("\r");
				add("\n");
				add("\t");
				add("\\");
				add("\"");
				add("-");
				add(";");
			}
		};
		String notNeedToConvert = "";
		for (int i = 0; i < val.length(); i++) {
			if (needToConvert
					.contains(new String(new char[] { val.charAt(i) }))) {
				if (!StringUtils.isEmpty(notNeedToConvert)) {
					sj.add("$$" + notNeedToConvert + "$$");
					notNeedToConvert = "";
				}
				sj.add(String.format("chr(%d)", (int) val.charAt(i)));
			} else {
				notNeedToConvert += val.charAt(i);
			}
		}
		if (!StringUtils.isEmpty(notNeedToConvert)) {
			sj.add("$$" + notNeedToConvert + "$$");
		}
		return "concat(" + sj.toString() + ")";
	}

	/**
	 * columnMapを取得します。
	 *
	 * @return columnMap
	 */
	public abstract List<Map<String, Object>> getColumnMap();

	/**
	 * keyColumnsを取得します。
	 *
	 * @return keyColumns
	 */
	public abstract List<Map<String, Object>> getKeyColumns();

}
