package jp.ill.photon.module.db;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ActionUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.StringUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;



/**
 * カラム設定を読んで、コード採番を実施するクラス
 *
 * @author m_fukukawa
 *
 */
public class SettingDataToGenerateCodeProcessModule extends SettingDataToManipulateBaseModule implements PhotonModule {

	/* モジュールパラメータ */
	@ModuleParam(required=true)
	private String tenantId;

	@ModuleParam(required=true)
	private String editTableCd;

	@ModuleParam(required=true)
	private List<Map<String, Object>> settingData;

	/* クラス内の複数メソッドで使用する変数 */
	private boolean isUpdateProcess = false;

	private Map<String, Object> updateDataMap;

	@Override
	public ModuleResult executeCustom(ModuleContext context)
			throws PhotonModuleException {

		// 設定情報から、各種リストを取得
		setBasicValuesFromRequestParam();

		// 更新パラメータの取得
		setAutogeneratedItemMap();

		ModuleResult result = new ModuleResult();

		result.getReturnData().put("json_str", (isUpdateProcess)? "" : JsonUtil.mapToJson(updateDataMap));
		result.getReturnData().put("map", (isUpdateProcess)? new HashMap<String, Object>() : updateDataMap);
		result.setResultCode("success");

		return result;

	}

	/**
	 * キー名を取得
	 *
	 * @param src
	 * @return
	 *
	 */
	protected String getKeyName(Map<String, Object> src) {
		return (String)src.get("col_cd");
	}

	/**
	 * 基本的なパラメータを取得
	 *
	 * @return 基本的なパラメータ
	 *
	 * */
	private void setBasicValuesFromRequestParam() {
		String upFlg = ActionUtil.getActionParamString(request, "up_flg", "");
		isUpdateProcess = ( "1".equals(upFlg) );
	}


	/***
	 * 採番項目の取得
	 */
	private void setAutogeneratedItemMap() {

		updateDataMap = new LinkedHashMap<String, Object>();
		for (Map<String, Object> mp : settingData) {

			String etcolAutonumUseSequence = (String)mp.get("etcol_autonum_use_sequence");
			String etcolAutonumDiv = (String)mp.get("etcol_autonum_div");
			Integer etcolAutonumLength = StringUtil.toInteger((String)mp.get("etcol_autonum_length"), null);

			// キー項目か、採番された項目だけをセットする
			if (!StringUtils.isEmpty(etcolAutonumUseSequence) || !"0".equals((String)mp.getOrDefault("key_no", "0"))) {

				String colKey = getKeyName(mp);
				String val = ActionUtil.getActionParamString(request, colKey, "");
				String dbColumnName = (String)mp.get("col_cd");

				// etcol_autonum_use_sequenceが設定されていないと、自動採番はされないようにしている
				// 1.x系とちがって、データのテーブルはentity_dataに纏められてしまったため
				if (!StringUtils.isEmpty(etcolAutonumUseSequence)) {

					// テナントID、テーブルコード、カラムコードで自動採番
					String no = getAutogeneratedCd(tenantId, editTableCd, dbColumnName);

					if (EtcolAutnumDiv.SERIAL.equals(etcolAutonumDiv)) {

						val = no;

					} else if (EtcolAutnumDiv.ZERO.equals(etcolAutonumDiv)) {

						if (no.length() < etcolAutonumLength) {
							val = StringUtils.right(StringUtils.repeat("0", etcolAutonumLength) + no, etcolAutonumLength);
						} else {
							// etcol_autonum_lengthの値が採番された値の桁数以下だった場合は、そのままの値を表示
							val = no;
						}

					}

				}

				updateDataMap.put(dbColumnName, val);

			}

		}

	}

	/**
	 * シーケンス名取得
	 *
	 * @param tenantId
	 * @param tableCd
	 * @param columnCd
	 * @return 新しく取得されたシーケンス番号
	 */
	private String getAutogeneratedCd(String tenantId, String tableCd, String columnCd) {

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao daoSeq = new JsonDataDaoImpl();

		List<Map<String, Object>> seqMap = tm.required(() -> {
			return daoSeq.getSeqSfNextVal(tenantId, tableCd, columnCd);
		});

		if (CollectionUtils.isEmpty(seqMap)) {
			return "";
		}

		return (MapUtils.isEmpty(seqMap.get(0)))? "" : String.valueOf(seqMap.get(0).get("seq"));

	}

	/**
	 * tenantIdを取得します。
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	/**
	 * editTableCdを取得します。
	 * @return editTableCd
	 */
	public String getEditTableCd() {
		return editTableCd;
	}

	/**
	 * editTableCdを設定します。
	 * @param editTableCd
	 */
	public void setEditTableCd(String editTableCd) {
		this.editTableCd = editTableCd;
	}

	/**
	 * settingDataを取得します。
	 * @return settingData
	 */
	public List<Map<String, Object>> getSettingData() {
		return settingData;
	}

	/**
	 * settingDataを設定します。
	 * @param settingData
	 */
	public void setSettingData(List<Map<String, Object>> settingData) {
		this.settingData = settingData;
	}

}
