package jp.ill.photon.module.db;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ActionUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

/**
 * カラム設定を読んで、GeneralSearchFormCreateModuleで使用できるデータを出力する基底クラス
 *
 * @author m_fukukawa
 * @see jp.ill.photon.module.db.GeneralSearchFormCreateModule
 *
 */
public abstract class SettingDataToGeneralFormBaseModule
		extends SettingDataToManipulateBaseModule implements PhotonModule {

	/** タイプ */
	protected static String TYPE_LABEL = "label";
	protected static String TYPE_TEXT = "text";
	protected static String TYPE_CHECKBOX = "checkbox";
	protected static String TYPE_RADIO = "radio";
	protected static String TYPE_COMBOBOX = "combobox";
	protected static String TYPE_LISTBOX = "listbox";
	protected static String TYPE_TEXTAREA = "textarea";
	protected static String TYPE_FILE = "file";
	protected static String TYPE_FILE_MULTI = "file_multi";
	protected static String TYPE_DATE = "date";
	protected static String TYPE_DATETIME = "datetime";
	protected static String TYPE_DATETIME_HMS = "datetime_hms";
	protected static String TYPE_RICHTEXT = "richtext";
	protected static String TYPE_FOREIGN = "foreign";
	protected static String TYPE_FOREIGN_MULTI = "foreign_multi";
	protected static String TYPE_COLOR = "color";
	protected static String TYPE_JSON = "json";
	protected static String TYPE_HIDDEN = "hidden";
	protected static String TYPE_PASSWORD = "password";

	protected static String TYPE_RANGE_TEXT = "range_text";
	protected static String TYPE_RANGE_NUMBER = "range_number";
	protected static String TYPE_RANGE_DATE = "range_date";
	protected static String TYPE_RANGE_DATETIME = "range_datetime";

//	/** データ入力方式 */
//	protected static class DataInputType {
//		/** ラベル */
//		protected static final String LABEL = "0";
//		/** 単一行テキスト */
//		protected static final String TEXT = "1";
//		/** チェックボックス */
//		protected static final String CHECKBOX = "2";
//		/** ラジオボタン */
//		protected static final String RADIO = "3";
//		/** コンボボックス */
//		protected static final String COMBOBOX = "4";
//		/** リストボックス */
//		protected static final String LISTBOX = "5";
//		/** テキストエリア */
//		protected static final String TEXTAREA = "6";
//		/** ファイル選択 */
//		protected static final String FILE = "7";
//		/** データコンボボックス */
//		protected static String DATA_COMBOBOX = "8";
//		/** パスワード入力 */
//		protected static String PASSWORD = "9";
//		/** 日付 */
//		protected static final String DATE = "21";
//		/** 日時 */
//		protected static final String DATETIME = "22";
//		/** リッチテキストエディタ */
//		protected static final String RICHTEXT = "23";
//		/** 選択ポップアップ */
//		protected static final String POPUP = "24";
//		/** 単一行テキスト(カラーパレット付) */
//		protected static final String COLOR = "25";
//		/** 範囲（文字列） */
//		protected static String RANGE_TEXT = "31";
//		/** 範囲（数値） */
//		protected static String RANGE_NUMBER = "32";
//		/** 範囲（日付） */
//		protected static String RANGE_DATE = "33";
//		/** 範囲（日時） */
//		protected static String RANGE_DATETIME = "34";
//		/** hidden */
//		protected static String HIDDEN = "99";
//	}

	/** 画面表示モード */
	protected String screenMode;
	/** タイプ */
	protected String type;
	/** 出力オブジェクトタイプ */
	protected String outputType;

	protected static class ColumnName {
		protected static final String COLUMN_NAME = "c_";
	}

	@Override
	public ModuleResult executeCustom(ModuleContext context)
			throws PhotonModuleException {

		makeScreenModeStr();
		initCustom();

		List<Map<String, Object>> forms = new ArrayList<Map<String, Object>>();
		List<Map<String, Object>> headerForms = new ArrayList<Map<String, Object>>();

		List<Map<String, Object>> addInputList = getAddInputList();
		if (addInputList != null) {
			forms.addAll(addInputList);
		}

		List<Map<String, Object>> settingData = getSettingData();
		if (settingData != null) {

			for (Map<String, Object> mp : settingData) {
				Map<String, Object> dst_data = new HashMap<String, Object>();
				makeSettingFromData(mp, dst_data);
				forms.add(dst_data);
			}

		}

		List<Map<String, Object>> headerSettingData = getHeaderSettingData();
		if (headerSettingData != null) {

			for (Map<String, Object> mp : headerSettingData) {
				Map<String, Object> dst_data = new HashMap<String, Object>();
				makeSettingFromData(mp, dst_data);
				headerForms.add(dst_data);
			}

		}

		ModuleResult result = new ModuleResult();
		// フォームデータのセット
		result.getReturnData().put("input_list", forms);
		result.getReturnData().put("header_input_list", headerForms);
		result.getReturnData().put("button_list", makeButtons());

		return result;

	}

	/**
	 * t_bd_edit_table_columnのデータから、汎用入力項目モジュールの項目を作り出す。
	 *
	 * @param src 元になるデータ
	 * @param dst 変更後のデータ
	 *
	 */
	private void makeSettingFromData(	Map<String, Object> src,
										Map<String, Object> dst) {

		dst.put("key", getKeyName(src));
		getTypes(src);
		dst.put("inputs", makeInputsMapFromData(src));
		dst.put("values", makeValuesMapFromData(src));

	}

	/**
	 * タイプ名を取得
	 *
	 * @param src 元になるデータ
	 *
	 */
	protected void getTypes(Map<String, Object> src) {

		type = (String) src.get("data_input_type");

		if (DataInputType.LABEL.equals(type)) {
			outputType = TYPE_LABEL;
		} else if (DataInputType.CHECKBOX.equals(type)) {
			outputType = TYPE_CHECKBOX;
		} else if (DataInputType.RADIO.equals(type)) {
			outputType = TYPE_RADIO;
		} else if (DataInputType.COMBOBOX.equals(type)) {
			outputType = TYPE_COMBOBOX;
		} else if (DataInputType.LISTBOX.equals(type)) {
			outputType = TYPE_LISTBOX;
		} else if (DataInputType.TEXTAREA.equals(type)) {
			outputType = TYPE_TEXTAREA;
		} else if (DataInputType.FILE.equals(type)) {
			outputType = TYPE_FILE;
		} else if (DataInputType.DATA_COMBOBOX.equals(type)) {
			outputType = TYPE_COMBOBOX;
		} else if (DataInputType.PASSWORD.equals(type)) {
			outputType = TYPE_PASSWORD;
		} else if (DataInputType.DATE.equals(type)) {
			outputType = TYPE_DATE;
		} else if (DataInputType.DATETIME.equals(type)) {
			outputType = TYPE_DATETIME;
		} else if (DataInputType.DATETIME_HMS.equals(type)) {
			outputType = TYPE_DATETIME_HMS;
		} else if (DataInputType.RICHTEXT.equals(type)) {
			outputType = TYPE_RICHTEXT;
		} else if (DataInputType.COLOR.equals(type)) {
			outputType = TYPE_COLOR;
		} else if (DataInputType.RANGE_TEXT.equals(type)) {
			outputType = TYPE_RANGE_TEXT;
		} else if (DataInputType.RANGE_NUMBER.equals(type)) {
			outputType = TYPE_RANGE_NUMBER;
		} else if (DataInputType.RANGE_DATE.equals(type)) {
			outputType = TYPE_RANGE_DATE;
		} else if (DataInputType.RANGE_DATETIME.equals(type)) {
			outputType = TYPE_RANGE_DATETIME;
		} else if (DataInputType.POPUP.equals(type)) {
			outputType = TYPE_FOREIGN;
		} else if (DataInputType.HIDDEN.equals(type)) {
			outputType = TYPE_HIDDEN;
		} else {
			outputType = TYPE_TEXT;
		}

	}

	/**
	 * ヘッダタイプ名を取得
	 *
	 * @param src 元になるデータ
	 *
	 */
	protected void getHeaderTypes(Map<String, Object> src) {
		getTypes(src);
	}

	/**
	 * 画面表示モードを表す文字列を作り出す。
	 *
	 */
	protected void makeScreenModeStr() {

		screenMode = makeScreenModeStr(getProcMode(), getRedisplayFlg());

	}

	/**
	 * 画面表示モードを表す文字列を作り出す。
	 *
	 * @param procMode
	 * @param redisplayFlg
	 * @return
	 *
	 */
	protected String makeScreenModeStr(String procMode, String redisplayFlg) {

		String ret = null;

		if (!StringUtils.isEmpty(procMode)) {
			ret = procMode;
		}

		if (!StringUtils.isEmpty(redisplayFlg)) {
			// 処理モードが空白でないときは、「処理モード_再表示フラグ」に
			// 処理モードが空白のときは「再表示フラグ」のみ
			ret = ((!StringUtils.isEmpty(ret)) ? ret + "_" : "") + "redisp";
		}

		if (ret == null) {
			ret = "";
		}

		return ret;

	}

	/**
	 * t_bd_edit_table_columnのデータから、汎用入力項目モジュールの"inputs"の項目を作り出す。
	 *
	 * @param src 元になるデータ
	 *
	 */
	protected Map<String, Object> makeInputsMapFromData(Map<String, Object> src) {

		Map<String, Object> ret = new HashMap<String, Object>();

		Map<String, Object> settingContent = new HashMap<String, Object>();

		Map<String, Object> htmlAttr = new HashMap<String, Object>();

		Map<String, Object> common = getCommon();

		settingContent.put("type", outputType);
		settingContent.put("text_data_format",
				(String) src.getOrDefault("text_data_format", ""));

		settingContent.put("is_editable", getEnable(src));

		settingContent.put("required_flg", (String) src.get("required_flg"));
		settingContent.put("text_half_zero_fill_flg", (String) src.get("text_half_zero_fill_flg"));
		settingContent.put("caption", getCaption(src, common));
		settingContent.put("column_guide", generateColumnGuideStr(src));
		settingContent.put("unit_name", (String) src.get("unit_name"));
		settingContent.put("replace_key_name",
				(String) src.get("replace_key_name"));
		settingContent.put("include_html", (String) src.get("include_html"));
		settingContent.put("group_name", (String) src.get("group_name"));
		settingContent.put("group_separator",
				(String) src.get("group_separator"));

		// 設定からリスト作成
		setChoiceLists(src, settingContent);

		// データコンボボックスの設定
		setDataTypeCombobox(src, settingContent);

		// 範囲系はセパレータを設定
		setRangeInput(src, settingContent);

		if (TYPE_FOREIGN.equals(outputType)
				|| TYPE_FOREIGN_MULTI.equals(outputType)) {
			// 設定から外部参照項目追加
			setForeignSetings(src, settingContent);
		}

		if (TYPE_FILE.equals(outputType)) {
			settingContent.put("file_store_path_system_setting_id",
					(String) src.get("file_store_path_system_setting_id"));
			settingContent.put("file_context_path_system_setting_id",
					(String) src.get("file_context_path_system_setting_id"));
			settingContent.put("file_name_select_query",
					(String) src.get("file_name_select_query"));
			settingContent.put("file_image_use_flg",
					(String) src.get("file_image_use_flg"));
			settingContent.put("file_image_disp_width",
					(String) src.get("file_image_disp_width"));
			settingContent.put("file_extension_list",
					(String) src.get("file_extension_list"));
			// settingContent.put("files_max_count",
			// (String)src.get("files_max_count"));
		}

		if (TYPE_TEXTAREA.equals(outputType)||TYPE_RICHTEXT.equals(outputType)||TYPE_TEXT.equals(outputType)) {
			htmlAttr.put("rows", "8");
			htmlAttr.put("cols", "3");
			htmlAttr.put("class", "edit_table_data_detail_textarea c-form-item-full");
		} else {
			// テキストボックスが存在するときには、size属性を設定する
			// 範囲検索時の項目にもsize属性は適用させる
			setHtmlSizeAttr(src, htmlAttr);
		}

		if (TYPE_PASSWORD.equals(outputType)) {

			// チェックボックス表示フラグ
			// ラベル文字列
			String procMode = StringUtils.defaultString(getProcMode(), "");
			String checked = StringUtils.isEmpty(getRedisplayFlg())?
								(String) src.get("pw_change_checkbox_checked"):
								ActionUtil.getActionParamString(request, getKeyName(src) + "_chk", "");

			settingContent.put("pw_change_checkbox_disp_flg", new HashMap<String, Object>(){{
				put("type", "static");
				// 新規の時には表示しない。更新のときには表示する
				put("val", (procMode.startsWith("update"))? "1" : "0");
			}});
			settingContent.put("pw_change_checkbox_checked", new HashMap<String, Object>(){{
				put("type", "static");
				// 新規の時には設定を参照し、画面再表示時には直前に選択していた値をセット
				put("val", checked);
			}});
			settingContent.put("pw_change_label_str", new HashMap<String, Object>(){{
				put("type", "static");
				put("val", (String) src.get("pw_change_label_str"));
			}});
			settingContent.put("pw_confirm_input_string", new HashMap<String, Object>(){{
				put("type", "static");
				put("val", (String) src.get("pw_confirm_input_string"));
			}});

		}

		String keyNo = StringUtil.defaultString(src.get("key_no"));
		if (!StringUtils.isEmpty(keyNo) && !"0".equals(keyNo)) {
			htmlAttr.put("keyItem", (String) src.get("col_cd"));
		}

		// additional_html_attrに記入されたJSON文字列を
		// Map化して、htmlAttrにセット
		// キーが被った時には上書きする
		// src.get("additional_html_attr")
		String additionalHtmlAttr = (String) src.get("additional_html_attr");
		if (!StringUtils.isEmpty(additionalHtmlAttr)) {
			htmlAttr.putAll(JsonUtil.jsonToMap(additionalHtmlAttr));
		}

		if (!htmlAttr.isEmpty()) {
			settingContent.put("html_attr", htmlAttr);
		}

		ret.put(screenMode, settingContent);
		return ret;

	}

	/**
	 * 項目のリストを作成
	 *
	 * @param src 元になるデータ
	 *
	 */
	protected void setChoiceLists(	Map<String, Object> src,
									Map<String, Object> dst) {

		// ラジオボタン、チェックボックスのときには、デフォルト選択値をセット

		String[] choiseItem = getStringArrayFromChoiceList(src);

		if (choiseItem != null) {

			List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();

			// 分解した値をリストに設定
			for (int i = 0; i < choiseItem.length; i = i + 2) {
				Map<String, Object> mp = new HashMap<String, Object>();
				String val = StringUtils.defaultString(choiseItem[i], "");
				String dispVal = "";
				if (choiseItem.length > (i + 1)) {
					dispVal = choiseItem[i + 1];
				}
				if (!StringUtils.isEmpty(dispVal)) {
					mp.put("list_value", val);
					mp.put("list_caption", dispVal);
					list.add(mp);
				}
			}

			// 入力タイプにより初期値を作成
			setInitItems(src, dst);
			dst.put("list_type", "static");
			dst.put("list", list);

		}

	}

	/**
	 * 外部参照に必要な情報を作成
	 *
	 * @param src 元になるデータ
	 *
	 */
	protected void setForeignSetings(	Map<String, Object> src,
										Map<String, Object> dst) {

		// String type = (String)src.get("search_input_type");
		// 入力タイプにより開くアクションを切り替え

		// 検索条件、および詳細画面で外部呼出しするとき、ボタンキャプションをラベルから設定できるようにする
		dst.put("sel_pop_data_name", getCaptionPopup(src, getCommon()));

		Map<String, Object> mp = new HashMap<String, Object>();
		String targetActionId = (String) src.get("target_action_id");
		if (StringUtils.isEmpty(targetActionId)) {

			mp.put("target_action_id", getTargetActionId());

			Map<String, Object> params = new HashMap<String, Object>();
			params.put("tenant_id", new HashMap<String, Object>() {
				{
					put("val", "tenant_id");
					put("type", ParamUtil.TransferTypes.PARAM);
				}
			});
			params.put("sel_popup_cd", new HashMap<String, Object>() {
				{
					put("val", (String) src.get("sel_popup_cd"));
					put("type", "static");
				}
			});

			mp.put("params", params);

		} else {

			mp.put("target_action_id", targetActionId);

			String targetActionParams = (String) src.get("target_action_params");

			if (!StringUtils.isEmpty(targetActionParams)) {

				Map<String, Object> params = JsonUtil.jsonToMap( targetActionParams ) ;

				for (Map.Entry<String, Object> e : params.entrySet()) {
					params.put(e.getKey(), new HashMap<String, Object>() {
						{
							put("val", e.getValue());
							put("type", "static");
						}
					});
				}

				mp.put("params", params);

			}

		}

		if (TYPE_FOREIGN.equals(outputType)) {
			mp.put("return_items", new HashMap<String, Object>() {
				{
					put("val", (String) src.get("ret_val_col_name"));
					put("disp", (String) src.get("ret_disp_col_name"));
				}
			});
		}

		dst.put("action", mp);

	}

	/**
	 * t_bd_edit_table_columnのデータから、汎用入力項目モジュールの"values"の項目を作り出す。
	 *
	 * @param src 元になるデータ
	 *
	 */
	protected Map<String, Object> makeValuesMapFromData(Map<String, Object> src) {

		Map<String, Object> ret = new HashMap<String, Object>();
		Map<String, Object> settingContent = new HashMap<String, Object>();

		// 値参照先の設定
		setValueReferences(src, settingContent);

		// ドロップダウンリストとラジオボタン、あと、あるかどうかわからないが、リストボックスが検索条件としてセットされる場合、
		// デフォルトの値を選択しておく
		setDetaultValue(src, settingContent);

		ret.put(screenMode, settingContent);
		return ret;

	}

	/**
	 * デフォルト値参照先の設定
	 *
	 * @param src
	 * @param content
	 *
	 */
	protected void setDefaultValueReferences(	Map<String, Object> src,
												Map<String, Object> content) {
		boolean needDefault = false;
		String paramType = ParamUtil.TransferTypes.PARAM;
		if (TYPE_CHECKBOX.equals(outputType) || TYPE_LISTBOX.equals(outputType)
				|| TYPE_FILE_MULTI.equals(outputType)
				|| TYPE_FOREIGN_MULTI.equals(outputType)) {
			paramType = ParamUtil.TransferTypes.PARAM_MULTI;
		}
		if (TYPE_FILE.equals(outputType)) {
			paramType = ParamUtil.TransferTypes.PARAM_FILE;
		}
		if (TYPE_FILE_MULTI.equals(outputType)) {
			paramType = ParamUtil.TransferTypes.PARAM_FILE_MULTI;
		}

		String value = getKeyName(src);
		// "refer_dto"が設定されているときには、指定されているDTOの値を参照する
		String referDto = (String) src.getOrDefault("refer_dto", "");
		if (!StringUtils.isEmpty(referDto)) {
			paramType = ParamUtil.TransferTypes.DTO;
			value = referDto;
		}

		// "input_value"が設定されているときには、指定されている値をそのまま参照する
		String inputValue = (String) src.getOrDefault("input_value", "");
		if (!StringUtils.isEmpty(inputValue)) {
			paramType = "static";
			value = inputValue;
		}

		String defVal = "";
		// "def_value"が設定されているときには、指定されている値をそのまま参照する
		String defValue = (String) src.getOrDefault("def_value", "");
		if (!StringUtils.isEmpty(defValue)) {
			defVal = defValue;
			needDefault = true;
		}

		String etcolAutonumUseSequence = (String) src
				.get("etcol_autonum_use_sequence");
		if (!StringUtils.isEmpty(etcolAutonumUseSequence)) {
			if (StringUtils.isEmpty(getProcMode())) {
				defVal = "自動採番されます。";
				needDefault = true;
			}
		}
		content.put("type", paramType);
		content.put("val", value);
		if (needDefault) {
			content.put("default", defVal);
		}
	}

	/**
	 *
	 * ボタン設定作成
	 *
	 * @return ボタン設定
	 * @throws PhotonModuleException
	 */
	@SuppressWarnings("unchecked")
	private Map<String, Object> makeButtons() throws PhotonModuleException {

		Map<String, Object> commonButtonSetting = getCommonButtonList();
		if (commonButtonSetting == null) {
			commonButtonSetting = new HashMap<String, Object>();
		}
		Map<String, Object> ret = new HashMap<String, Object>(
				commonButtonSetting);

		// add_button_info_listのリストが存在している場合は、その分だけボタンを追加する
		if (isAppendButtonFromSettingData()) {
			List<Map<String, Object>> addButtonInfoList = getAddButtonInfoList();
			if (!CollectionUtils.isEmpty(addButtonInfoList)) {

				List<String> screenModeList = getScreenModeList();

				if (!CollectionUtils.isEmpty(screenModeList)) {

					for (String sc : screenModeList) {

						// List<Map<String, Object>> list = (List<Map<String,
						// Object>>)ret.get(screenMode);
						List<Map<String, Object>> list = (List<Map<String, Object>>) ret
								.get(sc);
						if (list == null) {
							list = new ArrayList<Map<String, Object>>();
						}

						for (Map<String, Object> mp : addButtonInfoList) {

							String caption = (String) mp.get("caption");
							String icon = (String) mp.get("icon");
							String mode = (String) mp.get("mode");

							if (isAppendButtonOnThisMode(mode, sc)) { // TODO:一覧：現在参照できるscの値
																		// /
																		// 詳細：常にリストに入れる

								String actionStr = (String) mp.get("action");
								String htmlAttrStr = (String) mp
										.get("html_attr");
								Map<String, Object> action = StringUtils
										.isEmpty(actionStr) ? null
												: JsonUtil.jsonToMap(actionStr);
								Map<String, Object> htmlAttr = StringUtils
										.isEmpty(htmlAttrStr) ? null
												: JsonUtil
														.jsonToMap(htmlAttrStr);

								list.add(new HashMap<String, Object>() {
									{

										put("caption", caption);
										put("icon", icon);
										if (!MapUtils.isEmpty(action)) {
											put("action", action);
										}
										if (!MapUtils.isEmpty(htmlAttr)) {
											put("html_attr", htmlAttr);
										}

									}
								});

							}

						}

						// ret.put(screenMode, list);
						ret.put(sc, list);

					}

				}

			}

		}

		return ret;

	}

	/**
	 * キー名を取得
	 *
	 * @param src
	 * @return
	 *
	 */
	protected abstract String getKeyName(Map<String, Object> src);

	/**
	 * 有効か無効化を返却
	 *
	 * @param src
	 * @return
	 *
	 */
	protected abstract boolean getEnable(Map<String, Object> src);

	/**
	 * HTML属性の設定
	 *
	 * @param htmlContent
	 *
	 */
	protected abstract void setHtmlSizeAttr(Map<String, Object> src,
											Map<String, Object> htmlContent);

	/**
	 * 初期処理
	 *
	 */
	protected abstract void initCustom();

	/**
	 * 値参照先の設定
	 *
	 * @param src
	 * @param content
	 *
	 */
	protected abstract void setValueReferences(	Map<String, Object> src,
												Map<String, Object> content);

	/**
	 * デフォルト値の設定
	 *
	 * @param src
	 * @param content
	 *
	 */
	protected abstract void setDetaultValue(Map<String, Object> src,
											Map<String, Object> content);

	/**
	 * データコンボボックスの設定
	 *
	 * @param src
	 * @param content
	 *
	 */
	protected abstract void setDataTypeCombobox(Map<String, Object> src,
												Map<String, Object> content);

	/**
	 * 範囲入力項目の設定
	 *
	 * @param content
	 *
	 */
	protected abstract void setRangeInput(	Map<String, Object> src,
											Map<String, Object> content);

	/**
	 * target_action_idを取得
	 *
	 */
	protected abstract String getTargetActionId();

	/**
	 * 入力項目ガイド文字列を作成
	 *
	 * @param src 元になるデータ
	 *
	 */
	protected abstract String generateColumnGuideStr(Map<String, Object> src);

	/**
	 * 初期値を作成
	 *
	 * @param src 元になるデータ
	 * @param dst 設定後のデータ
	 *
	 */
	protected abstract void setInitItems(Map<String, Object> src, Map<String, Object> dst);

	/**
	 * editableを取得します。
	 *
	 * @return editable
	 */
	public abstract boolean isEditable();

	/**
	 *
	 * この画面表示モードでボタン追加を行うか
	 *
	 * @param settingMode
	 * @param screenMode
	 * @return
	 *
	 */
	protected abstract boolean isAppendButtonOnThisMode(String settingMode,
														String screenMode);

	/**
	 *
	 * テーブル定義情報によるボタン追加を行うか
	 *
	 * @return
	 *
	 */
	protected abstract boolean isAppendButtonFromSettingData();

	/**
	 * 画面表示モードリストを取得
	 *
	 * @return
	 *
	 */
	protected abstract List<String> getScreenModeList();

	/**
	 * settingDataを取得します。
	 *
	 * @return settingData
	 */
	public abstract List<Map<String, Object>> getSettingData();

	/**
	 * headerSettingDataを取得します。
	 *
	 * @return headerSettingData
	 */
	public abstract List<Map<String, Object>> getHeaderSettingData();

	/**
	 * addInputListを取得します。
	 *
	 * @return addInputList
	 */
	public abstract List<Map<String, Object>> getAddInputList();

	/**
	 * commonを取得します。
	 *
	 * @return common
	 */
	public abstract Map<String, Object> getCommon();

	/**
	 * redisplayFlgを取得します。
	 *
	 * @return redisplayFlg
	 */
	public abstract String getRedisplayFlg();

	/**
	 * procModeを取得します。
	 *
	 * @return procMode
	 */
	public abstract String getProcMode();

	/**
	 * externalListを取得します。
	 *
	 * @return externalList
	 */
	public abstract Map<String, Map<String, Object>> getExternalList();

	/**
	 * addButtonInfoListを取得します。
	 *
	 * @return addButtonInfoList
	 */
	public abstract List<Map<String, Object>> getAddButtonInfoList();

	/**
	 * commonButtonListを取得します。
	 *
	 * @return commonButtonList
	 */
	public abstract Map<String, Object> getCommonButtonList();

}
