package jp.ill.photon.module.db;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

/**
 * カラム設定を読んで、GeneralSearchFormCreateModuleで使用できるデータを出力する（詳細画面用）
 *
 * @author m_fukukawa
 * @see jp.ill.photon.module.db.GeneralSearchFormCreateModule
 *
 */
public class SettingDataToGeneralDetailFormModule
		extends SettingDataToGeneralFormBaseModule implements PhotonModule {

	@ModuleParam(required = true)
	protected List<Map<String, Object>> settingData;

	@ModuleParam(required = false)
	protected List<Map<String, Object>> addInputList;

	@ModuleParam(required = false)
	protected Map<String, Object> common;

	@ModuleParam(required = false)
	protected String redisplayFlg;

	@ModuleParam(required = false)
	protected String procMode;

	@ModuleParam(required = false)
	protected List<Map<String, Object>> data;

	@ModuleParam(required = false)
	protected Map<String, Map<String, Object>> externalList;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "static", transferVal = "true")
	protected boolean editable;

	@ModuleParam(required = false)
	protected List<Map<String, Object>> keyColumns;

	@ModuleParam(required = false)
	protected List<Map<String, Object>> addButtonInfoList;

	@ModuleParam(required = false)
	protected Map<String, Object> commonButtonList;

	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "static", transferVal = "1")
	protected String guideMessageCountType;

	/** 画面参照先 */
	protected String dtoRef = null;

	/** テキスト桁数カウント区分 */
	private static class TextLengthCountType {
		/** 1：半角単位 */
		private static final String HALF = "1";
		/** 2：全角単位 */
		private static final String FULL = "2";
	}

	/** 重複禁止フラグ */
	private static class UniqueFlg {
		/** 有効 */
		private static final String ON = "1";
		/** 無効 */
		private static final String OFF = "0";
	}

	/** ガイドメッセージカウント区分 */
	private static class GuideMessageCountType {
		/** 1：半角単位 */
		private static final String HALF = "1";
		/** 2：全角単位 */
		private static final String FULL = "2";
	}

	/**
	 * キー名を取得
	 *
	 * @param src
	 * @return
	 *
	 */
	@Override
	protected String getKeyName(Map<String, Object> src) {
		return (String) src.get("col_cd");
	}

	/**
	 * 入力項目ガイド文字列を作成
	 *
	 * @param src 元になるデータ
	 *
	 */
	protected String generateColumnGuideStr(Map<String, Object> src) {

		String guideStr = "";
		String inputHelpDispFlg = (String) src
				.getOrDefault("input_help_disp_flg", "0");
		boolean getDataFromOtherDto = isGotDataFromOtherDto();

		if ("1".equals(inputHelpDispFlg)) {

			String columnGuide = (String) src.getOrDefault("column_guide", "");
			String inputGuide = "";

			if (isAutoNumberItem(src)) {
				inputGuide = (getDataFromOtherDto) ? ""
						: generateAutoNumColumnGuideStr(src);
			} else if (isKeyItem(src)) {
				inputGuide = (getDataFromOtherDto) ? ""
						: generateColumnGuideStrDetail(src);
			} else {
				inputGuide = generateColumnGuideStrDetail(src);
			}

			if (!StringUtils.isEmpty(inputGuide)) {
				inputGuide = "【" + inputGuide + "】";
			}

			StringJoiner sj = new StringJoiner("<br />");
			if (!StringUtils.isEmpty(inputGuide)) {
				sj.add(inputGuide);
			}
			if (!StringUtils.isEmpty(columnGuide)) {
				sj.add(columnGuide);
			}
			guideStr = sj.toString();

		}

		return guideStr;

	}

	/**
	 * 自動採番項目用の入力項目ガイド文字列を作成
	 *
	 * @param src 元になるデータ
	 *
	 */
	private String generateAutoNumColumnGuideStr(Map<String, Object> src) {

		String guideStr = "";

		String etcolAutonumUseSequence = (String) src
				.get("etcol_autonum_use_sequence");
		String etcolAutonumDiv = (String) src.get("etcol_autonum_div");
		Integer etcolAutonumLength = StringUtil
				.toInteger((String) src.get("etcol_autonum_length"), null);

		if (!StringUtils.isEmpty(etcolAutonumUseSequence)) {

			if (EtcolAutnumDiv.ZERO.equals(etcolAutonumDiv)) {

				guideStr += String.format("（%d桁まで0埋めされます）", etcolAutonumLength);

			}

		}

		return guideStr;

	}

	/**
	 * 詳細な入力項目ガイド文字列を作成
	 *
	 * @param src 元になるデータ
	 *
	 */
	private String generateColumnGuideStrDetail(Map<String, Object> src) {

		String guideStr = "";

		Object fileExtensionList = src.get("file_extension_list");
		Object fileSizeMax = src.get("file_size_max");
		String textDataFormat = (String) src.getOrDefault("text_data_format",
				"0");
		String textLengthCountType = (String) src.getOrDefault(
				"text_length_count_type", TextLengthCountType.HALF);
		Integer textLengthLimitType = (src
				.get("text_length_limit_type") == null) ? null
						: Integer.parseInt(
								(String) src.get("text_length_limit_type"));
		String textLengthMax = (String) src.get("text_length_max");
		String textLengthMin = (String) src.get("text_length_min");
		String uniqueFlg = (String) src.getOrDefault("unique_flg",
				UniqueFlg.OFF);

		if (DataInputType.CHECKBOX.equals(type)
				|| DataInputType.LISTBOX.equals(type)) {
			guideStr = "複数選択可";
		} else if (DataInputType.FILE.equals(type)
				&& fileExtensionList != null) {
			guideStr = (String) fileExtensionList;
		} else {
			try {
				switch (Integer.parseInt(textDataFormat)) {
				case 1:
					guideStr = "半角数字";
					break;
				case 2:
					guideStr = "数値";
					break;
				case 3:
					guideStr = "半角英数字";
					break;
				case 4:
					guideStr = "半角";
					break;
				case 5:
					guideStr = "電話番号";
					break;
				case 6:
					guideStr = "メールアドレス";
					break;
				case 7:
					guideStr = "郵便番号";
					break;
				case 8:
					guideStr = "FAX番号";
					break;
				default:
					guideStr = "";
				}
			} catch (Exception e) {
				guideStr = "";
			}
		}

		if (!StringUtils.isEmpty(guideStr)) {
			guideStr += ":";
		}

		if (DataInputType.FILE.equals(type) && fileSizeMax != null) {
			guideStr += makeDigitStr((String) fileSizeMax, "MB以内");
		} else {

			String fullWidthStr = "";

			if (TextLengthCountType.FULL.equals(textLengthCountType)) {
				fullWidthStr = "全角";
				// if (GuideMessageCountType.FULL.equals(guideMessageCountType))
				// {
				// BigDecimal maxVal = new
				// BigDecimal(StringUtils.defaultString(textLengthMax, "0"));
				// BigDecimal minVal = new
				// BigDecimal(StringUtils.defaultString(textLengthMin, "0"));
				// textLengthMax = maxVal.divide(new BigDecimal(2),
				// RoundingMode.CEILING).setScale(0).toString();
				// textLengthMin = minVal.divide(new BigDecimal(2),
				// RoundingMode.CEILING).setScale(0).toString();
				// }
			}

			if (textLengthLimitType != null) {

				if (TextLengthCountType.FULL.equals(textLengthCountType)
						&& GuideMessageCountType.FULL
								.equals(guideMessageCountType)) {
					// 全角かつガイドメッセージカウント区分が 2：全角単位 の場合
					String textLengthMaxFull = "";
					String textLengthMinFull = "";
					if (GuideMessageCountType.FULL
							.equals(guideMessageCountType)) {
						BigDecimal maxVal = new BigDecimal(
								StringUtils.defaultString(textLengthMax, "0"));
						BigDecimal minVal = new BigDecimal(
								StringUtils.defaultString(textLengthMin, "0"));
						textLengthMaxFull = maxVal
								.divide(new BigDecimal(2), RoundingMode.CEILING)
								.setScale(0).toString();
						textLengthMinFull = minVal
								.divide(new BigDecimal(2), RoundingMode.CEILING)
								.setScale(0).toString();
					}
					switch (textLengthLimitType) {
					case 1:
						guideStr += makeDigitStr(textLengthMax, fullWidthStr,
								textLengthMaxFull, "字（半角", textLengthMax,
								"字）以内");
						break;
					case 2:
						guideStr += makeDigitStr(textLengthMin, fullWidthStr,
								textLengthMinFull, "字（半角", textLengthMin,
								"字）以上");
						break;
					case 3:
						guideStr += String.format("%s%s～%s（半角%s～%s）文字",
								fullWidthStr, textLengthMinFull,
								textLengthMaxFull, textLengthMin,
								textLengthMax);
						break;
					case 4:
						guideStr += makeDigitStr(textLengthMax, fullWidthStr,
								textLengthMaxFull, "文字（半角", textLengthMax,
								"字）");
						break;
					}
				} else {
					switch (textLengthLimitType) {
					case 1:
						guideStr += makeDigitStr(textLengthMax, fullWidthStr,
								textLengthMax, "字以内");
						break;
					case 2:
						guideStr += makeDigitStr(textLengthMin, fullWidthStr,
								textLengthMin, "字以上");
						break;
					case 3:
						guideStr += String.format("%s%s～%s文字", fullWidthStr,
								textLengthMin, textLengthMax);
						break;
					case 4:
						guideStr += makeDigitStr(textLengthMax, fullWidthStr,
								textLengthMax, "文字");
						break;
					}
				}
			}

			if (!TextLengthCountType.FULL.equals(textLengthCountType)) {

				if (!StringUtils.isEmpty(guideStr)) {
					if (!guideStr.endsWith(":")) {
						guideStr += ":";
					}
				}

				if (UniqueFlg.ON.equals(uniqueFlg)
						&& DataInputType.FILE.equals(type)) {
					guideStr += "重複禁止";
				}

				if (guideStr.endsWith(":")) {
					guideStr = guideStr.substring(0, guideStr.length() - 1);
				}

			}

		}

		return guideStr;

	}

	/**
	 * 桁数の入った文字列を返却する
	 *
	 * @param digitStr
	 * @param args
	 *
	 */
	private String makeDigitStr(String digitStr, String... args) {

		if (StringUtils.isEmpty(digitStr)) {
			return "";
		}

		int digit = 0;
		try {
			digit = Integer.parseInt(digitStr);
		} catch (Exception e) {
		}

		if (digit == 0) {
			return "";
		}

		StringBuffer sb = new StringBuffer("");
		for (String item : args) {
			sb.append(item);
		}

		return sb.toString();

	}

	/**
	 * 初期処理
	 *
	 */
	@SuppressWarnings("unchecked")
	@Override
	protected void initCustom() {

		if (!CollectionUtils.isEmpty(data)) {
			// パラメータ"data"がnull以外で、かつ、
			// 現在の画面表示モードについてのデータ取得先の記述が存在しているとき
			for (Map<String, Object> mp : data) {
				String scrMode = makeScreenModeStr(
						ParamUtil.getParamStrValueByType(mp.get("proc_mode"),
								dto),
						ParamUtil.getParamStrValueByType(
								mp.get("redisplay_flg"), dto));
				if (screenMode.equals(scrMode)) {
					Map<String, Object> contentMap = (Map<String, Object>) mp
							.get("content");
					dtoRef = (String) contentMap.get("val");
					break;
				}
			}
		}

	}

	/**
	 * 値参照先の設定
	 *
	 * @param src
	 * @param content
	 *
	 */
	@Override
	protected void setValueReferences(	Map<String, Object> src,
										Map<String, Object> content) {

		if (TYPE_FILE.equals(outputType) || (isGotDataFromOtherDto()
				&& !isKeyItem(src) && StringUtils.isEmpty(redisplayFlg))) {
			// 対象のDTOが見つかった場合、かつ、キー項目ではない場合、対象のDTOを参照するようにvalの文字列を生成する
			// ファイル項目のvalueは表示するファイル名のため、常にこちらを通るようにする
			content.put("type", ParamUtil.TransferTypes.DTO);
			content.put("val", dtoRef + "." + (String) src.get("col_cd"));
		} else {
			// 対象のDTOが見つからない場合、または、キー項目の場合は、リクエストパラメータから取得するように設定
			setDefaultValueReferences(src, content);
		}

	}

	/**
	 * デフォルト値の設定
	 *
	 * @param src
	 * @param content
	 *
	 */
	@Override
	protected void setDetaultValue(	Map<String, Object> src,
									Map<String, Object> content) {

		// "init_val"が設定されていた場合、"init_val"をデフォルト値に使用
		String initVal = (String) src.get("init_val");
		if (!StringUtils.isEmpty(initVal)) {

			content.put("default", initVal);

		} else {

			// ラジオボタン、コンボボックスのときには、デフォルト選択値をセット
			if (TYPE_RADIO.equals(outputType)
					|| TYPE_COMBOBOX.equals(outputType)) {

				String[] choiseItem = getStringArrayFromChoiceList(src);

				// 配列の最初の値をデフォルト値に
				if (choiseItem != null && choiseItem.length > 0) {
					content.put("default",
							StringUtils.defaultString(choiseItem[0], ""));

					if (TYPE_COMBOBOX.equals(outputType)) {
						String requiredFlg = (String) src.get("required_flg");
						if (!"1".equals(requiredFlg)) { // 必須でないとき
							content.put("default", "");
						}
					}

				}

			}

		}

	}

	/**
	 * HTML属性の設定
	 *
	 * @param htmlContent
	 *
	 */
	@Override
	protected void setHtmlSizeAttr(	Map<String, Object> src,
									Map<String, Object> htmlContent) {
		if (TYPE_TEXT.equals(outputType)) {
			if (htmlContent.getOrDefault("class", "")
					.equals("c-form-item-full")) {
				htmlContent.remove("class");
			}
			htmlContent.put("size", (String) src.get("text_input_size"));
			String textLengthMax = (String) src.get("text_length_max");
			// "text_length_max"が設定されていない、または0だった場合はmaxlength属性を設定しない
			if (!StringUtils.isEmpty(textLengthMax)
					&& !"0".equals(textLengthMax)) {
				htmlContent.put("maxlength", textLengthMax);
			}
		}
	}

	/**
	 * 初期値を作成
	 *
	 * @param src 元になるデータ
	 * @param dst 設定後のデータ
	 *
	 */
	@Override
	protected void setInitItems(Map<String, Object> src,
								Map<String, Object> dst) {

		String requiredFlg = (String) src.get("required_flg");

		if (!"1".equals(requiredFlg)) { // 必須でないとき

			// 入力タイプにより初期値を作成
			if (TYPE_COMBOBOX.equals(outputType)) {

				// ドロップダウン
				dst.put("list_top_items_type", "static");
				dst.put("list_top_items", new ArrayList<Map<String, Object>>() {
					{
						add(new HashMap<String, Object>() {
							{
								put("list_value", "");
								put("list_caption", "");
							}
						});
					}
				});

			}

		}

	}

	/**
	 * target_action_idを取得
	 *
	 */
	@Override
	protected String getTargetActionId() {
		return "edit_table_data_popup_list";
	}

	/**
	 * 範囲入力項目の設定
	 *
	 * @param content
	 *
	 */
	@Override
	protected void setRangeInput(	Map<String, Object> src,
									Map<String, Object> content) {
	}

	/**
	 * データコンボボックスの設定
	 *
	 * @param src
	 * @param content
	 *
	 */
	@Override
	protected void setDataTypeCombobox(	Map<String, Object> src,
										Map<String, Object> content) {
	}

	/**
	 * 有効か無効化を返却
	 *
	 * @param src
	 * @return
	 *
	 */
	@Override
	protected boolean getEnable(Map<String, Object> src) {

		// 全項目設定
		boolean editable = isEditable();
		if (!editable) {
			return editable;
		}

		if (isGotDataFromOtherDto()) {
			// キー項目に含まれているかどうか
			if (isKeyItem(src)) {
				return false;
			}
		}

		// 自動採番項目だった場合
		if (isAutoNumberItem(src)) {
			return false;
		}

		return true;

	}

	/**
	 *
	 * この画面表示モードでボタン追加を行うか
	 *
	 * @param settingMode
	 * @param screenMode
	 * @return
	 *
	 */
	protected boolean isAppendButtonOnThisMode(String mode, String screenMode) {
		return true;
	}

	/**
	 *
	 * テーブル定義情報によるボタン追加を行うか
	 *
	 * @return
	 *
	 */
	@Override
	protected boolean isAppendButtonFromSettingData() {
		return isGotDataFromOtherDto();
	}

	/**
	 * 画面表示モードリストを取得
	 *
	 * @return
	 *
	 */
	protected List<String> getScreenModeList() {
		return new ArrayList<String>() {
			{
				add(screenMode);
			}
		};
	}

	/**
	 * 値をデータから取得する方式で表示しているか<br />
	 * （DBからデータを取得しているか）
	 *
	 * @return
	 */
	private boolean isGotDataFromOtherDto() {
		return !StringUtils.isEmpty(dtoRef);
	}

	/**
	 * この項目はキー項目に含まれているか
	 *
	 * @param src
	 * @return
	 */
	private boolean isKeyItem(Map<String, Object> src) {
		String colCd = (String) src.get("col_cd");
		if (!CollectionUtils.isEmpty(keyColumns)) {
			for (Map<String, Object> mp : keyColumns) {
				if (colCd.equals(mp.get("col_cd"))) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * この項目は自動採番かを判定
	 *
	 * @param src
	 * @return
	 */
	private boolean isAutoNumberItem(Map<String, Object> src) {
		return !StringUtils
				.isEmpty((String) src.get("etcol_autonum_use_sequence"));
	}

	/**
	 * settingDataを取得します。
	 * 
	 * @return settingData
	 */
	@Override
	public List<Map<String, Object>> getSettingData() {
		return settingData;
	}

	/**
	 * headerSettingDataを取得します。
	 * 
	 * @return headerSettingData
	 */
	@Override
	public List<Map<String, Object>> getHeaderSettingData() {
		return null;
	}

	/**
	 * settingDataを設定します。
	 * 
	 * @param settingData
	 */
	public void setSettingData(List<Map<String, Object>> settingData) {
		this.settingData = settingData;
	}

	/**
	 * addInputListを取得します。
	 * 
	 * @return addInputList
	 */
	@Override
	public List<Map<String, Object>> getAddInputList() {
		return addInputList;
	}

	/**
	 * addInputListを設定します。
	 * 
	 * @param addInputList
	 */
	public void setAddInputList(List<Map<String, Object>> addInputList) {
		this.addInputList = addInputList;
	}

	/**
	 * commonを取得します。
	 * 
	 * @return common
	 */
	@Override
	public Map<String, Object> getCommon() {
		return common;
	}

	/**
	 * commonを設定します。
	 * 
	 * @param common
	 */
	public void setCommon(Map<String, Object> common) {
		this.common = common;
	}

	/**
	 * redisplayFlgを取得します。
	 * 
	 * @return redisplayFlg
	 */
	@Override
	public String getRedisplayFlg() {
		return redisplayFlg;
	}

	/**
	 * redisplayFlgを設定します。
	 * 
	 * @param redisplayFlg
	 */
	public void setRedisplayFlg(String redisplayFlg) {
		this.redisplayFlg = redisplayFlg;
	}

	/**
	 * procModeを取得します。
	 * 
	 * @return procMode
	 */
	@Override
	public String getProcMode() {
		return procMode;
	}

	/**
	 * procModeを設定します。
	 * 
	 * @param procMode
	 */
	public void setProcMode(String procMode) {
		this.procMode = procMode;
	}

	/**
	 * externalListを取得します。
	 * 
	 * @return externalList
	 */
	@Override
	public Map<String, Map<String, Object>> getExternalList() {
		return externalList;
	}

	/**
	 * externalListを設定します。
	 * 
	 * @param externalList
	 */
	public void setExternalList(Map<String, Map<String, Object>> externalList) {
		this.externalList = externalList;
	}

	/**
	 * dataを取得します。
	 * 
	 * @return data
	 */
	public List<Map<String, Object>> getData() {
		return data;
	}

	/**
	 * dataを設定します。
	 * 
	 * @param data
	 */
	public void setData(List<Map<String, Object>> data) {
		this.data = data;
	}

	/**
	 * editableを取得します。
	 * 
	 * @return editable
	 */
	@Override
	public boolean isEditable() {
		return editable;
	}

	/**
	 * editableを設定します。
	 * 
	 * @param editable
	 */
	public void setEditable(boolean editable) {
		this.editable = editable;
	}

	/**
	 * keyColumnsを取得します。
	 * 
	 * @return keyColumns
	 */
	public List<Map<String, Object>> getKeyColumns() {
		return keyColumns;
	}

	/**
	 * keyColumnsを設定します。
	 * 
	 * @param keyColumns
	 */
	public void setKeyColumns(List<Map<String, Object>> keyColumns) {
		this.keyColumns = keyColumns;
	}

	/**
	 * addButtonInfoListを取得します。
	 * 
	 * @return addButtonInfoList
	 */
	@Override
	public List<Map<String, Object>> getAddButtonInfoList() {
		return addButtonInfoList;
	}

	/**
	 * addButtonInfoListを設定します。
	 * 
	 * @param addButtonInfoList
	 */
	public void setAddButtonInfoList(List<Map<String, Object>> addButtonInfoList) {
		this.addButtonInfoList = addButtonInfoList;
	}

	/**
	 * addButtonListを取得します。
	 * 
	 * @return commonButtonList
	 */
	@Override
	public Map<String, Object> getCommonButtonList() {
		return commonButtonList;
	}

	/**
	 * addButtonListを設定します。
	 * 
	 * @param commonButtonList
	 */
	public void setCommonButtonList(Map<String, Object> commonButtonList) {
		this.commonButtonList = commonButtonList;
	}

	/**
	 * guideMessageCountTypeを取得します。
	 * 
	 * @return guideMessageCountType
	 */
	public String getGuideMessageCountType() {
		return guideMessageCountType;
	}

	/**
	 * guideMessageCountTypeを設定します。
	 * 
	 * @param guideMessageCountType
	 */
	public void setGuideMessageCountType(String guideMessageCountType) {
		this.guideMessageCountType = guideMessageCountType;
	}

}
