package jp.ill.photon.module.db;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.CheckUtil;
import jp.ill.photon.util.FileUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;



/**
 * カラム設定を読んで、DB更新処理を実施するクラス
 *
 * @author m_fukukawa
 *
 */
public class SettingDataToDeleteProcessModule extends SettingDataToManipulateQueryBaseModule implements PhotonModule {

	/* モジュールパラメータ */
	@ModuleParam(required=false)
	private Map<String, Object> common;

	@ModuleParam(required=true)
	private List<Map<String, Object>> settingData;

	@ModuleParam(required=true)
	private String[] deleteTargetData;

	@ModuleParam(required=false)
	private Map<String, Object> user;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "0")
	private String deleteType;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "0")
	private String cascade;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "0")
	private String cascadeType;

	@ModuleParam(required=false)
	private List<Map<String, Object>> childTableList;

	@ModuleParam(required=true)
	private String tenantId;

	@ModuleParam(required=true)
	private String editTableCd;

	@ModuleParam(required=false)
	private Map<String, Object> after;

	@ModuleParam(required=false)
	private Map<String, Object> messages;

	/* クラス内の複数メソッドで使用する変数 */
	private List<Map<String, Object>> dispColumns;
	private List<Map<String, Object>> keyColumns;
	private List<Map<String, Object>> fileColumns;

	private Map<String, Object> systemSetting;

	private int sqlResult = -1;
	private JsonDataDao dao;
	private String userName;
//	private List<String> whereStatementList;

	@SuppressWarnings("unchecked")
	@Override
	public ModuleResult executeCustom(ModuleContext context)
			throws PhotonModuleException {

		// 設定情報から、各種リストを取得
		if (user != null) {
			Map<String, Object> params = new LinkedHashMap<String, Object>();
			for (Map.Entry<String, Object> param : user.entrySet()) {
				params.put((String) param.getKey(),
						ParamUtil.getParamValueByType(
								String.valueOf(
										((Map) param.getValue()).get("type")),
								((Map) param.getValue()).get("val"),
								context.getDto()));
			}
			userName = (String)params.get("name");
		}

		dispColumns = new ArrayList<Map<String, Object>>();
		keyColumns = new ArrayList<Map<String, Object>>();
		fileColumns = new ArrayList<Map<String, Object>>();
		setSettingItems(settingData, dispColumns, keyColumns, fileColumns);
		setSystemSettingFromCommon();

		// Delete処理実施
		doDelete();

		ModuleResult result = new ModuleResult();

		result.getReturnData().put("sql-result", sqlResult);
		String resultCd = (sqlResult == -1)? "failed" : "success";
		result.getReturnData().put("result_code", resultCd);
		result.setResultCode(resultCd);

		// モジュールのパラメータ"after"に値がセットされていたとき
		if (after != null) {
			Map<String, Object> mp = (Map<String, Object>)after.get(result.getResultCode());
			if (mp != null) {
				result.getReturnData().put("after_params", mp.get("params"));
				result.setNextPath((String)mp.get("next_path"));
				result.setResultType((String)mp.get("result_type"));
			}
		}

		if (!MapUtils.isEmpty(messages)) {
			String mes = ParamUtil.getParamStrValueByType(messages.get(result.getResultCode()), dto);
			if (!StringUtils.isEmpty(mes)) {
				Map<String, List<String>> errorMessageListMap = new HashMap<String, List<String>>();
				errorMessageListMap.put("", new ArrayList<String>(){{
					add(mes);
				}});
				result.getReturnData().put("messages", errorMessageListMap);
			}
		}

		return result;

	}

	/**
	 *
	 * 設定情報から、<br />
	 * 画面に表示されていた項目のリスト、<br />
	 * キーの項目のリスト、<br />
	 * 別途リストも取得する必要のある項目のリスト<br />
	 * を取得
	 *
	 * @param src
	 * @param dispColList
	 * @param keyColList
	 * @param fileColList
	 */
	private void setSettingItems(List<Map<String, Object>> src,
									List<Map<String, Object>> dispColList,
									List<Map<String, Object>> keyColList,
									List<Map<String, Object>> fileColList) {

		if (!CollectionUtils.isEmpty(src)) {

			for (Map<String, Object> mp : src) {

				// 画面に表示されていた項目のリスト
				dispColList.add(mp);

				// キーの項目のリスト
				if (!"0".equals((String)mp.getOrDefault("key_no", "0"))) {
					keyColList.add(mp);
				}

				// ファイル項目のリスト
				String dataInputtype = (String)mp.get("data_input_type");
				if(DataInputType.FILE.equals(dataInputtype)) {
					fileColList.add(mp);
				}

			}

		}

	}

	/**
	 *
	 * 共通設定データ
	 *
	 */
	@SuppressWarnings("unchecked")
	private void setSystemSettingFromCommon() {
		if (common != null && !common.isEmpty()) {
			systemSetting =(Map<String, Object>) common.get("systemsetting");
		}
	}

	/***
	 *
	 * 対象テーブルマスタ更新処理
	 *
	 */
	private void doDelete() {

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		dao = new JsonDataDaoImpl();

		// トランザクション開始
		sqlResult = tm.required(() -> {

			try {

				if (deleteTargetData != null) {

					Map<String, Object> tableInfoDataMap = dao.getTableInfo(tenantId, editTableCd);

					for (String jsonStr : deleteTargetData) {

						Map<String, Object> json = JsonUtil.jsonToMap(jsonStr);
						where = makeKeyJsonStr(keyColumns, json, false);

						// ◆対象テーブルの既存データをDBから取得
						// 　物理ファイルを先に削除する
						String select = makeSelectFieldStrFromColumnMap(dispColumns);
						Map<String, Object> orgData  = dao.getEditTableRow(tenantId, editTableCd, select, where);
						// ファイル削除・更新
						// Map<String, Object> updateMap = createKeyColumnMap(keyColumns, request, true);
						Map<String, Object> updateMap = createKeyColumnMap(keyColumns, json, false);
						deleteFiles(orgData, fileColumns, editTableCd, where, deleteType, updateMap);

						// ◆対象テーブルの既存データをDBから削除
						// "delete_proc"に名前が指定されていないときは、通常のSP実行用のDaoメソッドを呼ぶ
						// "delete_proc"に名前が指定されていたときは、その名前のSP実行用のDaoメソッドを呼ぶ
						String funcName = (String)tableInfoDataMap.get("delete_proc");
						if (DeleteType.LOGICALLY.equals(deleteType)) {
							if (StringUtils.isEmpty(funcName)) {
								dao.deleteEditTableDataLogically(tenantId, editTableCd, userName, where);
							} else {
								dao.deleteEditTableDataLogicallyBySpecifiedFunction(funcName, tenantId, editTableCd, userName, where);
							}
						} else {
							if (StringUtils.isEmpty(funcName)) {
								dao.deleteEditTableData(tenantId, editTableCd, where);
							} else {
								dao.deleteEditTableDataBySpecifiedFunction(funcName, tenantId, editTableCd, where);
							}
						}

						// ◆cascadeオプションが1のとき、子データを削除する
						if ("1".equals(cascade)) {
							// ◆cascade_typeが1のときは、ファイルなどを無視した削除方法
							// ◆cascade_typeが0のときは、ファイルも削除する削除方法
							// ◆cascade_typeが上記以外の時は削除を行わない
							if ("1".equals(cascadeType)) {
								deleteChildDataSimply(json);
							} else if ("0".equals(cascadeType)) {
								deleteChildData(json);
							}
						}

					}

				}

			} catch (Exception e) {
				e.printStackTrace();
				tm.setRollbackOnly();
				return -1;
			}

			return 0;

		});

	}

	/**
	 *
	 * 子テーブルのデータ削除
	 *
	 * @param parentJson 親データのJSON
	 *
	 * @throws Exception
	 *
	 */
	private void deleteChildDataSimply(Map<String, Object> parentJson) throws Exception {
		if (!CollectionUtils.isEmpty(childTableList)) {

			for (Map<String, Object> mp : childTableList) {

				String childDeleteType = (String)mp.get("delete_type");
				String childEditTableCd = (String)mp.get("child_edit_table_cd");
				String childColCd = (String)mp.get("child_col_cd");

				String childWhere = String.format("val->>$$%s$$ = $$%s$$", childColCd, (String)parentJson.get(childColCd));

				// 削除対象になった親テーブルデータのキーを持つ子テーブルデータを削除
				// 削除処理
				// 参照する削除タイプは子テーブルのものを見ること

				if (DeleteType.LOGICALLY.equals(childDeleteType)) {
					dao.deleteEditTableChildDataLogically(
							tenantId, childEditTableCd, childWhere, userName
					);
				} else {
					dao.deleteEditTableChildData(
							tenantId, childEditTableCd, childWhere
					);
				}

			}
		}
	}

	/**
	 *
	 * 子テーブルのデータ削除
	 *
	 * @param parentJson 親データのJSON
	 *
	 * @throws Exception
	 *
	 */
	@SuppressWarnings("unchecked")
	private void deleteChildData(Map<String, Object> parentJson) throws Exception {

		if (!CollectionUtils.isEmpty(childTableList)) {

			for (Map<String, Object> mp : childTableList) {

				String childDeleteType = (String)mp.get("delete_type");
				String childEditTableCd = (String)mp.get("child_edit_table_cd");
				String childColCd = (String)mp.get("child_col_cd");

				String childWhere = String.format("val->>$$%s$$ = $$%s$$", childColCd, (String)parentJson.get(childColCd));

				List<Map<String, Object>> childSettingData = JsonUtil.jsonToList((String)mp.get("child_table_col_list"));
				List<Map<String, Object>> childDispColumns = new ArrayList<Map<String, Object>>();
				List<Map<String, Object>> childKeyColumns = new ArrayList<Map<String, Object>>();
				List<Map<String, Object>> childFileColumns = new ArrayList<Map<String, Object>>();
				setSettingItems(childSettingData, childDispColumns, childKeyColumns, childFileColumns);

				String select = makeSelectFieldStrFromColumnMap(childDispColumns);

				// 削除対象になった親テーブルデータのキーを持つ子テーブルデータのリストを取得
				List<Map<String, Object>> list = dao.selectEditTableDataList(
						tenantId, childEditTableCd, select, childWhere,
						null, null, null
				);

				if (!CollectionUtils.isEmpty(list)) {
					int max = list.size();
					for (int i = 0; i < max; i++) {
						// データを1件取得
						// ファイル項目が存在していた場合、ここでパスを生成し、削除する。
						// ◆対象テーブルの既存データをDBから取得
						// 　物理ファイルを先に削除する
						Map<String, Object> orgData  = list.get(i);
						// データ更新（ファイル項目の値をnullにする）

						// ファイル削除・更新
						Map<String, Object> updateMap = createKeyColumnMap(childKeyColumns, orgData, false);
						String childWhereJson = JsonUtil.mapToJson(updateMap);
						deleteFiles(orgData, childFileColumns, childEditTableCd, childWhereJson, childDeleteType, updateMap);

						// 削除処理
						// 参照する削除タイプは子テーブルのものを見ること
						if (DeleteType.LOGICALLY.equals(childDeleteType)) {
							dao.deleteEditTableDataLogically(tenantId, childEditTableCd, userName, childWhereJson);
						} else {
							dao.deleteEditTableData(tenantId, childEditTableCd, childWhereJson);
						}
					}
				}

			}

		}

	}

	/**
	 *
	 * ファイル削除
	 *
	 * @param orgData 対象テーブルの元データ
	 * @param fileColumns ファイル項目のカラム定義
	 * @param editTableCd テーブルコード
	 * @param where 条件
	 * @param deleteType 削除タイプ
	 * @param updateMap 更新用マップ
	 * @throws Exception
	 */
	private void deleteFiles(Map<String, Object> orgData,
							List<Map<String, Object>> fileColumns,
							String editTableCd,
							String where,
							String deleteType,
							Map<String, Object> updateMap) throws Exception {

		if (!CollectionUtils.isEmpty(fileColumns)) {

			for (Map<String, Object> col : fileColumns) {
				String fileNameFromDB = (String)orgData.get(getKeyName(col));

				// ファイル名updateのクエリ作成
				// dataの文字列作成
				// 削除なので、項目名の値をクリアする固定のSQLを実行する
				// （物理削除のときにはスキップする）
				if (DeleteType.LOGICALLY.equals(deleteType)) {
					//Map<String, Object> mp = createKeyColumnMap(keyColumns, request, true);
//					Map<String, Object> mp = new HashMap<String, Object>(updateMap){{
//						put(getKeyName(col), null);
//					}};
					//mp.put(getKeyName(col), null);
					//String data = JsonUtil.mapToJson(mp);
					String data = JsonUtil.mapToJson(new HashMap<String, Object>(updateMap){{
						put(getKeyName(col), null);
					}});
					dao.updateEditTableData(tenantId, editTableCd, userName, data);
				}

				deleteAttachedFile(col, fileNameFromDB, editTableCd, where);

			}

		}

	}

	/**
	 *
	 * データ1件の指定列に添付されているファイルを削除する
	 *
	 * @param col カラム定義
	 * @param fileNameFromDB ファイル名
	 * @param editTableCd テーブルコード
	 * @param where 条件
	 * @throws Exception
	 */
	@SuppressWarnings("rawtypes")
	private void deleteAttachedFile(Map<String, Object> col,
									String fileNameFromDB,
									String editTableCd,
									String where) throws Exception {

		// ファイルユーティリティクラスのインスタンス生成
		FileUtil fileUtil = new FileUtil();

		// ルートパスを取得
		//String rootPath = getRootPathFromColumnInfo(col, editTableCd, where);
		String dir = (String)col.get("file_store_path_system_setting_id");
		String select = (String)col.get("sub_dir_name_select_query");
		String fileDir = "";

		if (!CheckUtil.isEmpty(select)) {
			//fileDir = getFileNameByColumnInfo(select, getKeyName(col));
			String dispName = getKeyName(col);
			Map<String, Object> mp = dao.getEditTableRow(tenantId, editTableCd, (select + " AS " + dispName), where);
			fileDir = (MapUtils.isEmpty(mp)) ? null : (String)mp.get(dispName);
		}
		String rootPath = MapUtils.isEmpty(systemSetting)? "" : (String)((Map)systemSetting.get(dir)).get("note") + File.separator + fileDir;

		if (!StringUtils.isEmpty(rootPath) && !StringUtils.isEmpty(fileNameFromDB)) {
			// 実ファイル削除
			fileUtil.delFile(rootPath, fileNameFromDB);
		}

	}

	/**
	 * columnMapを取得します。
	 * @return columnMap
	 */
	@Override
	public List<Map<String, Object>> getColumnMap() {
		return dispColumns;
	}

	/**
	 * keyColumnsを取得します。
	 * @return keyColumns
	 */
	@Override
	public List<Map<String, Object>> getKeyColumns() {
		return keyColumns;
	}

	/**
	 * commonを取得します。
	 * @return common
	 */
	public Map<String, Object> getCommon() {
		return common;
	}	/**
	 * commonを取得します。
	 * @return common
	 */


	/**
	 * commonを設定します。
	 * @param common
	 */
	public void setCommon(Map<String, Object> common) {
		this.common = common;
	}

	/**
	 * settingDataを取得します。
	 * @return settingData
	 */
	public List<Map<String, Object>> getSettingData() {
		return settingData;
	}

	/**
	 * settingDataを設定します。
	 * @param settingData
	 */
	public void setSettingData(List<Map<String, Object>> settingData) {
		this.settingData = settingData;
	}

	/**
	 * userを取得します。
	 * @return user
	 */
	public Map<String, Object> getUser() {
		return user;
	}

	/**
	 * userを設定します。
	 * @param user
	 */
	public void setUser(Map<String, Object> user) {
		this.user = user;
	}

	/**
	 * deleteTypeを取得します。
	 * @return deleteType
	 */
	public String getDeleteType() {
		return deleteType;
	}

	/**
	 * deleteTypeを設定します。
	 * @param deleteType
	 */
	public void setDeleteType(String deleteType) {
		this.deleteType = deleteType;
	}

	/**
	 * deleteTargetDataを取得します。
	 * @return deleteTargetData
	 */
	public String[] getDeleteTargetData() {
		return deleteTargetData;
	}

	/**
	 * deleteTargetDataを設定します。
	 * @param deleteTargetData
	 */
	public void setDeleteTargetData(String[] deleteTargetData) {
		this.deleteTargetData = deleteTargetData;
	}

	/**
	 * cascadeを取得します。
	 * @return cascade
	 */
	public String getCascade() {
		return cascade;
	}

	/**
	 * cascadeを設定します。
	 * @param cascade
	 */
	public void setCascade(String cascade) {
		this.cascade = cascade;
	}

	/**
	 * childTableListを取得します。
	 * @return childTableList
	 */
	public List<Map<String, Object>> getChildTableList() {
		return childTableList;
	}

	/**
	 * childTableListを設定します。
	 * @param childTableList
	 */
	public void setChildTableList(List<Map<String, Object>> childTableList) {
		this.childTableList = childTableList;
	}

	/**
	 * tenantIdを取得します。
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	/**
	 * editTableCdを取得します。
	 * @return editTableCd
	 */
	public String getEditTableCd() {
		return editTableCd;
	}

	/**
	 * editTableCdを設定します。
	 * @param editTableCd
	 */
	public void setEditTableCd(String editTableCd) {
		this.editTableCd = editTableCd;
	}

	/**
	 * afterを取得します。
	 * @return after
	 */
	public Map<String, Object> getAfter() {
		return after;
	}

	/**
	 * afterを設定します。
	 * @param after
	 */
	public void setAfter(Map<String, Object> after) {
		this.after = after;
	}

	/**
	 * messagesを取得します。
	 * @return messages
	 */
	public Map<String, Object> getMessages() {
		return messages;
	}

	/**
	 * messagesを設定します。
	 * @param messages
	 */
	public void setMessages(Map<String, Object> messages) {
		this.messages = messages;
	}

	/**
	 * cascadeTypeを取得します。
	 * @return cascadeType
	 */
	public String getCascadeType() {
		return cascadeType;
	}

	/**
	 * cascadeTypeを設定します。
	 * @param cascadeType
	 */
	public void setCascadeType(String cascadeType) {
		this.cascadeType = cascadeType;
	}

}
