package jp.ill.photon.module.db;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.annotation.ModuleVersion;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ParamUtil;

import org.apache.commons.lang.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;

@ModuleVersion("1.0.0")
public class GeneralTableDataSelectModule implements PhotonModule {

	@ModuleParam(required=true)
	private String tenantId;

	@ModuleParam(required=true)
	private String tableCd;

	@ModuleParam(required=true)
	private Map<String, Object> columns;

	@ModuleParam(required=true)
	private Map<String, Object> searchConditions;

	@ModuleParam(required=true)
	private Map<String, Object> sortConditions;

	@ModuleParam(required=true)
	private Map<String, Object> paramJson;

	@ModuleParam(required=true)
	private Map<String, Object> cacheKeys;

	@ModuleParam(required=false)
	private List<Map<String, Object>> cacheList;

	@ModuleParam(required=true)
	private boolean cacheExist = false;

	@ModuleParam(required=true)
	private int offset = 0;

	@ModuleParam(required=true)
	private int limit = 5;

	public String getTenantId() {
		return tenantId;
	}

	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	public String getTableCd() {
		return tableCd;
	}

	public void setTableCd(String tableCd) {
		this.tableCd = tableCd;
	}

	public Map<String, Object> getTargetColumns() {
		return columns;
	}

	public void setColumns(Map<String, Object> columns) {
		this.columns = columns;
	}

	public Map<String, Object> getSearchConditions() {
		return searchConditions;
	}

	public void setSearchConditions(Map<String, Object> searchConditions) {
		this.searchConditions = searchConditions;
	}

	public Map<String, Object> getSortConditions() {
		return sortConditions;
	}

	public void setSortConditions(Map<String, Object> sortConditions) {
		this.sortConditions = sortConditions;
	}

	public Map<String, Object> getParamJson() {
		return paramJson;
	}

	public void setParamJson(Map<String, Object> paramJson) {
		this.paramJson = paramJson;
	}

	public Map<String, Object> getCacheKeys() {
		return cacheKeys;
	}

	public void setCacheKeys(Map<String, Object> cacheKeys) {
		this.cacheKeys = cacheKeys;
	}

	public List<Map<String, Object>> getCacheList() {
		return cacheList;
	}

	public void setCacheList(List<Map<String, Object>> cacheList) {
		this.cacheList = cacheList;
	}

	public boolean isCacheExist() {
		return cacheExist;
	}

	public void setCacheExist(boolean cacheExist) {
		this.cacheExist = cacheExist;
	}

	public int getOffset() {
		return offset;
	}

	public void setOffset(int offset) {
		this.offset = offset;
	}

	public int getLimit() {
		return limit;
	}

	public void setLimit(int limit) {
		this.limit = limit;
	}

	@Override
	public ModuleResult execute ( ModuleContext context ) throws PhotonModuleException {
		ModuleResult result = new ModuleResult();

		Map<String, Object> params = new LinkedHashMap<String, Object>();
		if (paramJson != null) {
			Map<String, String> paramInfo = null;
			for (Map.Entry<String, Object> param : paramJson.entrySet()) {
				paramInfo = (Map) param.getValue();
				params.put(param.getKey(), ParamUtil.getParamValueByType(
						paramInfo.get("type"), paramInfo.get("val"), context.getDto()));
			}
		}

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();

		if (Boolean.valueOf(cacheExist)) {
			// キャッシュがある場合はキャッシュをセットするだけ
			result.getReturnData().put("list", cacheList);
		} else {
			// キャッシュがない場合は検索を実行
			List<Map<String, Object>> queryResult = tm.required(() -> {
				return dao.selectTableData(tenantId, tableCd,
						buildTargetColumns(columns),
						buildSearchConditions(searchConditions, params),
						buildSortConditions(sortConditions), offset, limit);
			});
			result.getReturnData().put("list", queryResult);
			if (queryResult.size() > 0) {
				result.getReturnData().put("count", queryResult.get(0).get("count"));
			}
			else {
				result.getReturnData().put("count", 0);
			}
		}

		return result;
	}

	private static final String JSON_TABLE_ALIAS = "DT.VAL";
	private static final String PARAM_HOLDER = "{1}";

	private String buildTargetColumns(Map<String, Object> targetColumns) {
		StringBuilder builder = new StringBuilder();
		for (String key : targetColumns.keySet()) {
			String value = (String) targetColumns.get(key);

			String prefix = JSON_TABLE_ALIAS + "->>$$";

			String convertValue = StringUtils.replaceEach(value,
					new String[] { "{{", "}}" }, new String[] { prefix, "$$" });
			builder.append(convertValue.replace("'", "$$"));

			// 列名
			builder.append(" ");
			builder.append(key);

			builder.append(",");
		}

		String columns = builder.toString();

		return columns.substring(0, columns.length() - 1);
	}

	private String buildSearchConditions(	Map<String, Object> searchConditions,
											Map<String, Object> params) {
		StringBuilder builder = new StringBuilder();

		for (String key : searchConditions.keySet()) {
			String value = (String) searchConditions.get(key);

			if (value != null && !StringUtils.isEmpty((String)params.get(key))) {
				String prefix = JSON_TABLE_ALIAS + "->>$$";
				String convertValue = StringUtils.replaceEach(value,
						new String[] { "{{", "}}" },
						new String[] { prefix, "$$" });

				// keyを元に取得した外部パラメータと結合
				String searchData = (String) params.get(key);
				String whereQueryValue = searchData != null
						? convertValue.replace(PARAM_HOLDER, searchData)
						: convertValue;

				builder.append(" AND ");
				builder.append(whereQueryValue.replace("'", "$$"));
			}
		}

		return builder.toString();
	}

	private String buildSortConditions(Map<String, Object> sortConditions) {
		StringBuilder builder = new StringBuilder();
		boolean isFirst = true;
		for (String key : sortConditions.keySet()) {
			String prefix = JSON_TABLE_ALIAS + "->>$$";
			String convertKey = StringUtils.replaceEach(key,
					new String[] { "{{", "}}" }, new String[] { prefix, "$$" });
			if (!isFirst) {
				builder.append(" AND ");
			}
			builder.append(convertKey.replace("'", "$$"));

			String value = (String) sortConditions.get(key);
			// 昇順/降順
			if (value != null) {
				builder.append(" ");
				builder.append(value);
			}

			isFirst = false;
		}

		return builder.toString();
	}

}
