package jp.ill.photon.module.db;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.builder.DomaSQLQueryBuilder;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

/**
 * [general_search_form_create] 検索、更新フォームの初期データを作成するモジュール.
 *
 * <p>
 * フォーム項目を動的に生成するためのモジュール。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>form_items</dt>
 * <dd>フォーム表示用データマップ。</dd>
 * </dl>
 * </p>
 *
 * @author m_fukukawa
 *
 */
public class GeneralSearchFormCreateModule implements PhotonModule {

	/** タイプ */
	public static String TYPE_LABEL = "label";
	public static String TYPE_TEXT = "text";
	public static String TYPE_CHECKBOX = "checkbox";
	public static String TYPE_RADIO = "radio";
	public static String TYPE_COMBOBOX = "combobox";
	public static String TYPE_LISTBOX = "listbox";
	public static String TYPE_TEXTAREA = "textarea";
	public static String TYPE_FILE = "file";
	public static String TYPE_FILE_MULTI = "file_multi";
	public static String TYPE_DATE = "date";
	public static String TYPE_DATETIME = "datetime";
	public static String TYPE_RICHTEXT = "richtext";
	public static String TYPE_FOREIGN = "foreign";
	public static String TYPE_FOREIGN_MULTI = "foreign_multi";
	public static String TYPE_JSON = "json";
	public static String TYPE_HIDDEN = "hidden";
	protected static String TYPE_PASSWORD = "password";

	public static String TYPE_RANGE_TEXT = "range_text";
	public static String TYPE_RANGE_NUMBER = "range_number";
	public static String TYPE_RANGE_DATE = "range_date";
	public static String TYPE_RANGE_DATETIME = "range_datetime";

	/** 処理モード */
	private String procMode;

	/** 入力欄を隠すかどうか */
	private boolean hideInput;

	/**
	 * 再表示指定用モジュールパラメータ.
	 *
	 * <p>
	 * 空以外の値が渡された場合は初期表示ではなく、入力後の表示だと判断し、<br/>
	 * 初期値ではなく入力値を出力する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String redisplayFlg;

	/**
	 * システム設定用モジュールパラメータ.
	 *
	 * <p>
	 * 通常はDTOの「common.systemsetting」を指定する。<br/>
	 * システム設定の商品分類関連設定を読み込み、処理を行う。
	 * </p>
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>dto</dd>
	 * <dt>transfer_val</dt>
	 * <dd>common.systemsetting</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = false)
	private Map<String, Object> common;

	/**
	 * ユーザー入力値用モジュールパラメータ.
	 *
	 */
	@ModuleParam(required = false)
	private List<Map<String, Object>> inputList;

	/**
	 * 見出表示項目用モジュールパラメータ.
	 *
	 * <p>
	 * データ編集機能の子一覧画面で親情報として表示する項目の一覧を指定する際に利用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private List<Map<String, Object>> headerInputList;

	/**
	 * ボタン表示定義用モジュールパラメータ.
	 *
	 * <p>
	 * データ編集用機能で使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private Map<String, List<Map<String, Object>>> buttonList;

	/**
	 * ソート条件定義用モジュールパラメータ.
	 *
	 * <p>
	 * データ編集用機能で使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private List<Map<String, Object>> sortCond;

	/**
	 * 表示件数定義用モジュールパラメータ.
	 *
	 * <p>
	 * データ編集用機能で使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private List<Map<String, Object>> limit;

	/**
	 * 検索フォームタイトル用モジュールパラメータ.
	 *
	 */
	@ModuleParam(required = false)
	@DefaultParamSetting(transferType = "static", transferVal = "条件検索")
	private String condTitleText;

	/**
	 * 検索フォーム開閉設定用モジュールパラメータ.
	 *
	 * <p>
	 * データ編集用機能で使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String disableCondTitleOpenclose;

	/**
	 * commonを取得します。
	 *
	 * @return common
	 */
	public Map<String, Object> getCommon() {
		return common;
	}

	/**
	 * commonを設定します。
	 *
	 * @param common
	 */
	public void setCommon(Map<String, Object> common) {
		this.common = common;
	}

	/**
	 * inputListを取得します。
	 *
	 * @return inputList
	 */
	public List<Map<String, Object>> getInputList() {
		return inputList;
	}

	/**
	 * inputListを設定します。
	 *
	 * @param inputList
	 */
	public void setInputList(List<Map<String, Object>> inputList) {
		this.inputList = inputList;
	}

	/**
	 * headerInputListを取得します。
	 *
	 * @return headerInputList
	 */
	public List<Map<String, Object>> getHeaderInputList() {
		return headerInputList;
	}

	/**
	 * headerInputListを設定します。
	 *
	 * @param headerInputList
	 */
	public void setHeaderInputList(List<Map<String, Object>> headerInputList) {
		this.headerInputList = headerInputList;
	}

	/**
	 * buttonListを取得します。
	 *
	 * @return buttonList
	 */
	public Map<String, List<Map<String, Object>>> getButtonList() {
		return buttonList;
	}

	/**
	 * buttonListを設定します。
	 *
	 * @param buttonList
	 */
	public void setButtonList(Map<String, List<Map<String, Object>>> buttonList) {
		this.buttonList = buttonList;
	}

	/**
	 * sortCondを取得します。
	 *
	 * @return sortCond
	 */
	public List<Map<String, Object>> getSortCond() {
		return sortCond;
	}

	/**
	 * sortCondを設定します。
	 *
	 * @param sortCond
	 */
	public void setSortCond(List<Map<String, Object>> sortCond) {
		this.sortCond = sortCond;
	}

	/**
	 * limitを取得します。
	 *
	 * @return limit
	 */
	public List<Map<String, Object>> getLimit() {
		return limit;
	}

	/**
	 * limitを設定します。
	 *
	 * @param limit
	 */
	public void setLimit(List<Map<String, Object>> limit) {
		this.limit = limit;
	}

	/**
	 * redisplayFlgを取得します。
	 *
	 * @return redisplayFlg
	 */
	public String getRedisplayFlg() {
		return redisplayFlg;
	}

	/**
	 * redisplayFlgを設定します。
	 *
	 * @param redisplayFlg
	 */
	public void setRedisplayFlg(String redisplayFlg) {
		this.redisplayFlg = redisplayFlg;
	}

	/**
	 * procModeを取得します。
	 *
	 * @return procMode
	 */
	public String getProcMode() {
		return procMode;
	}

	/**
	 * procModeを設定します。
	 *
	 * @param procMode
	 */
	public void setProcMode(String procMode) {
		this.procMode = procMode;
	}

	/**
	 * condTitleTextを取得します。
	 *
	 * @return condTitleText
	 */
	public String getCondTitleText() {
		return condTitleText;
	}

	/**
	 * condTitleTextを設定します。
	 *
	 * @param condTitleText
	 */
	public void setCondTitleText(String condTitleText) {
		this.condTitleText = condTitleText;
	}

	/**
	 * disableCondTitleOpencloseを取得します。
	 *
	 * @return disableCondTitleOpenclose
	 */
	public String getDisableCondTitleOpenclose() {
		return disableCondTitleOpenclose;
	}

	/**
	 * disableCondTitleOpencloseを設定します。
	 *
	 * @param disableCondTitleOpenclose
	 */
	public void setDisableCondTitleOpenclose(String disableCondTitleOpenclose) {
		this.disableCondTitleOpenclose = disableCondTitleOpenclose;
	}

	@SuppressWarnings({ "unchecked" })
	@Override
	public ModuleResult execute(ModuleContext context) {

		ActionDto dto = context.getDto();

		Map<String, Object> forms = new LinkedHashMap<String, Object>();
		Map<String, Object> forms_sort = new LinkedHashMap<String, Object>();
		Map<String, Object> forms_limit = new LinkedHashMap<String, Object>();
		Map<String, Object> forms_input = new LinkedHashMap<String, Object>();
		Map<String, Object> forms_header_input = new LinkedHashMap<String, Object>();
		Map<String, List<Map<String, Object>>> forms_button = new HashMap<String, List<Map<String, Object>>>();

		// 再表示かどうかを設定
		if (StringUtils.isEmpty(redisplayFlg)) {
			// 設定されていない場合、input_listのなかに入っていた場合は、再表示とする
			redisplayFlg = getParamValFromInputList("redisplay_flg", dto);
		}
		// 処理モードを設定
		procMode = getParamValFromInputList("proc_mode", dto);

		forms.put("proc_mode", procMode);
		forms.put("redisplay_flg", redisplayFlg);

		// 入力項目セット
		hideInput = true;
		setInputParametersFromStaticJson(inputList, forms_input,
				!CollectionUtils.isEmpty(headerInputList), dto);
		forms.put("input", forms_input);

		forms.put("grouped_input",
				groupedFormsInput((LinkedHashMap<String, Object>) forms_input));

		// ヘッダ入力項目セット
		setInputParametersFromStaticJson(headerInputList, forms_header_input,
				false, dto);
		forms.put("header_input", forms_header_input);

		for (Map.Entry<String, Object> e : forms_input.entrySet()) {
			Map<String, Object> value = (Map<String, Object>) e.getValue();
			if (!MapUtils.isEmpty(value)
					&& !TYPE_HIDDEN.equals(value.get("type"))) {
				hideInput = false;
			}
		}

		forms.put("hide_input", hideInput);

		// ボタン項目セット
		setButtonsFromStaticJson(buttonList, forms_button, dto);
		forms.put("button", forms_button);

		// ソート項目セット
		setParametersFromStaticJson(sortCond, forms_sort, dto);
		forms.put("sort", (Map<String, Object>) forms_sort.get("sort"));

		// 件数項目セット
		setParametersFromStaticJson(limit, forms_limit, dto);
		forms.put("limit", (Map<String, Object>) forms_limit.get("limit"));

		// 開閉処理を付与
		forms.put("disable_cond_title_openclose", disableCondTitleOpenclose);

		// 件数条件タイトルテキスト
		forms.put("cond_title_text", condTitleText);

		ModuleResult result = new ModuleResult();
		// フォームデータのセット
		result.getReturnData().put("form_items", forms);

		return result;
	}

	/**
	 * 空白かどうかを返す
	 *
	 * @param input 入力値
	 * @param type タイプ
	 * @return 空白かどうか
	 *
	 */
	@SuppressWarnings("rawtypes")
	private boolean isEmpty(Object input, String type) {
		if (input == null) {
			return true;
		}
		if ("param_multi".equals(type)) {
			String[] arr = (String[]) input;
			if (arr.length == 0) {
				return true;
			}
			if (arr.length == 1 && "".equals(arr[0])) {
				return true;
			}
			return false;
		}

		Class clazz = input.getClass();
		if (String.class.equals(clazz)) {
			return StringUtils.isEmpty((String) input);
		}

		return false;
	}

	/**
	 * 再表示かどうかをセットする
	 *
	 * @param dto DTO
	 *
	 */
	private String getParamValFromInputList(String findKey, ActionDto dto) {

		String ret = "";

		if (inputList != null) {

			for (Map<String, Object> input : inputList) {

				if (findKey.equals(input.get("key"))) {

					Map<String, String> paramValuesInfo = getParamValuesInfo(
							input.get("values"));

					ret = (String) ParamUtil.getParamValueByType(
							(String) paramValuesInfo.get("type"),
							paramValuesInfo.get("val"), dto);

				}

			}

		}

		return ret;

	}

	/**
	 * static_jsonプロパティからフォーム情報をセットする
	 *
	 * @param src static_json
	 * @param dst フォーム情報をセットする先
	 * @param dto DTO
	 *
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private void setParametersFromStaticJson(	List<Map<String, Object>> src,
												Map<String, Object> dst,
												ActionDto dto) {

		if (src != null) {

			Map<String, Object> paramValuesInfo = null;
			Map<String, Object> paramInputsInfo = null;

			for (Map<String, Object> param : src) {

				paramValuesInfo = (Map) param.get("values");
				paramInputsInfo = (Map) param.get("inputs");

				String type = (String) paramValuesInfo.get("type");
				String val = (String) paramValuesInfo.get("val");

				Object value = ParamUtil.getParamValueByType(type,
						paramValuesInfo.get("val"), dto);

				if (isEmpty(value, type)) {
					if (StringUtils.isEmpty(redisplayFlg)) {
						value = getValueFromStringOrObjectSetting(
								paramValuesInfo.get("default"), "type", "val",
								dto);
					}
				}

				Map<String, Object> valMap = new HashMap<String, Object>();
				valMap.put("caption", getValueFromStringOrObjectSetting(
						paramInputsInfo.get("caption"), "type", "val", dto));
				valMap.put("name", (String) paramValuesInfo.get("val"));
				valMap.put("type", (String) paramInputsInfo.get("type"));
				valMap.put("list",
						getList((String) paramInputsInfo.get("list_type"),
								paramInputsInfo.get("list"),
								(String) paramInputsInfo
										.get("list_top_items_type"),
								paramInputsInfo.get("list_top_items"), dto));
				valMap.put("foreign", getForeignSetting(
						paramInputsInfo.get("action"), val, dto));
				valMap.put("value", value);
				if (paramInputsInfo.containsKey("html_attr")) {
					valMap.put("html_attr",
							getHtmlAttr(paramInputsInfo.get("html_attr")));
				}
				dst.put((String) param.get("key"), valMap);

			}

		}

	}

	/**
	 * 文字列、オブジェクト、両方の値を採りうる項目から値取得
	 *
	 * @param val 文字列、オブジェクト、両方の値を採りうる項目
	 * @param typeKey オブジェクトだった時のtype属性のキー
	 * @param valKey オブジェクトだった時のval属性のキー
	 * @param dto DTO
	 *
	 */
	@SuppressWarnings("rawtypes")
	private Object getValueFromStringOrObjectSetting(	Object val,
														String typeKey,
														String valKey,
														ActionDto dto) {

		Object ret = val;

		if (ret != null) {

			if (!String.class.equals(ret.getClass())) {

				// 文字列でない場合はオブジェクトとみなして値を取得する
				Map mp = (Map) ret;
				ret = ParamUtil.getParamValueByType((String) mp.get(typeKey),
						mp.get(valKey), dto);

			}

		}

		return ret;

	}

	/**
	 * 入力項目のstatic_jsonプロパティからフォーム情報をセットする
	 *
	 * @param src static_json
	 * @param dst フォーム情報をセットする先
	 * @param dto DTO
	 *
	 */
	@SuppressWarnings({ "rawtypes" })
	private void setInputParametersFromStaticJson(	List<Map<String, Object>> src,
													Map<String, Object> dst,
													boolean isHideInput,
													ActionDto dto) {

		if (src != null) {

			Map<String, String> paramValuesInfo = null;
			Map<String, Object> paramInputsInfo = null;

			for (Map<String, Object> param : src) {

				String key = (String) param.get("key");

				paramValuesInfo = getParamValuesInfo(param.get("values"));
				paramInputsInfo = getParamInputsInfo(param.get("inputs"));

				String type = (String) paramValuesInfo.get("type");

				Object val = paramValuesInfo.get("val");

				Object value = null;

				if (!StringUtils.isEmpty(type)) {
					value = ParamUtil.getParamValueByType(type, val, dto);

					// パスワード入力で、かつ初期表示のときには、valueは空にしておく
					if (TYPE_PASSWORD
							.equals((String) paramInputsInfo.get("type"))
							&& StringUtils.isEmpty(redisplayFlg)) {
						value = null;
					}
				}

				if (isEmpty(value, type)) {
					if (TYPE_LABEL
							.equals((String) paramInputsInfo.get("type"))) {
						value = paramValuesInfo.get("default");
					} else {
						if (StringUtils.isEmpty(redisplayFlg)) {
							value = paramValuesInfo.get("default");
						}
					}
				}

				Map<String, Object> valMap = new HashMap<String, Object>();
				String inputType = ((isHideInput) ? TYPE_HIDDEN
						: (String) paramInputsInfo.get("type"));

				valMap.put("type", inputType);
				valMap.put("required_flg", StringUtils.defaultString(
						(String) paramInputsInfo.get("required_flg"), "0"));
				valMap.put("unit_name",
						(String) paramInputsInfo.get("unit_name"));
				valMap.put("include_html",
						(String) paramInputsInfo.get("include_html"));
				valMap.put("group_name",
						(String) paramInputsInfo.get("group_name"));
				valMap.put("group_separator",
						(String) paramInputsInfo.get("group_separator"));
				valMap.put("text_data_format",
						paramInputsInfo.getOrDefault("text_data_format", ""));
				valMap.put("text_half_zero_fill_flg",
						paramInputsInfo.getOrDefault("text_half_zero_fill_flg", ""));


				// caption、column_guide、sel_pop_data_nameは、文字列、オブジェクト、両方の値を採りうるため、個別に処理する
				valMap.put("caption", getValueFromStringOrObjectSetting(
						paramInputsInfo.get("caption"), "type", "val", dto));
				valMap.put("column_guide",
						getValueFromStringOrObjectSetting(
								paramInputsInfo.get("column_guide"), "type",
								"val", dto));
				valMap.put("sel_pop_data_name",
						getValueFromStringOrObjectSetting(
								paramInputsInfo.get("sel_pop_data_name"),
								"type", "val", dto));

				// "is_editable"は、未設定のときはtrueとする
				Boolean isEditable = (Boolean) ObjectUtils.defaultIfNull(
						paramInputsInfo.get("is_editable"), true);
				valMap.put("is_editable", isEditable);

				if (TYPE_CHECKBOX.equals(inputType)
						|| TYPE_RADIO.equals(inputType)
						|| TYPE_COMBOBOX.equals(inputType)
						|| TYPE_LISTBOX.equals(inputType)) {

					valMap.put("list", getList(
							(String) paramInputsInfo.get("list_type"),
							paramInputsInfo.get("list"),
							(String) paramInputsInfo.get("list_top_items_type"),
							paramInputsInfo.get("list_top_items"), dto));

				}

				if (TYPE_FOREIGN.equals(inputType)
						|| TYPE_FOREIGN_MULTI.equals(inputType)) {

					valMap.put("foreign", getForeignSetting(
							paramInputsInfo.get("action"), (String) val, dto));

				}

				if (TYPE_PASSWORD.equals(inputType)) {

					valMap.put("pw_change_checkbox_disp_flg",
							ParamUtil.getParamStrValueByType(
									paramInputsInfo
											.get("pw_change_checkbox_disp_flg"),
									dto));
					valMap.put("pw_change_checkbox_checked",
							ParamUtil.getParamStrValueByType(
									paramInputsInfo
											.get("pw_change_checkbox_checked"),
									dto));
					valMap.put("pw_change_label_str",
							ParamUtil.getParamStrValueByType(
									paramInputsInfo.get("pw_change_label_str"),
									dto));
					valMap.put("pw_confirm_input_string",
							ParamUtil.getParamStrValueByType(paramInputsInfo
									.get("pw_confirm_input_string"), dto));

					String valueChk = (String) ParamUtil
							.getParamValueByType(type, val + "_chk", dto);
					String valueCon = (String) ParamUtil
							.getParamValueByType(type, val + "_con", dto);

					valMap.put("value_chk", valueChk);
					valMap.put("value_con", valueCon);

				}

				if (TYPE_FILE.equals(inputType)) {

					String contextPath = (paramInputsInfo
							.get("file_context_path_system_setting_id") == null)
									? ""
									: getCommonByKey(paramInputsInfo.get(
											"file_context_path_system_setting_id"));
					String fileName = (value == null) ? "" : (String) value;
					String filePath = "";
					if (!StringUtils.isEmpty(contextPath)
							&& !StringUtils.isEmpty(fileName)) {
						filePath = contextPath + "/" + fileName;
					}

					valMap.put("file_store_path_system_setting_id",
							(String) paramInputsInfo
									.get("file_store_path_system_setting_id"));
					valMap.put("file_context_path_system_setting_id",
							(String) paramInputsInfo.get(
									"file_context_path_system_setting_id"));
					valMap.put("file_name_select_query",
							(String) paramInputsInfo
									.get("file_name_select_query"));
					valMap.put("file_image_use_flg",
							(String) paramInputsInfo.get("file_image_use_flg"));
					valMap.put("file_image_disp_width", (String) paramInputsInfo
							.get("file_image_disp_width"));
					valMap.put("file_path", filePath);

				}

				if (TYPE_FILE_MULTI.equals(inputType)) {

					Map filesMaxCount = (Map) paramInputsInfo
							.get("files_max_count");

					Object filesMaxCountVal = (String) ParamUtil
							.getParamValueByType(
									(String) filesMaxCount.get("type"),
									filesMaxCount.get("val"), dto);
					String filesMaxCountStr = ((filesMaxCountVal == null) ? "0"
							: (String) filesMaxCountVal);
					valMap.put("files_max_count",
							Integer.parseInt(filesMaxCountStr));

					String filesListVal = (String) paramInputsInfo
							.get("files_list");
					List list = (List) ParamUtil.getParamValueByType("dto",
							filesListVal, dto);
					valMap.put("files_list", list);

				}

				if (TYPE_LABEL.equals(inputType)
						|| TYPE_TEXT.equals(inputType)) {

					valMap.put("autonum_div",
							(String) paramInputsInfo.get("autonum_div"));

				}

				if (TYPE_JSON.equals(inputType)) {

					valMap.put("input_id",
							(String) paramInputsInfo.get("input_id"));

				}

				if (
						(
							TYPE_RANGE_TEXT.equals(inputType)
							|| TYPE_RANGE_NUMBER.equals(inputType)
							|| TYPE_RANGE_DATE.equals(inputType)
							|| TYPE_RANGE_DATETIME.equals(inputType)
						)
					||  (
							// 単一テキスト（数値）のときには、range_number
							TYPE_TEXT.equals(inputType)
							&& SettingDataToManipulateBaseModule.TextDataFormat.NUMERIC
							.equals(paramInputsInfo.get("text_data_format")))
						) {

							setRangeValue(valMap, type, val, dto);
							String rangeSeparator = ParamUtil.getParamStrValueByType(
									paramInputsInfo.get("range_separator"), dto);
							String unit = ParamUtil.getParamStrValueByType(
									paramInputsInfo.get("unit"), dto);
							valMap.put("range_separator", rangeSeparator);
							valMap.put("unit_from", unit);
							valMap.put("unit_to", unit);
							if (TYPE_RANGE_DATETIME.equals(inputType)) {
								setRangeDatetimeValue(valMap, type, val, dto);
							}

				}

				// 商品データの拡張項目の日付と日時の場合
				// 登録するのは範囲の項目としてではないが、検索時は範囲の項目として扱う必要がある
				if (key.indexOf("_ex_field_item") == 0
						&& TYPE_DATE.equals(inputType)) {
					setRangeValue(valMap, type, val, dto);
				}
				if (key.indexOf("_ex_field_item") == 0
						&& TYPE_DATETIME.equals(inputType)) {
					setRangeDatetimeValue(valMap, type, val, dto);
				}

				String replaceKeyName = (String) paramInputsInfo
						.getOrDefault("replace_key_name", "");
				valMap.put("name", (StringUtils.isEmpty(replaceKeyName) ? key
						: replaceKeyName));
				valMap.put("value", value);

				if (paramInputsInfo.containsKey("html_attr")) {
					valMap.put("html_attr",
							getHtmlAttr(paramInputsInfo.get("html_attr")));
				}

				dst.put(key, valMap);

			}

		}

	}

	/***
	 *
	 * 範囲の値取得
	 *
	 * @param dest
	 * @param type
	 * @param val
	 * @param dto
	 */
	private void setRangeValue(	Map<String, Object> dest,
								String type,
								Object val,
								ActionDto dto) {
		String valueFrom = (String) ParamUtil.getParamValueByType(type,
				val + "_from", dto);
		String valueTo = (String) ParamUtil.getParamValueByType(type,
				val + "_to", dto);
		dest.put("value_from", valueFrom);
		dest.put("value_to", valueTo);
	}

	/***
	 *
	 * 範囲の値取得
	 *
	 * @param dest
	 * @param type
	 * @param val
	 * @param dto
	 */
	private void setRangeDatetimeValue(	Map<String, Object> dest,
										String type,
										Object val,
										ActionDto dto) {
		String valueDataFrom = (String) ParamUtil.getParamValueByType(type,
				val + "_date_from", dto);
		String valueDataTo = (String) ParamUtil.getParamValueByType(type,
				val + "_date_to", dto);
		String valueTimeFrom = (String) ParamUtil.getParamValueByType(type,
				val + "_time_from", dto);
		String valueTimeTo = (String) ParamUtil.getParamValueByType(type,
				val + "_time_to", dto);
		dest.put("value_date_from", valueDataFrom);
		dest.put("value_date_to", valueDataTo);
		dest.put("value_time_from", valueTimeFrom);
		dest.put("value_time_to", valueTimeTo);
	}

	/**
	 * システム設定値取得
	 *
	 * @param key
	 * @return システム設定値
	 *
	 */
	@SuppressWarnings("unchecked")
	private String getCommonByKey(Object key) {
		Map<String, Object> mp = (Map<String, Object>) common.get((String) key);
		return (mp == null) ? null : (String) mp.get("note");
	}

	/**
	 * static_jsonの"inputs"から設定情報取得
	 *
	 * @param src
	 * @return "html_attr"属性設定
	 *
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private Map<String, Object> getParamInputsInfo(Object map) {

		String workMode = "";

		if (!StringUtils.isEmpty(procMode)) {
			workMode = procMode;
		}

		if (!StringUtils.isEmpty(redisplayFlg)) {
			// 処理モードが空白でないときは、「処理モード_再表示フラグ」に
			// 処理モードが空白のときは「再表示フラグ」のみ
			// 「処理モード_再表示フラグ」がJSONの設定の中にない場合、「再表示フラグ」のみにする

			workMode = ((!StringUtils.isEmpty(workMode)) ? workMode + "_" : "")
					+ "redisp";

			if (!((Map) map).containsKey(workMode)) {
				workMode = "redisp";
			}

		}

		if (!((Map) map).containsKey(workMode)) {
			workMode = "";
		}

		Map<String, Object> ret = (Map<String, Object>) ((Map) map)
				.get(workMode);

		return (ret == null) ? new HashMap<String, Object>() : ret;

	}

	/**
	 * 処理モード、再表示フラグに応じた画面項目情報取得
	 *
	 * @param src
	 * @return "html_attr"属性設定
	 *
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private Map<String, String> getParamValuesInfo(Object map) {

		String workMode = "";

		if (!StringUtils.isEmpty(procMode)) {
			workMode = procMode;
		}

		if (!StringUtils.isEmpty(redisplayFlg)) {
			// 処理モードが空白でないときは、「処理モード_再表示フラグ」に
			// 処理モードが空白のときは「再表示フラグ」のみ
			// 「処理モード_再表示フラグ」がJSONの設定の中にない場合、「再表示フラグ」のみにする

			workMode = ((!StringUtils.isEmpty(workMode)) ? workMode + "_" : "")
					+ "redisp";

			if (!((Map) map).containsKey(workMode)) {
				workMode = "redisp";
			}

		}

		if (!((Map) map).containsKey(workMode)) {
			// JSONの設定の中に、指定された処理モード、及び再表示フラグが存在しない場合、
			// デフォルトの設定を採用する
			workMode = "";
		}

		Map<String, String> ret = (Map<String, String>) ((Map) map)
				.get(workMode);

		return (ret == null) ? new HashMap<String, String>() : ret;

	}

	/**
	 * ボタン項目のstatic_jsonプロパティからフォーム情報をセットする
	 *
	 * @param src static_json
	 * @param dst フォーム情報をセットする先
	 * @param dto DTO
	 *
	 */
	private void setButtonsFromStaticJson(	Map<String, List<Map<String, Object>>> src,
											Map<String, List<Map<String, Object>>> dst,
											ActionDto dto) {

		if (src != null) {
			for (Map.Entry<String, List<Map<String, Object>>> e : src
					.entrySet()) {
				List<Map<String, Object>> tmp = new LinkedList<Map<String, Object>>();
				for (Map<String, Object> item : e.getValue()) {
					// "action"が存在したとき、urlを組み立てる
					if (item.containsKey("action")) {
						// target_idとparamsを組み合わせてURL作成
						setConbineActionIdAndParams(item, dto);
					}
					tmp.add(item);
				}
				dst.put(e.getKey(), tmp);
			}
		}
		;
	}

	/**
	 * "html_attr"属性設定取得
	 *
	 * @param src
	 * @return "html_attr"属性設定
	 *
	 */
	@SuppressWarnings("unchecked")
	private Map<String, String> getHtmlAttr(Object src) {

		Map<String, String> ret = new HashMap<String, String>();
		if (src != null) {
			ret = (Map<String, String>) src;
		}
		return ret;

	}

	/**
	 * 指定されたlistTypeから、リストを作成
	 *
	 * @param listType
	 * @param list
	 * @param topItemType
	 * @param topItem
	 * @param dto DTO
	 *
	 */
	@SuppressWarnings({ "unchecked" })
	private List<Map<String, String>> getList(	String listType,
												Object list,
												String topItemType,
												Object topItem,
												ActionDto dto) {

		List<Map<String, String>> ret = new ArrayList<Map<String, String>>();

		// リストの頭の項目追加
		if (!StringUtils.isEmpty(topItemType) && topItem != null) {
			List<Map<String, Object>> topItemList = null;
			if ("static".equals(topItemType)) {
				topItemList = (List<Map<String, Object>>) topItem;
			} else if ("dto".equals(topItemType)) {
				topItemList = (List<Map<String, Object>>) ParamUtil
						.getParamValueByType("dto", (String) topItem, dto);
			}
			if (topItemList != null) {
				for (Map<String, Object> mp : topItemList) {
					ret.add(new HashMap<String, String>() {
						{
							put("list_value", (String) mp.get("list_value"));
							put("list_caption",
									(String) mp.get("list_caption"));
						}
					});
				}
			}
		}

		if ("static".equals(listType)) {

			// staticのときは、"list"属性に定義されているマップを使用
			ret.addAll((List<Map<String, String>>) list);

		} else if ("dto".equals(listType)) {

			// dtoのときは、dtoの中から、"list"属性に指定されているプロパティを指定してマップを取得
			String key = (String) list;
			List<Map<String, Object>> resultList = (List<Map<String, Object>>) ParamUtil
					.getParamValueByType("dto", key, dto);
			if (resultList != null && !resultList.isEmpty()) {
				for (Map<String, Object> data : resultList) {
					ret.add(new HashMap<String, String>() {
						{
							put("list_value", (String) data.get("value"));
							put("list_caption", (String) data.get("caption"));
						}
					});
				}
			}

		} else if ("query".equals(listType)) {

			Map<String, Object> map = (Map<String, Object>) list;
			String query = (String) map.get("query");
			List<Map<String, String>> condList = (List<Map<String, String>>) map
					.get("cond");
			Map<String, Object> params = new HashMap<String, Object>();

			for (Map<String, String> mp : condList) {
				String value = (String) ParamUtil.getParamValueByType(
						mp.get("transfer_type"), mp.get("transfer_val"), dto);
				params.put(mp.get("key"), value);
			}

			List<Map<String, Object>> resultList = new DomaSQLQueryBuilder()
					.getListResult(query, params, dto);
			if (resultList != null && !resultList.isEmpty()) {
				for (Map<String, Object> data : resultList) {
					ret.add(new HashMap<String, String>() {
						{
							put("list_value", (String) data.get("value"));
							put("list_caption", (String) data.get("caption"));
						}
					});
				}
			}

		}

		return ret;

	}

	/**
	 * 指定されたactionから、外部参照先の文字列を作成
	 *
	 * @param action
	 * @param dto DTO
	 *
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private Map<String, String> getForeignSetting(	Object action,
													String val,
													ActionDto dto) {

		Map<String, String> ret = new HashMap<String, String>();
		StringBuffer actionUrl = new StringBuffer();
		StringJoiner sjParam = new StringJoiner("&");

		if (action != null) {

			Map<String, Object> actionMap = ((Map) action);

			String targetActionId = (String) actionMap.get("target_action_id");
			actionUrl.append(targetActionId);

			Map<String, String> returnItems = (Map<String, String>) actionMap
					.get("return_items");

			Map<String, Map<String, String>> params = (Map<String, Map<String, String>>) actionMap
					.get("params");
			if (params != null) {
				int cnt = 0;
				for (Map.Entry<String, Map<String, String>> mp : params
						.entrySet()) {
					actionUrl.append(((cnt == 0) ? "?" : "&"));
					String value = ParamUtil
							.getParamStrValueByType(mp.getValue(), dto);
					actionUrl.append(mp.getKey() + "=" + value);
					sjParam.add(mp.getKey() + "=" + value);
					cnt++;
				}
			}
			String disp = (String) ParamUtil.getParamValueByType("param",
					val + "_disp", dto);
			ret.put("action", actionUrl.toString());
			ret.put("action_id", targetActionId);
			ret.put("params", sjParam.toString());
			ret.put("disp", disp);
			if (returnItems != null) {
				ret.put("foreign_ret_value", returnItems.get("val"));
				ret.put("foreign_ret_disp", returnItems.get("disp"));
			}
		}

		return ret;

	}

	/**
	 * 指定されたactionから、外部参照先の文字列を作成
	 *
	 * @param action
	 * @param dto DTO
	 *
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private void setConbineActionIdAndParams(	Map<String, Object> item,
												ActionDto dto) {

		String ret = "";
		Map<String, Object> dataMap = new HashMap<String, Object>();
		if (item != null) {

			Map<String, Object> actionMap = ((Map) item.get("action"));
			Map<String, Map<String, String>> params = (Map<String, Map<String, String>>) actionMap
					.get("params");

			if (params != null) {
				for (Map.Entry<String, Map<String, String>> mp : params
						.entrySet()) {
					dataMap.put(mp.getKey(),
							ParamUtil.getParamValueByType(
									mp.getValue().get("type"),
									mp.getValue().get("val"), dto));
				}
			}

			StringJoiner actionUrl = new StringJoiner("?");
			StringJoiner actionUrlParam = new StringJoiner("&");
			String targetActionId = (String) actionMap.get("target_action_id");
			for (Map.Entry<String, Object> mp : dataMap.entrySet()) {
				actionUrlParam.add(mp.getKey() + "=" + mp.getValue());
			}
			if (!StringUtils.isEmpty(targetActionId)) {
				actionUrl.add(targetActionId);
			}
			if (actionUrlParam.length() > 0) {
				actionUrl.add(actionUrlParam.toString());
			}
			ret = actionUrl.toString();

		}

		item.put("action", ret);
		if (!MapUtils.isEmpty(dataMap)) {
			item.put("data_json", JsonUtil.mapToJson(dataMap));
		}

	}

	/***
	 *
	 * グループ化した入力項目を表示
	 *
	 * @param input
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private Map<String, Object> groupedFormsInput(LinkedHashMap<String, Object> input) {

		Map<String, Object> ret = new LinkedHashMap<String, Object>();

		for (Map.Entry<String, Object> e : input.entrySet()) {

			Map<String, Object> mp = (Map<String, Object>) e.getValue();
			String groupName = (String) mp.get("group_name");
			String groupSeparator = StringUtils
					.defaultIfEmpty((String) mp.get("group_separator"), "");

			if (StringUtils.isEmpty(groupName)) {

				// グループ設定がされていないときは、何もせず返却用のマップに詰める
				ret.put(e.getKey(), e.getValue());

			} else {

				// グループ設定がされているときは、キーをグループ名、タイプ属性を"group"にしたオブジェクトにして、その配下にオブジェクトを詰める
				Map<String, Object> mpVal = (Map<String, Object>) ret
						.get(groupName);
				if (mpVal == null) {
					mpVal = new LinkedHashMap<String, Object>() {
						{
							put("type", "group");
							put("separator", groupSeparator);
							put("name", groupName);
						}
					};
				}

				List<Map<String, Object>> list = (List<Map<String, Object>>) mpVal
						.get("list");
				if (list == null) {
					list = new ArrayList<Map<String, Object>>();
				}
				list.add(mp);

				mpVal.put("list", list);

				ret.put(groupName, mpVal);

			}
		}

		return ret;

	}

}
