package jp.ill.photon.module.db;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.builder.DomaSQLQueryBuilder;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;

import org.apache.commons.lang3.StringUtils;

public class GeneralDbSelectListModule implements PhotonModule {

	@ModuleParam(required=true)
	private String searchTargetTable;

	@ModuleParam(required=true)
	private List<Map<String, Object>> columns;

	@ModuleParam(required=false)
	private List<Map<String, Object>> searchConditions;

	@ModuleParam(required=false)
	private Map<String, Object> sortConditions;

	@ModuleParam(required=false)
	private Map<String, Object> formItems;

	@ModuleParam(required=false)
	private String limit;

	@ModuleParam(required=false)
	private List<Map<String, Object>> dataFormat;

	/**
	 * searchTargetTableを取得します。
	 * @return searchTargetTable
	 */
	public String getSearchTargetTable() {
		return searchTargetTable;
	}

	/**
	 * searchTargetTableを設定します。
	 * @param searchTargetTable
	 */
	public void setSearchTargetTable(String searchTargetTable) {
		this.searchTargetTable = searchTargetTable;
	}

	/**
	 * columnsを取得します。
	 * @return columns
	 */
	public List<Map<String, Object>> getColumns() {
		return columns;
	}

	/**
	 * columnsを設定します。
	 * @param columns
	 */
	public void setColumns(List<Map<String, Object>> columns) {
		this.columns = columns;
	}

	/**
	 * searchConditionsを取得します。
	 * @return searchConditions
	 */
	public List<Map<String, Object>> getSearchConditions() {
		return searchConditions;
	}

	/**
	 * searchConditionsを設定します。
	 * @param searchConditions
	 */
	public void setSearchConditions(List<Map<String, Object>> searchConditions) {
		this.searchConditions = searchConditions;
	}

	/**
	 * sortConditionsを取得します。
	 * @return sortConditions
	 */
	public Map<String, Object> getSortConditions() {
		return sortConditions;
	}

	/**
	 * sortConditionsを設定します。
	 * @param sortConditions
	 */
	public void setSortConditions(Map<String, Object> sortConditions) {
		this.sortConditions = sortConditions;
	}

	/**
	 * formItemsを取得します。
	 * @return formItems
	 */
	public Map<String, Object> getFormItems() {
		return formItems;
	}

	/**
	 * formItemsを設定します。
	 * @param formItems
	 */
	public void setFormItems(Map<String, Object> formItems) {
		this.formItems = formItems;
	}

	/**
	 * limitを取得します。
	 * @return limit
	 */
	public String getLimit() {
		return limit;
	}

	/**
	 * limitを設定します。
	 * @param limit
	 */
	public void setLimit(String limit) {
		this.limit = limit;
	}

	/**
	 * dataFormatを取得します。
	 * @return dataFormat
	 */
	public List<Map<String, Object>> getDataFormat() {
		return dataFormat;
	}

	/**
	 * dataFormatを設定します。
	 * @param dataFormat
	 */
	public void setDataFormat(List<Map<String, Object>> dataFormat) {
		this.dataFormat = dataFormat;
	}

	/**
	 * 型マップを取得ます。
	 * @return 型マップ
	 */
	private Map<String, Class<?>> getTypeConvertMap() {
		Map<String, Class<?>> typeConvertMap = new HashMap<>();
		typeConvertMap.put("int", Integer.class);
		typeConvertMap.put("int[]", Integer[].class);
		typeConvertMap.put("text", String.class);
		typeConvertMap.put("text[]", String[].class);
		return typeConvertMap;
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	@Override
	public ModuleResult execute(ModuleContext context) {

		ActionDto dto = context.getDto();

		List<Map<String, Object>> sortConditionList = new LinkedList<Map<String, Object>>();
		String selectedSortValue = "";
		if (sortConditions != null) {
			sortConditionList = (List) sortConditions.get("list");
			selectedSortValue = (String)sortConditions.get("value");
		}

		Map<String, Object> condParams = new LinkedHashMap<String, Object>();
		setCondParam(condParams, dto);

		Object offsetObj = getParamValue(condParams, "p", null);
		String offset = (offsetObj != null)? offsetObj.toString() : null;

		String query = buildSql(
				searchTargetTable,
				buildSqlFields(columns),
				buildSqlSearchConditions(searchConditions, condParams),
				buildSqlSortConditions(sortConditionList, selectedSortValue),
				limit, offset);

		ModuleResult result = new ModuleResult();
		result.getReturnData().put("data", new HashMap<String, Object>());
		result.getReturnData().put("list", new ArrayList<>());
		result.getReturnData().put("cnt", 0);

		if (!query.isEmpty()) {
			DomaSQLQueryBuilder queryBuilder = new DomaSQLQueryBuilder();
			List<Map<String, Object>> resultList = queryBuilder
					.getListResult(query, condParams, dto);
			formatData(resultList, dto);
			result.getReturnData().put("list", resultList);
			int cnt = 0;
			if (resultList != null && !resultList.isEmpty() && resultList.get(0).get("count") != null) {
				cnt = Integer.parseInt(String.valueOf(resultList.get(0).get("count")));
				result.getReturnData().put("data", resultList.get(0));
			}
			result.getReturnData().put("cnt", cnt);
		}
		// 現在ページ番号のセット
		result.getReturnData().put("p", offset);


		return result;
	}

	/**
	 * フォーマット
	 *
	 * @param list 設定するパラメータ
	 * @param dto DTO
	 *
	 * */
	@SuppressWarnings("unchecked")
	private void formatData(List<Map<String, Object>> list, ActionDto dto) {

		Map<String, Class<?>> typeConvertMap = getTypeConvertMap();

		Map<String, String> convertMap = new HashMap<String, String>();
		if (dataFormat != null) {
			for(Map<String, Object> map : dataFormat) {
				for(String key : map.keySet()) {
					convertMap.put(key, (String)map.get(key));
				}
			}
		}

		if (list != null) {

			for(Map<String, Object> map : list) {

				for(String key : map.keySet()) {

					if (convertMap.containsKey(key) && typeConvertMap.containsKey(convertMap.get(key))) {

						String typeNameVal = convertMap.get(key);
						Object val = map.get(key);
						Object convertedVal = null;

						if (val != null) {
							switch (typeNameVal) {
							case "text[]":
								convertedVal = JsonUtil.jsonToList((String)val).toArray(new String[0]);
								break;
							case "int[]":
								convertedVal = JsonUtil.jsonToList((String)val).toArray(new Integer[0]);
								break;
							default:
								convertedVal = val;
							}
						} else {
							convertedVal = null;
						}

						map.put(key, convertedVal);

					}

				}

			}

		}

	}

	/**
	 * パラメータ設定
	 *
	 * @param condParams 設定するパラメータ
	 * @param dto DTO
	 *
	 * */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private void setCondParam(Map<String, Object> condParams, ActionDto dto) {

		if (formItems != null && !formItems.isEmpty()) {

			for (Map.Entry<String, Object> mp : formItems.entrySet()) {

				Map<String, Object> value = (Map)mp.getValue();
				Object convVal = value.get("value");

				// この時点で"values"が存在しているということは、
				// フォームモジュールなどで整形されていない状態ということ
				if (value.containsKey("values")) {

					String type = (String)((Map<String, Object>)value.get("values")).get("type");
					String val = (String)((Map<String, Object>)value.get("values")).get("val");

					convVal = ParamUtil.getParamValueByType(
							type, val, dto);

				}

				condParams.put(mp.getKey(), convVal);

			}

		}

	}

	/**
	 * SQL組立
	 *
	 * @param table テーブルの文字列
	 * @param fields フィールドの文字列
	 * @param conditions 条件の文字列。条件の入力値は、別の箇所で設定する
	 * @param sorts ソートフィールド
	 * @param limit limit
	 * @param offset offset
	 * @return 組み立てられたSQL
	 *
	 * */
	private String buildSql(String table, String fields, String conditions, String sorts, String limit, String offset) {

		StringBuffer sql = new StringBuffer();

		sql.append("SELECT ");
		sql.append(fields);
		sql.append("FROM ").append(table).append(" ");
		sql.append("WHERE 1 = 1 ");
		if (!conditions.isEmpty()) {
			sql.append(conditions);
		}
		if (!sorts.isEmpty()) {
			sql.append("ORDER BY ");
			sql.append(sorts);
		}
		if (!StringUtils.isEmpty(limit)) {
			sql.append("LIMIT " + limit + " ");
			if (!StringUtils.isEmpty(offset)) {
				int offsetVal = Integer.parseInt(limit) * (Integer.parseInt(offset) - 1);
				sql.append("OFFSET " + offsetVal + " ");
			}
		}
		return sql.toString();
	}

	/**
	 * フィールド文字列組立
	 *
	 * @param fields フィールド
	 * @return 組み立てられたフィールド文字列
	 *
	 * */
	private String buildSqlFields(List<Map<String, Object>> fields) {
		StringBuffer sb = new StringBuffer();
		int cnt = 0;
		for (Map<String, Object> map : fields) {
			if (cnt > 0) {
				sb.append(",");
			}
			for(String key : map.keySet()) {
				sb.append(map.get(key) + " as " + key + " ");
			}
			cnt++;
		}
		return sb.toString();
	}

	/**
	 * 条件文字列組立
	 *
	 * @param conditions 条件
	 * @param params パラメータ値
	 * @return 組み立てられた条件文字列
	 *
	 * */
	@SuppressWarnings({ "unused", "unchecked", "rawtypes" })
	private String buildSqlSearchConditions(List<Map<String, Object>> conditions, Map<String, Object> params) {
		StringBuffer sb = new StringBuffer();
		if (conditions != null) {
			for (Map<String, Object> map : conditions) {

				String key = (String) getParamValue(map, "key", "");
				String type = (String) getParamValue(map, "type", "");
				String value = (String) getParamValue(map, "value", "");

				// 値が取得された場合にのみ、WHERE句を付ける
				boolean isBlank = false;
				String wherePiece = "";

				if ("or".equals(type)){

					String paramValue = (String) getParamValue(params, key, "");
					isBlank = StringUtils.isEmpty(paramValue);
					if (!isBlank) {

						// "or"設定時のときは
						// field1 = value or field2 = value or field3 = valueといった風にする
						StringBuffer sbOr = new StringBuffer();
						List<Map<String, Object>> targets = (List)getParamValue(map, "targets", "");

						List<String> wherePieceList = targets.stream().collect(
								() -> new ArrayList<String>(),
								(targetList, targetMap) -> {
									targetList.add(
											(String) targetMap.get("field")
											+ " " + getOperandFromTypeString((String) targetMap.get("type"))
											+ " " + value
											);
								},
								(container1, container2) -> {
									container1.addAll(container2);
								}
						);

						wherePiece = wherePieceList.stream()
								.filter( p -> p != null && !p.equals("") )
								.collect(Collectors.joining(" or ", "(", ")"));

					}

				} else {

					String targetField = (String) getParamValue(map, "target_field", "");
					if ("in".equals(type)) {
						String[] paramValues = (String[])getParamValue(params, key, new String[0]);
						if (paramValues == null) {
							isBlank = true;
						} else {
							if (paramValues.length == 0) {
								isBlank = true;
							} else if(paramValues.length == 1  && StringUtils.isEmpty(paramValues[0]) ) {
								isBlank = true;
							}
						}
					} else {
						String paramValue = (String) getParamValue(params, key, "");
						isBlank = StringUtils.isEmpty(paramValue);
					}

					wherePiece = targetField + " " + getOperandFromTypeString(type) + " " + value + " ";

				}

				if (!isBlank) {
					sb.append(" and " + wherePiece);
				}

			}
		}
		return sb.toString();
	}

   /**
	* ソートフィールド文字列組立
	*
	* @param sorts ソートフィールド
	* @return 組み立てられたソートフィールド文字列
	*
	* */
	@SuppressWarnings("unchecked")
	private String buildSqlSortConditions(List<Map<String, Object>> sorts, String selectedValue) {
		StringBuffer sb = new StringBuffer();
		if (sorts != null) {
			for (Map<String, Object> map : sorts) {
				if (selectedValue.equals(map.get("list_value"))) {
					if (map.get("orders") != null) {
						List<String> list = (List<String>)map.get("orders");
						int cnt = 0;
						for (String val : list) {
							if (cnt > 0) {
								sb.append(", ");
							}
							sb.append(val + " ");
							cnt++;
						}
					}
				}
			}
		}
		return sb.toString();
	}

	/**
	 * オペランド文字列取得
	 *
	 * @param type タイプ文字列
	 * @return オペランド
	 *
	 * */
	private String getOperandFromTypeString(String type) {
		String ret = "=";
		if ("eq".equals(type)) {
			ret = "=";
		} else if ("ne".equals(type)) {
			ret = "!=";
		} else if ("gt".equals(type)) {
			ret = ">";
		} else if ("ge".equals(type)) {
			ret = ">=";
		} else if ("lt".equals(type)) {
			ret = "<";
		} else if ("le".equals(type)) {
			ret = "<=";
		} else if ("in".equals(type)) {
			ret = "in";
		} else if ("like".equals(type)) {
			ret = "like";
		}
		return ret;
	}


}
