package jp.ill.photon.module.db;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;



/**
 * 一括登録処理を実施するクラス
 *
 * @author m_fukukawa
 *
 */
public class BulkModifyModule implements PhotonModule {

	/* モジュールパラメータ */
	@ModuleParam(required=true)
	private String tenantId;

	@ModuleParam(required=true)
	private String editTableCd;

	@ModuleParam(required=true)
	private String dataJson;

	@ModuleParam(required=false)
	private String userName;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "insert")
	private String modifyType;

	@ModuleParam(required=false)
	private Map<String, Object> after;

	@ModuleParam(required=false)
	private Map<String, Object> messages;

	/* モジュール内で使用する変数 */
	private static class ModifyType {
		public static final String INSERT = "insert";
		public static final String UPDATE = "update";
		public static final String MERGE = "merge";
	}
	private List<Map<String, Object>> json;

	@SuppressWarnings("unchecked")
	@Override
	public ModuleResult execute(ModuleContext context) {

		int sqlResult = -1;
		try {
			json = JsonUtil.jsonToList(dataJson);
			doModify();
			sqlResult = 0;
		} catch (Exception e) {
			sqlResult = -1;
		}

		ModuleResult result = new ModuleResult();
		result.getReturnData().put("sql-result", sqlResult);
		String resultCd = (sqlResult == -1)? "failed" : "success";
		result.getReturnData().put("result_code", resultCd);
		result.setResultCode(resultCd);

		// モジュールのパラメータ"after"に値がセットされていたとき
		if (after != null) {
			Map<String, Object> mp = (Map<String, Object>)after.get(result.getResultCode());
			if (mp != null) {
				result.getReturnData().put("after_params", mp.get("params"));
				result.setNextPath((String)mp.get("next_path"));
				result.setResultType((String)mp.get("result_type"));
			}
		}

		if (!MapUtils.isEmpty(messages)) {
			String mes = ParamUtil.getParamStrValueByType(messages.get(result.getResultCode()), context.getDto());
			if (!StringUtils.isEmpty(mes)) {
				Map<String, List<String>> errorMessageListMap = new HashMap<String, List<String>>();
				errorMessageListMap.put("", new ArrayList<String>(){{
					add(mes);
				}});
				result.getReturnData().put("messages", errorMessageListMap);
			}
		}

		return result;

	}

	/**
	 *
	 * 対象テーブルマスタ更新処理
	 *
	 * @throws Exception
	 *
	 */
	private void doModify() throws Exception {

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();

		// トランザクション開始
		tm.required(() -> {

			if (!CollectionUtils.isEmpty(json)) {

				Map<String, Object> key = dao.getKeyColumns(tenantId, editTableCd);
				if (!MapUtils.isEmpty(key)) {

					String keyColumns = (String)key.get("val");

					for (Map<String, Object> mp : json) {

						// INSERT文実行
						String data = JsonUtil.mapToJson(mp);
						Map<String, Object> ret = null;
						if (ModifyType.UPDATE.equals(modifyType)) {
							ret = dao.updateEditTableDataByKeyColumns(tenantId, editTableCd, userName, data, keyColumns);
						} else if(ModifyType.MERGE.equals(modifyType)) {
							// TODO:merge処理
							ret = dao.updateMergeEditTableData(tenantId, editTableCd, userName, data, keyColumns);
						} else {
							ret = dao.insertEditTableDataByKeyColumns(tenantId, editTableCd, userName, data, keyColumns);
						}

					}

				}

			}

			return 0;

		});

	}

	/**
	 * tenantIdを取得します。
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	/**
	 * editTableCdを取得します。
	 * @return editTableCd
	 */
	public String getEditTableCd() {
		return editTableCd;
	}

	/**
	 * editTableCdを設定します。
	 * @param editTableCd
	 */
	public void setEditTableCd(String editTableCd) {
		this.editTableCd = editTableCd;
	}

	/**
	 * dataJsonを取得します。
	 * @return dataJson
	 */
	public String getDataJson() {
		return dataJson;
	}

	/**
	 * dataJsonを設定します。
	 * @param dataJson
	 */
	public void setDataJson(String dataJson) {
		this.dataJson = dataJson;
	}

	/**
	 * userNameを取得します。
	 * @return userName
	 */
	public String getUserName() {
		return userName;
	}

	/**
	 * userNameを設定します。
	 * @param userName
	 */
	public void setUserName(String userName) {
		this.userName = userName;
	}

	/**
	 * modifyTypeを取得します。
	 * @return modifyType
	 */
	public String getModifyType() {
		return modifyType;
	}

	/**
	 * modifyTypeを設定します。
	 * @param modifyType
	 */
	public void setModifyType(String modifyType) {
		this.modifyType = modifyType;
	}

	/**
	 * afterを取得します。
	 * @return after
	 */
	public Map<String, Object> getAfter() {
		return after;
	}

	/**
	 * afterを設定します。
	 * @param after
	 */
	public void setAfter(Map<String, Object> after) {
		this.after = after;
	}

	/**
	 * messagesを取得します。
	 * @return messages
	 */
	public Map<String, Object> getMessages() {
		return messages;
	}

	/**
	 * messagesを設定します。
	 * @param messages
	 */
	public void setMessages(Map<String, Object> messages) {
		this.messages = messages;
	}

}
