package jp.ill.photon.module.date;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;

/**
 * [current_datetime] 現在日時を指定フォーマットで返すモジュール.
 * 
 * <p>
 * タイムスタンプ文字列等で利用可能なフォーマット済み現在日時を返す。<br/>
 * フォーマットパターンはパラメータで指定可能。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>date</dt>
 * <dd>現在日時オブジェクト（LocalDateTime型）</dd>
 * <dt>formatted</dt>
 * <dd>フォーマット済み日時（String型）</dd>
 * </dl>
 * </p>
 * 
 * @author h_tanaka
 *
 */
public class CurrentDateTimeModule implements PhotonModule {

	/**
	 * フォーマット指定用モジュールパラメータ.
	 * 
	 * <p>
	 * DateTimeFormatterで利用可能なパターン文字列を受け取る。<br/>
	 * このパラメータを指定しなかった場合は「yyyyMMddHHmmss」形式でフォーマットされる。
	 * </p>
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static</dd>
	 * <dt>transfer_val</dt>
	 * <dd>yyyyMMddHHmmss</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = false)
	private String formatPattern;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		// 結果オブジェクト生成
		ModuleResult result = new ModuleResult();

		LocalDateTime localDT = LocalDateTime.now();

		String formattedDateTime = DateTimeFormatter
				.ofPattern(getFormatPattern()).format(localDT);

		result.getReturnData().put("date", localDT);
		result.getReturnData().put("formatted", formattedDateTime);

		return result;
	}

	public String getFormatPattern() {
		if (formatPattern == null || formatPattern.length() == 0) {
			formatPattern = "yyyyMMddHHmmss";
		}
		return formatPattern;
	}

	public void setFormatPattern(String formatPattern) {
		this.formatPattern = formatPattern;
	}

}
