package jp.ill.photon.module.date;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.exception.PhotonPageNotFoundException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;

/**
 * [add_date_time] 日時の加算結果を返すモジュール.
 * 
 * <p>
 * 入力: <br/>
 * date: 加算対象日時（入力がない場合は何も行いません）<br/>
 * formatPattern: 【必須】フォーマット<br/>
 * field: 【必須】加算対象フィールド（y：年、M：月、d：日、H：時、m：分、s：秒）<br/>
 * amount: 【必須】加算する日付または時間の量（マイナス値は減算します）<br/>
 * </p>
 * 
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>date</dt>
 * <dd>加算後の日時(String)</dd>
 * </dl>
 * </p>
 * 
 */
public class AddDateTimeModule implements PhotonModule {

	@ModuleParam(required = false)
	private String date;

	@ModuleParam(required = true)
	private String formatPattern;

	@ModuleParam(required = true)
	private String field;

	@ModuleParam(required = true)
	private String amount;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException, PhotonPageNotFoundException {

		// 結果オブジェクト生成
		ModuleResult result = new ModuleResult();

		// 加算対象日時のブランクチェック
		if (getDate() == null || getDate().length() == 0) {
			// 入力がない場合は何もしない
			return result;
		}

		long longAmount = Long.parseLong(getAmount());

		// LocalDateTime は時刻がないと計算できないため
		// フォーマットと加算対象日時に時刻部分を追加する
		String tempFormatter = addTimeToFormatPattern();
		String tempDate = addTimeToDate();

		LocalDateTime localDT = LocalDateTime.parse(tempDate,
				DateTimeFormatter.ofPattern(tempFormatter));

		// 加算
		if ("y".equals(getField())) {
			// 年
			localDT = localDT.plusYears(longAmount);

		} else if ("M".equals(getField())) {
			// 月
			localDT = localDT.plusMonths(longAmount);

		} else if ("d".equals(getField())) {
			// 日
			localDT = localDT.plusDays(longAmount);

		} else if ("H".equals(getField())) {
			// 時
			localDT = localDT.plusHours(longAmount);

		} else if ("m".equals(getField())) {
			// 分
			localDT = localDT.plusMinutes(longAmount);

		} else if ("s".equals(getField())) {
			// 秒
			localDT = localDT.plusSeconds(longAmount);

		} else {
			// 加算対象フィールドの指定に誤りがある場合
			// 加算操作は行わない
		}

		String resultDT = localDT
				.format(DateTimeFormatter.ofPattern(getFormatPattern()));

		result.getReturnData().put("date", resultDT);

		return result;
	}

	// フォーマットに時分秒の追加
	private String addTimeToFormatPattern() {

		String tempFormatPattern = getFormatPattern();

		// 時
		if (!getFormatPattern().contains("HH")) {
			tempFormatPattern = tempFormatPattern + " HH";
		}

		// 分
		if (!getFormatPattern().contains("mm")) {
			tempFormatPattern = tempFormatPattern + " mm";
		}

		// 秒
		if (!getFormatPattern().contains("ss")) {
			tempFormatPattern = tempFormatPattern + " ss";
		}

		return tempFormatPattern;
	}

	// 日時に時分秒の追加
	private String addTimeToDate() {

		String tempDate = getDate();

		// 時
		if (!getFormatPattern().contains("HH")) {
			tempDate = tempDate + " 00";
		}

		// 分
		if (!getFormatPattern().contains("mm")) {
			tempDate = tempDate + " 00";
		}

		// 秒
		if (!getFormatPattern().contains("ss")) {
			tempDate = tempDate + " 00";
		}

		return tempDate;
	}

	public String getDate() {
		return date;
	}

	public void setDate(String date) {
		this.date = date;
	}

	public String getFormatPattern() {
		return formatPattern;
	}

	public void setFormatPattern(String formatPattern) {
		this.formatPattern = formatPattern;
	}

	public String getField() {
		return field;
	}

	public void setField(String field) {
		this.field = field;
	}

	public String getAmount() {
		return amount;
	}

	public void setAmount(String amount) {
		this.amount = amount;
	}

}
