package jp.ill.photon.module.csv;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.postgresql.util.PGobject;
import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.doma.PhotonSqlUpdateQuery;
import jp.ill.photon.doma.PhotonUpdateCommand;
import jp.ill.photon.doma.SqlSelectQueryFactory;
import jp.ill.photon.doma.SqlUpdateQueryFactory;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.MapListParam;
import jp.ill.photon.model.SearchForm;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.EscapeChars;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.StringUtil;

/**
 * データベースのデータをCSVファイルに出力するモジュール
 *
 * @author h_tanaka
 *
 */
public class Sql2CsvModule implements PhotonModule {

	@ModuleParam(required = true, domainObject = true)
	private SearchForm searchForm;

	@ModuleParam(required = true)
	private String sqlFileDirPath;

	@ModuleParam(required = true)
	private String sqlFilePath;

	@ModuleParam(required = false)
	private String afterRowSqlFilePath;

	@ModuleParam(required = false)
	private String progressSqlFilePath;

	@ModuleParam(required = false)
	private String csvOutputDirPath;

	@ModuleParam(required = false)
	private String defaultCsvOutputDirPath;

	@ModuleParam(required = true)
	private String csvOutputFilePath;

	@ModuleParam(required = true)
	private String encoding;

	@ModuleParam(required = true)
	private String lineBreak;

	@ModuleParam(required = true)
	private String delimiter;

	@ModuleParam(required = false)
	private String encloser;

	@ModuleParam(required = true)
	private int queryLimit;

	@ModuleParam(required = true)
	private int totalCount;

	@ModuleParam(required = false)
	private String scheduleId;

	@ModuleParam(required = false)
	private MapListParam csvFieldList;

	@ModuleParam(required = false)
	private MapListParam formFieldList;

	@ModuleParam(required = false)
	private MapListParam convertSettings;

	@ModuleParam(required = false)
	private int formatOnly = 0;

	/** データタイプ */
	public static final class DataTypes {
		public static final String TEXT = "text";
		public static final String NUMBER = "number";
	}

	/** 外部連携区分コード */
	public static final String EXRENKEI_DIV = "ex_renkei_upload_div";

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(Sql2CsvModule.class);

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		// 結果オブジェクト生成
		ModuleResult result = new ModuleResult();

		SqlSelectQueryFactory selectFactory = SqlSelectQueryFactory
				.newInstance();
		SqlSelectQuery selectQuery = selectFactory.createSelectQueryFromFile(
				getSearchForm().getParamMap(), context.getDto(),
				getSqlFileDirPath(), getSqlFilePath());

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		String outputFilePath = getCsvOutputFilePath();

		SqlUpdateQueryFactory updateQueryFactory = SqlUpdateQueryFactory
				.newInstance();
		String updateProgressSql = "";
		if (!StringUtils.isEmpty(getProgressSqlFilePath())) {
			updateProgressSql = updateQueryFactory
					.readSqlFile(getSqlFileDirPath(), getProgressSqlFilePath());
		}

		SqlSelectQuery afterRowQuery = null;
		if (!StringUtils.isEmpty(getAfterRowSqlFilePath())) {
			afterRowQuery = selectFactory.createSelectQueryFromFile(
					new HashMap<String, Object>(), context.getDto(),
					getSqlFileDirPath(), getAfterRowSqlFilePath());
		}

		if (getCsvFieldList() == null
				|| getCsvFieldList().getParamList().size() == 0) {
			logger.warn("csv項目が指定されていません");
			result.getReturnData().put("filepath",
					Paths.get(getCsvOutputDirPath(), outputFilePath)
							.toAbsolutePath().toString());
			return result;
		}

		if (getFormFieldList() == null
				|| getFormFieldList().getParamList().size() == 0) {
			logger.warn("フォーム項目が指定されていません");
			result.getReturnData().put("filepath",
					Paths.get(getCsvOutputDirPath(), outputFilePath)
							.toAbsolutePath().toString());
			return result;
		}

		List<Map<String, Object>> formFieldList = getFormFieldList()
				.getParamList();
		Map<String, Object> formFieldMap = formFieldList.stream()
				.collect(Collectors.toMap(s -> StringUtil
						.defaultString(s.get("form_field_cd"), ""), s -> s));

		List<Map<String, Object>> csvFieldList = getCsvFieldList()
				.getParamList();
		Map<String, Object> csvFieldMap = csvFieldList.stream()
				.collect(Collectors.toMap(s -> StringUtil
						.defaultString(s.get("csv_field_cd"), ""), s -> s));

		List<Map<String, Object>> convertSettings = getConvertSettings()
				.getParamList();

		// ファイル出力開始
		try (BufferedWriter bw = getBufferedWriter(
				Paths.get(getCsvOutputDirPath(), outputFilePath),
				getEncoding())) {

			// ヘッダ出力
			bw.write(csvFieldList.stream()
					.map(s -> csvEscapeString(
							StringUtil.defaultString(s.get("csv_field_cd"))))
					.map(s -> encloseString(StringUtil.defaultString(s, "")))
					.collect(Collectors.joining(getDelimiter())));
			bw.write(getLineBreakString());

			if (getFormatOnly() != 1) {
				int cnt = -1;
				int limit = getQueryLimit();
				int offset = 0;
				String newLine = null;
				while (cnt != 0) {
					selectQuery.addParameter("limit", Integer.class, limit);
					selectQuery.addParameter("offset", Integer.class, offset);
					selectQuery.prepare();

					// 検索処理を実行
					List<Map<String, Object>> resultList = tm.required(() -> {
						SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
								selectQuery, new MapResultListHandler(
										MapKeyNamingType.NONE));
						return command.execute();
					});

					// 検索結果が空ではない場合
					if (resultList != null && !resultList.isEmpty()) {

						Map<String, Object> convertedLine = null;
						for (Map<String, Object> line : resultList) {
							// フォーム変換設定に基づいて行データを変換
							convertedLine = convertLine(line, formFieldMap,
									csvFieldMap, convertSettings);

							newLine = createNewLine(csvFieldList,
									convertedLine);

							// 改行を変換する（改行⇒スペース）
							newLine = EscapeChars.forCSV(newLine);

							// 行データ出力
							bw.write(newLine);
							bw.write(getLineBreakString());

							updateAfterRow(afterRowQuery, context.getDto(),
									line, convertedLine, tm);
						}

						// クエリの結果件数を設定
						cnt = resultList.size();
					} else {
						cnt = 0;
					}

					if (!StringUtils.isEmpty(updateProgressSql)) {
						// 進捗率を更新
						updateProgress(updateQueryFactory, updateProgressSql,
								context.getDto(), outputFilePath, offset, tm);
					}

					offset += limit;
				}
			}

			// ファイル出力完了
		} catch (IOException e) {
			throw new PhotonModuleException("ファイルの出力に失敗しました:" + outputFilePath,
					e);
		}

		// 結果オブジェクトに結果をセット
		result.getReturnData().put("filepath",
				Paths.get(getCsvOutputDirPath(), outputFilePath)
						.toAbsolutePath().toString());

		return result;
	}

	/**
	 * 出力ストリームを生成（UTF-8の場合はBOM文字列を追加）
	 *
	 */
	protected BufferedWriter getBufferedWriter(Path path, String encoding)
			throws IOException {
		BufferedWriter bw = null;
		if (encoding.toLowerCase().equals("utf-8")) {
			OutputStream fos = Files.newOutputStream(path);
			fos.write(0xef);
			fos.write(0xbb);
			fos.write(0xbf);
			bw = new BufferedWriter(new OutputStreamWriter(fos, encoding));
		} else {
			bw = Files.newBufferedWriter(path, Charset.forName(encoding));
		}

		return bw;
	}

	/**
	 * 出力用行を生成
	 *
	 * @param convertedLine
	 * @param columnSettingList
	 * @return
	 */
	protected String createNewLine(	List<Map<String, Object>> csvFieldList,
									Map<String, Object> convertedLine) {

		List<String> newLine = new ArrayList<>();
		String value = null;
		for (Map<String, Object> csvField : csvFieldList) {
			value = StringUtil.defaultString(convertedLine
					.getOrDefault(csvField.get("csv_field_cd"), ""));
			newLine.add(encloseString(csvEscapeString(value)));
		}
		return newLine.stream().collect(Collectors.joining(getDelimiter()));
	}

	/**
	 * 出力ジョブ管理テーブルの進捗率を更新する
	 *
	 * @param factory
	 * @param sql
	 * @param dto
	 * @param filePath
	 * @param offset
	 * @param tm
	 * @throws PhotonModuleException
	 */
	protected void updateProgress(	SqlUpdateQueryFactory factory,
									String sql,
									ActionDto dto,
									String filePath,
									int offset,
									TransactionManager tm)
			throws PhotonModuleException {

		double progress = 0;
		if (offset > 0 && getTotalCount() > 0) {
			progress = (double) offset / (double) getTotalCount() * 100;
		}

		Map<String, Object> srcParams = new HashMap<>();
		PhotonSqlUpdateQuery updateQuery = factory.createUpdateQuery(srcParams,
				dto, sql);

		updateQuery.addParameter("filePath", String.class, filePath);
		updateQuery.addParameter("status", String.class, "1");
		updateQuery.addParameter("isEnd", String.class, "0");
		updateQuery.addParameter("progress", String.class,
				String.valueOf((int) progress));
		updateQuery.addParameter("tenantId", String.class, dto.getTenantId());
		updateQuery.addParameter("scheduleId", String.class, getScheduleId());

		updateQuery.prepare();

		// 更新クエリ実行
		tm.requiresNew(() -> {
			PhotonUpdateCommand command = new PhotonUpdateCommand(updateQuery);
			return command.execute();
		});
	}

	/**
	 * 行処理後クエリが指定されている場合は実行する
	 *
	 * @param factory
	 * @param sql
	 * @param dto
	 * @param line
	 * @param convertedLine
	 * @throws PhotonModuleException
	 */
	protected List<Map<String, Object>> updateAfterRow(	SqlSelectQuery selectQuery,
														ActionDto dto,
														Map<String, Object> line,
														Map<String, Object> convertedLine,
														TransactionManager tm)
			throws PhotonModuleException {

		if (selectQuery == null) {
			return new ArrayList<Map<String, Object>>();
		}

		selectQuery.addParameter("line", Map.class, line);
		selectQuery.addParameter("convertedLine", Map.class, convertedLine);
		selectQuery.addParameter("tenantId", String.class, dto.getTenantId());
		selectQuery.addParameter("scheduleId", String.class, getScheduleId());
		selectQuery.prepare();

		// 更新クエリ実行
		return tm.requiresNew(() -> {
			SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
					selectQuery,
					new MapResultListHandler(MapKeyNamingType.NONE));
			return command.execute();
		});
	}

	/**
	 * フォーム変換設定に基づいて行データを変換
	 *
	 * @param dataMap
	 * @return
	 */
	protected Map<String, Object> convertLine(	Map<String, Object> dataMap,
												Map<String, Object> formFieldMap,
												Map<String, Object> csvFieldMap,
												List<Map<String, Object>> convertSettings) {

		Map<String, Object> filteredDataMap = convertJsonValToCol(dataMap)
				.entrySet().stream()
				.filter(p -> formFieldMap.containsKey(p.getKey()))
				.collect(Collectors.toMap(s -> s.getKey(),
						s -> StringUtil.defaultString(s.getValue(), "")));

		Map<String, Map<String, Object>> convertedMapMap = CsvConverter
				.convertForm2Csv(filteredDataMap, csvFieldMap, convertSettings);

		Map<String, Object> newDataMap = new LinkedHashMap<>();
		Map<String, Object> convertedMap = null;
		for (Map.Entry<String, Map<String, Object>> entry : convertedMapMap
				.entrySet()) {
			convertedMap = entry.getValue();
			newDataMap.put(entry.getKey(), convertedMap.get("value"));
		}

		return newDataMap;
	}

	/**
	 * JSONBカラムが存在したら通常カラムとして使用できるように変換する
	 *
	 * @param dataMap
	 * @return
	 */
	protected Map<String, Object> convertJsonValToCol(Map<String, Object> dataMap) {
		// TODO SqlFileSelectModuleのコンバート処理と共通化する
		Map<String, Object> newMap = new LinkedHashMap<>();
		for (Map.Entry<String, Object> entry : dataMap.entrySet()) {
			if (entry.getValue() != null) {
				if (PGobject.class.equals(entry.getValue().getClass())
						&& "jsonb".equals(((PGobject) entry.getValue())
								.getType().toLowerCase())) {
					PGobject pgVal = (PGobject) entry.getValue();
					newMap.putAll(JsonUtil.jsonToMap(pgVal.getValue()));
					newMap.remove(entry.getKey());
				} else {
					newMap.put(entry.getKey(), entry.getValue());
				}
			}
		}

		return newMap;
	}

	/**
	 * 出力ファイル名に日付情報をセットする
	 *
	 * @param filePattern
	 * @param now
	 * @return
	 */
	protected String getFormattedFilePath(	String filePattern,
											LocalDateTime now) {
		Pattern p = Pattern.compile("(<([^>]+)>)");
		Matcher m = p.matcher(filePattern);
		if (m.find()) {
			String outerMatch = m.group(1);
			String innerMatch = m.group(2);
			String formattedInnerMatch = now
					.format(DateTimeFormatter.ofPattern(innerMatch));
			return filePattern.replaceAll(outerMatch, formattedInnerMatch);
		}

		return filePattern;
	}

	/**
	 * 改行コードを返す
	 *
	 * @return
	 */
	protected String getLineBreakString() {
		if (getLineBreak().equals("CRLF")) {
			return "\r\n";
		} else if (getLineBreak().equals("CR")) {
			return "\r";
		} else {
			return "\n";
		}
	}

	/**
	 * CSV出力用に文字列をエスケープする
	 *
	 * @param str
	 * @return
	 */
	protected String csvEscapeString(String str) {
		String s = StringUtil.defaultString(str, "");
		if(getEncloser() == ""){
			return s;
		}
		return s.replaceAll(getEncloser(), getEncloser() + getEncloser());

	}

	/**
	 * 囲み文字で囲んで返す
	 *
	 * @param str
	 * @return
	 */
	protected String encloseString(String str) {
		return String.format("%s%s%s", getEncloser(), str, getEncloser());
	}

	public SearchForm getSearchForm() {
		return searchForm;
	}

	public void setSearchForm(SearchForm searchForm) {
		this.searchForm = searchForm;
	}

	public String getSqlFileDirPath() {
		if (sqlFileDirPath == null) {
			sqlFileDirPath = "";
		}
		return sqlFileDirPath;
	}

	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	public String getSqlFilePath() {
		return sqlFilePath;
	}

	public void setSqlFilePath(String sqlFilePath) {
		this.sqlFilePath = sqlFilePath;
	}

	public String getCsvOutputFilePath() {
		return csvOutputFilePath;
	}

	public void setCsvOutputFilePath(String csvOutputFilePath) {
		this.csvOutputFilePath = csvOutputFilePath;
	}

	public String getEncoding() {
		return encoding;
	}

	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	public int getQueryLimit() {
		return queryLimit;
	}

	public void setQueryLimit(int queryLimit) {
		this.queryLimit = queryLimit;
	}

	public String getCsvOutputDirPath() {
		if (csvOutputDirPath == null || csvOutputDirPath.length() == 0) {
			return defaultCsvOutputDirPath;
		}
		return csvOutputDirPath;
	}

	public void setCsvOutputDirPath(String csvOutputDirPath) {
		this.csvOutputDirPath = csvOutputDirPath;
	}

	public String getDefaultCsvOutputDirPath() {
		return defaultCsvOutputDirPath;
	}

	public void setDefaultCsvOutputDirPath(String defaultCsvOutputDirPath) {
		this.defaultCsvOutputDirPath = defaultCsvOutputDirPath;
	}

	public String getLineBreak() {
		return lineBreak;
	}

	public void setLineBreak(String lineBreak) {
		this.lineBreak = lineBreak;
	}

	public String getDelimiter() {
		return delimiter;
	}

	public void setDelimiter(String delimiter) {
		this.delimiter = delimiter;
	}

	public String getEncloser() {
		if(encloser == null){
			return "";
		}
		return encloser;
	}

	public void setEncloser(String encloser) {
		this.encloser = encloser;
	}

	public int getTotalCount() {
		return totalCount;
	}

	public void setTotalCount(int totalCount) {
		this.totalCount = totalCount;
	}

	public String getProgressSqlFilePath() {
		return progressSqlFilePath;
	}

	public void setProgressSqlFilePath(String progressSqlFilePath) {
		this.progressSqlFilePath = progressSqlFilePath;
	}

	public String getScheduleId() {
		return scheduleId;
	}

	public void setScheduleId(String scheduleId) {
		this.scheduleId = scheduleId;
	}

	public MapListParam getConvertSettings() {
		return convertSettings;
	}

	public void setConvertSettings(MapListParam convertSettings) {
		this.convertSettings = convertSettings;
	}

	public MapListParam getCsvFieldList() {
		return csvFieldList;
	}

	public void setCsvFieldList(MapListParam csvFieldList) {
		this.csvFieldList = csvFieldList;
	}

	public MapListParam getFormFieldList() {
		return formFieldList;
	}

	public void setFormFieldList(MapListParam formFieldList) {
		this.formFieldList = formFieldList;
	}

	public String getAfterRowSqlFilePath() {
		return afterRowSqlFilePath;
	}

	public void setAfterRowSqlFilePath(String afterRowSqlFilePath) {
		this.afterRowSqlFilePath = afterRowSqlFilePath;
	}

	public int getFormatOnly() {
		return formatOnly;
	}

	public void setFormatOnly(int formatOnly) {
		this.formatOnly = formatOnly;
	}
}
