package jp.ill.photon.module.csv;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringEscapeUtils;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.StringUtil;
import jp.ill.photon.util.CsvUtil;

public class CsvFileInfoModule implements PhotonModule {

	@ModuleParam(required = false)
	private String csvDirPath;

	@ModuleParam(required = true)
	private String csvFilePath;

	@ModuleParam(required = true)
	private String encoding;

	@ModuleParam(required = true)
	private String lineBreak;

	@ModuleParam(required = true)
	private String delimiter;

	@ModuleParam(required = true)
	private String encloser;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		// 結果オブジェクト生成
		ModuleResult result = new ModuleResult();

		Path filePath = Paths.get(getCsvDirPath(), getCsvFilePath());

		int totalLineCount = 1;
		try {
			totalLineCount = (int) Files
					.lines(filePath, Charset.forName(getEncoding())).count();
		} catch (IOException e1) {
			logger.error("ファイル行数の取得に失敗しました:" + getCsvFilePath(), e1);
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setMessages(createErrorMessage(context));
			result.setResultCode("ioerror");
			result.getReturnData().put("line_count", 0);
			result.getReturnData().put("column_list", new ArrayList<>());
			return result;
		}

		List<String> columnList = new ArrayList<>();

		// ファイル入力開始
		try (BufferedReader br = Files.newBufferedReader(filePath,
				Charset.forName(getEncoding()))) {

			String line = br.readLine();
			if (line != null) {
				columnList = csvLineToList(line);
			}

			// ファイル入力完了
		} catch (Exception e) {
			logger.error("ファイルの取込に失敗しました:" + getCsvFilePath(), e);
			result.setResultCode("ioerror");
			result.setResultType(ModuleResult.ResultTypes.ERROR);
			result.setMessages(createErrorMessage(context));
			result.getReturnData().put("line_count", 0);
			result.getReturnData().put("column_list", new ArrayList<>());
			return result;
		}

		result.setResultCode("success");
		result.getReturnData().put("line_count", totalLineCount);
		result.getReturnData().put("column_list", columnList);

		return result;
	}

	protected Map<String, Object> createErrorMessage(ModuleContext context) {
		Map<String, Object> actionModule = context.getActionModule();
		String returnPrefix = StringUtil.defaultString(
				actionModule.getOrDefault("return_prefix", ""), "");

		Map<String, Object> messageMap = new LinkedHashMap<>();
		List<ActionDtoMessage> messageList = new ArrayList<>();
		ActionDtoMessage message = new ActionDtoMessage();
		message.setMessageId("commonNoFileRead");
		message.setReturnPrefix(returnPrefix);

		messageList.add(message);

		messageMap.put("error_page", messageList);
		return messageMap;
	}

	/**
	 * CSV行データをListに変換
	 * 
	 * @param line
	 * @return
	 */
	protected List<String> csvLineToList(String line) {
		String unescaped = StringUtil.excludeUTF8Bom(line);
		List<String> dataList = CsvUtil.csvTextToList(getDelimiter(), getEncloser(), unescaped);

		return dataList;
	}

	public String getCsvDirPath() {
		if (csvDirPath == null) {
			csvDirPath = "";
		}
		return csvDirPath;
	}

	public void setCsvDirPath(String csvDirPath) {
		this.csvDirPath = csvDirPath;
	}

	public String getCsvFilePath() {
		return csvFilePath;
	}

	public void setCsvFilePath(String csvFilePath) {
		this.csvFilePath = csvFilePath;
	}

	public String getEncoding() {
		return encoding;
	}

	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	public String getLineBreak() {
		return lineBreak;
	}

	public void setLineBreak(String lineBreak) {
		this.lineBreak = lineBreak;
	}

	public String getDelimiter() {
		return delimiter;
	}

	public void setDelimiter(String delimiter) {
		this.delimiter = delimiter;
	}

	public String getEncloser() {
		return encloser;
	}

	public void setEncloser(String encloser) {
		this.encloser = encloser;
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(CsvFileInfoModule.class);
}
