package jp.ill.photon.module.csv;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

import jp.ill.photon.util.ArrayUtil;
import jp.ill.photon.util.StringUtil;

public class CsvConverter {

	/**
	 * フォームからCSVに変換する
	 * 
	 * @param formFieldData
	 * @param formFieldSettings
	 * @param convertSettings
	 * @return
	 */
	public static Map<String, Map<String, Object>> convertForm2Csv(	Map<String, Object> srcData,
																	Map<String, Object> csvFieldMap,
																	List<Map<String, Object>> convertSettings) {

		String inputFieldsKey = "form_fields";
		String outputFieldsKey = "csv_fields";

		Map<String, Map<String, Object>> convertedData = new LinkedHashMap<>();

		Map<String, String> convertedResult = null;
		List<String> inputFieldCdList = null;
		List<String> outputFieldCdList = null;
		String joinedValue = null;
        Map<String, Map<String, Object>> splittedData = null;
        Map<String, Object> outputValue = null;
		for (Map<String, Object> setting : convertSettings) {
			outputFieldCdList = getConvertFieldList(setting, outputFieldsKey);
			if (outputFieldCdList == null || outputFieldCdList.size() == 0) {
				continue;
			}

			inputFieldCdList = getConvertFieldList(setting, inputFieldsKey);

			joinedValue = getJoinedValue(inputFieldCdList, srcData);

			convertedResult = convertValueBeforeSplit(joinedValue, setting);
			joinedValue = convertedResult.get("value");

            splittedData = createCsvOutputData(joinedValue, csvFieldMap,
					convertedResult.get("cutback"), outputFieldCdList);
            for (Map.Entry<String, Map<String, Object>> entry : splittedData.entrySet()) {
                outputValue = entry.getValue();
                outputValue.put("value", convertValueAfterSplit(StringUtil.defaultString(outputValue.getOrDefault("value", ""), ""), setting));
                convertedData.put(entry.getKey(), outputValue);
            }
		}

		Map<String, Object> addingField = null;
		for (Map.Entry<String, Object> entry : csvFieldMap.entrySet()) {
			if (!convertedData.containsKey(entry.getKey())) {
				addingField = new HashMap<>();
				addingField.put("value", "");
				convertedData.put(entry.getKey(), addingField);
			}
		}

		return convertedData;
	}

	/**
	 * CSVからフォームに変換する
	 * 
	 * @param formFieldData
	 * @param formFieldSettings
	 * @param convertSettings
	 * @return
	 */
	public static Map<String, Map<String, Object>> convertCsv2Form(	Map<String, Object> srcData,
																	Map<String, Object> formFieldMap,
																	List<Map<String, Object>> convertSettings) {

		String inputFieldsKey = "csv_fields";
		String outputFieldsKey = "form_fields";

		Map<String, Map<String, Object>> convertedData = new LinkedHashMap<>();

		Map<String, String> convertedResult = null;
		List<String> inputFieldCdList = null;
		List<String> outputFieldCdList = null;
		String joinedValue = null;
        Map<String, Map<String, Object>> splittedData = null;
        Map<String, Object> outputValue = null;
		for (Map<String, Object> setting : convertSettings) {
			outputFieldCdList = getConvertFieldList(setting, outputFieldsKey);
			if (outputFieldCdList == null || outputFieldCdList.size() == 0) {
				continue;
			}

			inputFieldCdList = getConvertFieldList(setting, inputFieldsKey);

			joinedValue = getJoinedValue(inputFieldCdList, srcData);

			convertedResult = convertValueBeforeSplit(joinedValue, setting);
			joinedValue = convertedResult.get("value");

            splittedData = createFormOutputData(joinedValue, formFieldMap,
					convertedResult.get("cutback"), outputFieldCdList);
            for (Map.Entry<String, Map<String, Object>> entry : splittedData.entrySet()) {
                outputValue = entry.getValue();
                outputValue.put("value", convertValueAfterSplit(StringUtil.defaultString(outputValue.getOrDefault("value", ""), ""), setting));
                convertedData.put(entry.getKey(), outputValue);
            }
		}

		Map<String, Object> addingField = null;
		for (Map.Entry<String, Object> entry : formFieldMap.entrySet()) {
			if (!convertedData.containsKey(entry.getKey())) {
				addingField = new HashMap<>();
				addingField.put("value", "");
				convertedData.put(entry.getKey(), addingField);
			}
		}

		return convertedData;
	}

	/**
	 * フォームフィールドコードリスト取得
	 * 
	 * @param convertSetting
	 * @return
	 */
	protected static List<String> getConvertFieldList(	Map<String, Object> convertSetting,
														String fieldKey) {
		List<String> fieldCdList = new ArrayList<>();

		for (Map.Entry<String, String> formField : ((Map<String, String>) convertSetting
				.get(fieldKey)).entrySet()) {
			fieldCdList.add(formField.getValue());
		}

		return fieldCdList;
	}

	/**
	 * 入力データ作成
	 * 
	 * @param fieldCdList
	 * @param formFieldData
	 * @return
	 */
	protected static String getJoinedValue(	List<String> fieldList,
											Map<String, Object> formFieldData) {

		if (fieldList == null | fieldList.size() == 0) {
			return "";
		}

		StringBuffer value = new StringBuffer();
		for (String fieldCd : fieldList) {
			if (formFieldData.containsKey(fieldCd)) {
				value.append(StringUtil
						.defaultString(formFieldData.get(fieldCd), ""));
			}
		}

		return value.toString();
	}

	/**
	 * 値変換処理（分割前）
	 * 
	 * @param value
	 * @param convertSetting
	 * @return
	 */
	protected static Map<String, String> convertValueBeforeSplit(	String value,
														Map<String, Object> convertSetting) {

		String cutback = "0";

		Map<String, Object> convertMap = (Map<String, Object>) convertSetting
				.getOrDefault("convert_settings", null);
		if (convertMap != null) {
			if (convertMap.containsKey("convert_str_map")) {
				Map<String, String> convertStrMap = (Map<String, String>) convertMap
						.get("convert_str_map");
				if (convertStrMap.containsKey(value)) {
					value = StringUtil.defaultString(convertStrMap.get(value),
							"");
				}
			}

			if (convertMap.getOrDefault("cutback_by_length", "0").equals("1")) {
				cutback = "1";
			}
		}

		Map<String, String> result = new HashMap<>();
		result.put("cutback", cutback);
		result.put("value", value);

		return result;
	}

	/**
	 * 値変換処理（分割後）
	 * 
	 * @param value
	 * @param convertSetting
	 * @return
	 */
	protected static String convertValueAfterSplit(	String value,
														Map<String, Object> convertSetting) {

		Map<String, Object> convertMap = (Map<String, Object>) convertSetting
				.getOrDefault("convert_settings", null);
		if (convertMap != null) {
			if (convertMap.containsKey("default_value")) {
				if (StringUtils.isEmpty(value)) {
					value = StringUtil
							.defaultString(convertMap.get("default_value"), "");
				}
			}

			if (convertMap.containsKey("fill_text")
					&& convertMap.containsKey("fill_digits")) {
				String fillDigits = StringUtil.defaultString(
						convertMap.getOrDefault("fill_digits", "0"), "0");
				try {
					value = StringUtil.leftPadding(value,
							StringUtil.defaultString(
									convertMap.get("fill_text"), ""),
							Integer.parseInt(fillDigits));
				} catch (NumberFormatException e) {
				}
			}
		}

		return value;
	}

	/**
	 * フォーム出力データを作成
	 * 
	 * @param value
	 * @param convertSetting
	 * @param formFieldSettings
	 * @param cutback
	 * @return
	 */
	protected static Map<String, Map<String, Object>> createFormOutputData(	String value,
																			Map<String, Object> formFieldMap,
																			String cutback,
																			List<String> formFieldCdList) {
		Map<String, Map<String, Object>> convertedData = new HashMap<>();
		Map<String, Integer> formFieldLengthMap = createFormFieldLengthMap(
				formFieldCdList, formFieldMap);
		String outputValue = null;
		String targetValue = new String(value);
		int maxByteLength = 0;
		String[] splittedValue = null;
		String[] restValues = null;
		Map<String, Object> outputFieldMap = null;
		String lastFormFieldCd = null;
		for (String formFieldCd : formFieldCdList) {
			lastFormFieldCd = formFieldCd;

			maxByteLength = formFieldLengthMap.getOrDefault(formFieldCd,
					Integer.MAX_VALUE);

			if (targetValue.length() > 0) {
				splittedValue = StringUtil.lengthSplitBytes(targetValue,
						maxByteLength, "windows-31j");
				outputValue = splittedValue[0];
				restValues = ArrayUtil.rest(splittedValue);
				if (restValues.length > 0) {
					targetValue = String.join("", restValues);
				} else {
					targetValue = "";
				}
			} else {
				outputValue = targetValue;
			}

			outputFieldMap = new HashMap<>();
			outputFieldMap.put("value", outputValue);
			convertedData.put(formFieldCd, outputFieldMap);
		}

		// 足切り時エラーで入力値が残っていたらエラーをセット
		if (targetValue.length() > 0) {
			if (lastFormFieldCd != null
					&& convertedData.containsKey(lastFormFieldCd)) {
				if (cutback.equals("0")) {
					List<String> errors = new ArrayList<>();
					errors.add("max_byte_length_error");
					convertedData.get(lastFormFieldCd).put("errors", errors);
					String exValue = StringUtil.defaultString(
							convertedData.get(lastFormFieldCd).get("value"),
							"");
					convertedData.get(lastFormFieldCd).put("value",
							exValue + targetValue);
				}
			}
		}

		return convertedData;
	}

	/**
	 * CSV出力データを作成
	 * 
	 * @param value
	 * @param convertSetting
	 * @param formFieldSettings
	 * @param cutback
	 * @return
	 */
	protected static Map<String, Map<String, Object>> createCsvOutputData(	String value,
																			Map<String, Object> csvFieldMap,
																			String cutback,
																			List<String> csvFieldCdList) {
		Map<String, Map<String, Object>> convertedData = new HashMap<>();
		Map<String, Integer> csvFieldLengthMap = createCsvFieldLengthMap(
				csvFieldCdList, csvFieldMap);
		String outputValue = null;
		String targetValue = new String(value);
		int maxByteLength = 0;
		String[] splittedValue = null;
		String[] restValues = null;
		Map<String, Object> outputFieldMap = null;
		String lastCsvFieldCd = null;
		for (String csvFieldCd : csvFieldCdList) {
			lastCsvFieldCd = csvFieldCd;

			maxByteLength = csvFieldLengthMap.getOrDefault(csvFieldCd,
					Integer.MAX_VALUE);

			if (targetValue.length() > 0) {
				splittedValue = StringUtil.lengthSplitBytes(targetValue,
						maxByteLength, "windows-31j");
				outputValue = splittedValue[0];
				restValues = ArrayUtil.rest(splittedValue);
				if (restValues.length > 0) {
					targetValue = String.join("", restValues);
				} else {
					targetValue = "";
				}
			} else {
				outputValue = targetValue;
			}

			outputFieldMap = new HashMap<>();
			outputFieldMap.put("value", outputValue);
			convertedData.put(csvFieldCd, outputFieldMap);
		}

		// 足切り時エラーで入力値が残っていたらエラーをセット
		if (cutback.equals("0") && targetValue.length() > 0) {
			if (lastCsvFieldCd != null
					&& convertedData.containsKey(lastCsvFieldCd)) {
				List<String> errors = new ArrayList<>();
				errors.add("max_byte_length_error");
				convertedData.get(lastCsvFieldCd).put("errors", errors);
			}
		}

		return convertedData;
	}

	/**
	 * CSVフィールドコード毎の最大バイト数を格納したマップを生成する
	 * 
	 * @param csvFieldCdList
	 * @param csvFieldMap
	 * @return
	 */
	protected static Map<String, Integer> createCsvFieldLengthMap(	List<String> csvFieldCdList,
																	Map<String, Object> csvFieldMap) {
		Map<String, Integer> lengthMap = new HashMap<>();

		Map<String, Object> csvField = null;
		int maxByteLength = 0;
		String fieldLength = null;
		for (String csvFieldCd : csvFieldCdList) {
			maxByteLength = 0;
			if (csvFieldMap.containsKey(csvFieldCd)) {
				csvField = (Map<String, Object>) csvFieldMap.get(csvFieldCd);
				if (csvField.containsKey("field_length")) {
					fieldLength = StringUtil
							.defaultString(csvField.get("field_length"), "");
					if (fieldLength.length() > 0) {
						try {
							maxByteLength = Integer.parseInt(fieldLength);
						} catch (NumberFormatException e) {
						}
					}
				}
			}
			if (maxByteLength == 0) {
				maxByteLength = Integer.MAX_VALUE;
			}
			lengthMap.put(csvFieldCd, maxByteLength);
		}

		return lengthMap;
	}

	/**
	 * フォームフィールドコード毎の最大バイト数を格納したマップを生成する
	 * 
	 * @param csvFieldCdList
	 * @param csvFieldMap
	 * @return
	 */
	protected static Map<String, Integer> createFormFieldLengthMap(	List<String> formFieldCdList,
																	Map<String, Object> formFieldMap) {
		Map<String, Integer> lengthMap = new HashMap<>();

		Map<String, Object> formField = null;
		int maxByteLength = 0;
		String fieldLength = null;
		for (String formFieldCd : formFieldCdList) {
			maxByteLength = 0;
			if (formFieldMap.containsKey(formFieldCd)) {
				formField = (Map<String, Object>) formFieldMap.get(formFieldCd);
				if (formField.containsKey("max_byte_length")) {
					fieldLength = StringUtil.defaultString(
							formField.get("max_byte_length"), "");
					if (fieldLength.length() > 0) {
						try {
							maxByteLength = Integer.parseInt(fieldLength);
						} catch (NumberFormatException e) {
						}
					}
				}
			}
			if (maxByteLength == 0) {
				maxByteLength = Integer.MAX_VALUE;
			}
			lengthMap.put(formFieldCd, maxByteLength);
		}

		return lengthMap;
	}
}
