package jp.ill.photon.module.csv;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.internal.jdbc.command.MapResultListHandler;
import org.seasar.doma.internal.jdbc.sql.SqlParser;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.command.SelectCommand;
import org.seasar.doma.jdbc.query.SqlSelectQuery;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.MetaObjectDao;
import jp.ill.photon.dao.MetaObjectDaoImpl;
import jp.ill.photon.doma.PhotonSqlUpdateQuery;
import jp.ill.photon.doma.PhotonUpdateCommand;
import jp.ill.photon.doma.SqlUpdateQueryFactory;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.model.ErrorLog;
import jp.ill.photon.model.MapListParam;
import jp.ill.photon.model.MapParam;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.module.validation.FormValidator;
import jp.ill.photon.util.CsvUtil;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.MessageUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

/**
 * CSVファイルのデータをデータベースに登録するモジュール
 *
 * @author h_tanaka
 *
 */
public class Csv2SqlModule implements PhotonModule {

	@ModuleParam(required = true)
	private MapParam sqlParamMap;

	@ModuleParam(required = true)
	private String sqlFileDirPath;

	@ModuleParam(required = true)
	private String sqlFilePath;

	@ModuleParam(required = false)
	private String progressSqlFilePath;

	@ModuleParam(required = false)
	private String csvImportDirPath;

	@ModuleParam(required = true)
	private String csvImportFilePath;

	@ModuleParam(required = true)
	private String encoding;

	@ModuleParam(required = true)
	private String lineBreak;

	@ModuleParam(required = true)
	private String delimiter;

	@ModuleParam(required = false)
	private String encloser;

	@ModuleParam(required = false)
	private MapListParam csvFieldList;

	@ModuleParam(required = false)
	private MapListParam formFieldList;

	@ModuleParam(required = false)
	private MapListParam deleteFormFieldList;

	@ModuleParam(required = false)
	private MapListParam convertSettings;

	@ModuleParam(required = false)
	private String scheduleId;

	/**
	 * 一括更新可能か？（0:不可、1：可。デフォルトは0:不可）
	 */
	@ModuleParam(required = false)
	private String isBulkUpdatable;

	/**
	 * 一括更新する件数
	 */
	@ModuleParam(required = false)
	private String bulkNum;

	/**
	 * インポートステータス
	 *
	 * @author h_tanaka
	 *
	 */
	public static final class ImportStats {
		/** ステータス：未処理 */
		public static final String INIT = "0";

		/** ステータス：インポート中 */
		public static final String IMPORTING = "1";

		/** ステータス：インポート正常終了 */
		public static final String END_OK = "2";

		/** ステータス：インポートシステムエラー終了 */
		public static final String END_HARD_ERROR = "9";

		/** ステータス：インポートエラーあり終了 */
		public static final String END_SOFT_ERROR = "91";
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(Csv2SqlModule.class);

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		// 結果オブジェクト生成
		ModuleResult result = new ModuleResult();

		// クエリオブジェクトを生成
		SqlSelectQuery sfQuery = createQuery(context);

		// 検索処理を実行
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		String csvLine = null;
		Map<String, Object> dataMap = null;
		List<String> columnList = new ArrayList<>();

		Path filePath = null;
		if (getCsvImportDirPath() != null && getCsvImportDirPath() != "") {
			filePath = Paths.get(getCsvImportDirPath(), getCsvImportFilePath());
		} else {
			filePath = Paths.get(getCsvImportFilePath());
		}

		SqlUpdateQueryFactory updateProgressFactory = SqlUpdateQueryFactory
				.newInstance();
		String updateProgressSql = updateProgressFactory
				.readSqlFile(getSqlFileDirPath(), getProgressSqlFilePath());

		int totalLineCount = 1;
		try {
			totalLineCount = (int) Files
					.lines(filePath, Charset.forName(getEncoding())).count();
		} catch (IOException e1) {
			logger.error("ファイルの行数読み込みに失敗しました", e1);
			result.setResultCode("has-error");
			result.getReturnData().put("import_status",
					ImportStats.END_SOFT_ERROR);
			return result;
		} catch (Exception e2) {
			logger.error("ファイルの行数取得に失敗しました", e2);
			result.setResultCode("has-error");
			result.getReturnData().put("import_status",
					ImportStats.END_SOFT_ERROR);
			return result;
		}

		final int countChunk = totalLineCount / 10;
		List<Integer> countList = IntStream.range(1, 10)
				.mapToObj(i -> new Integer(countChunk * i))
				.collect(Collectors.toList());

		if (getCsvFieldList() == null
				|| getCsvFieldList().getParamList().size() == 0) {
			logger.warn("csv項目が指定されていません");
			result.setResultCode("has-error");
			result.getReturnData().put("import_status",
					ImportStats.END_SOFT_ERROR);
			return result;
		}

		if (getFormFieldList() == null
				|| getFormFieldList().getParamList().size() == 0) {
			logger.warn("フォーム項目が指定されていません");
			result.setResultCode("has-error");
			result.getReturnData().put("import_status",
					ImportStats.END_SOFT_ERROR);
			return result;
		}

		List<Map<String, Object>> formFieldList = getFormFieldList()
				.getParamList();
		Map<String, Object> formFieldMap = formFieldList.stream()
				.collect(Collectors.toMap(s -> StringUtil
						.defaultString(s.get("form_field_cd"), ""), s -> s));

		List<Map<String, Object>> deleteFormFieldList = null;
		if (getDeleteFormFieldList() != null) {
			deleteFormFieldList = getDeleteFormFieldList().getParamList();
		} else {
			deleteFormFieldList = new ArrayList<>();
		}

		String editTableCd = null;
		for (Map<String, Object> formField : formFieldList) {
			editTableCd = StringUtil.defaultString(
					formField.getOrDefault("edit_table_cd", ""), "");
			if (editTableCd.length() > 0) {
				break;
			}
		}

		List<Map<String, Object>> csvFieldList = getCsvFieldList()
				.getParamList();
		Map<String, Object> csvFieldMap = csvFieldList.stream()
				.collect(Collectors.toMap(s -> StringUtil
						.defaultString(s.get("csv_field_cd"), ""), s -> s));

		List<Map<String, Object>> convertSettings = getConvertSettings()
				.getParamList();

		FormValidator validator = new FormValidator(
				context.getDto().getTenantId(), "csv_import", formFieldList);

		FormValidator deleteValidator = new FormValidator(
				context.getDto().getTenantId(), "csv_import",
				deleteFormFieldList);

		// 一括更新件数を取得
		int bulkUpdateNum = 20;
		if (getBulkNum() != null) {
			try {
				bulkUpdateNum = Integer.parseInt(getBulkNum());
			} catch (NumberFormatException e) {
			}
		}
		List<Map<String, Object>> updateBuffer = new ArrayList<>();

		// ファイル入力開始
		int errorCount = 0;
		try (BufferedReader br = Files.newBufferedReader(filePath,
				Charset.forName(getEncoding()))) {

			StringBuffer lineBuffer = new StringBuffer();
			List<Map<String, Object>> sqlResult = null;
			int lineIndex = 0;
			String line = null;
			while ((csvLine = br.readLine()) != null
					|| lineBuffer.length() > 0) {
				if (csvLine != null) {
					csvLine = csvLine.replace("\uFEFF", "");
					lineBuffer.append(csvLine);
				}

				if (!isEndOfLine(lineBuffer.toString()) && csvLine != null) {
					lineBuffer.append(getLineBreakString());
					lineIndex++;
					continue;
				} else {
					line = lineBuffer.toString();
					lineBuffer = new StringBuffer();
				}

				if (line == null || line.trim().length() == 0) {
					lineIndex++;
					continue;
				}

				// ヘッダ情報を取得
				if (lineIndex == 0) {
					columnList = csvLineToList(line);
					lineIndex++;
					continue;
				}

				// 行データをコンバートしてオブジェクトに変換
				dataMap = convertLineToMap(line, columnList, csvFieldMap);

				// フォーム変換設定に基づいて行データを変換
				dataMap = convertCsv2Form(dataMap, convertSettings,
						formFieldMap);

				// 変換後にバリデーション処理
				Map<String, Object> errors = null;

				// 削除か更新かでバリデータを呼び分け
				if (StringUtil
						.defaultString(dataMap
								.getOrDefault("ex_renkei_upload_div", ""))
						.equals("1")) {
					errors = deleteValidator.validate(dataMap,
							new HashMap<String, Map<String, Object>>(),
							context.getDto(), false, true);
				} else {
				Map<String, Object> extraParamMap = new HashMap<>();
				Map<String, Object> extraParamMapVal = new HashMap<>();
				Map<String, Object> extraParam = new HashMap<>();

				Map<String, Object> extraParamValue = new HashMap<>();
				extraParamValue.put("type", "static");
				extraParamValue.put("data_type", "object_map");
				extraParamValue.put("val", dataMap);
				extraParam.put("requests", extraParamValue);

				extraParamValue = new HashMap<>();
				extraParamValue.put("type", "static");
				extraParamValue.put("val", editTableCd);
				extraParam.put("editTableCd", extraParamValue);

				extraParamMapVal.put("val", extraParam);
				extraParamMap.put("search_form", extraParamMapVal);
				validator.setExtraParamMap(extraParamMap);
					errors = validator.validate(dataMap,
							new HashMap<String, Map<String, Object>>(),
							context.getDto(), false, true);
				}

				if (!errors.isEmpty()) {
					// エラーログを追加
					MessageUtil.buildMessage(errors, context.getDto());
					writeErrorLogs(tm, lineIndex, errors, context.getDto());
					validator.clearMessages();
					deleteValidator.clearMessages();
					errorCount++;
					lineIndex++;
					continue;
				}

				if (getIsBulkUpdatable().equals("1")) {
					dataMap.put("json", JsonUtil.mapToJson(dataMap));
					updateBuffer.add(dataMap);

					if (updateBuffer.size() > bulkUpdateNum) {

						sqlResult = this.executeBulkUpdateQuery(tm, sfQuery,
								updateBuffer);
						updateBuffer.clear();
					}
				} else {
					// 行データの内容をSQLパラメータとしてクエリにセット
					sfQuery.addParameter("scheduleId", String.class,
							getScheduleId());
					sfQuery.addParameter("fields", Map.class, dataMap);
					sfQuery.prepare(); // 更新クエリ実行
					sqlResult = tm.requiresNew(() -> {
						SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
								sfQuery, new MapResultListHandler(
										MapKeyNamingType.NONE));
						return command.execute();
					});
				}

				lineIndex++;

				if (countList.contains(new Integer(lineIndex))) {
					// 進捗率を更新
					int progress = (countList.indexOf(new Integer(lineIndex))
							+ 1) * 10;
					updateProgress(updateProgressFactory, updateProgressSql,
							context.getDto(), progress, tm);
				}
			}

			// バッファに残ったデータを登録
			if (getIsBulkUpdatable().equals("1")) {
				if (updateBuffer.size() > 0) {
					sqlResult = this.executeBulkUpdateQuery(tm, sfQuery,
							updateBuffer);
				}
			}

			// ファイル入力完了
		} catch (IOException e) {
			logger.error("ファイルの取込に失敗しました", e);
			result.setResultCode("has-error");
			result.getReturnData().put("import_status",
					ImportStats.END_SOFT_ERROR);
			return result;
		}

		// 結果をセット
		if (errorCount == 0) {
			result.setResultCode("success");
			result.getReturnData().put("import_status", ImportStats.END_OK);
		} else {
			result.setResultCode("has-error");
			result.getReturnData().put("import_status",
					ImportStats.END_SOFT_ERROR);
		}

		return result;
	}

	/**
	 * データを更新
	 * 
	 * @param tm
	 * @param sfQuery
	 * @param updateBuffer
	 * @return
	 */
	protected List<Map<String, Object>> executeBulkUpdateQuery(	TransactionManager tm,
														SqlSelectQuery sfQuery,
														List<Map<String, Object>> updateBuffer) {
		// 行データの内容をSQLパラメータとしてクエリにセット
		sfQuery.addParameter("scheduleId", String.class, getScheduleId());
		sfQuery.addParameter("fieldList", List.class, updateBuffer);
		sfQuery.prepare();

		// 更新クエリ実行
		List<Map<String, Object>> sqlResult = tm.requiresNew(() -> {
			SelectCommand<List<Map<String, Object>>> command = new SelectCommand<>(
					sfQuery, new MapResultListHandler(MapKeyNamingType.NONE));
			return command.execute();
		});

		return sqlResult;
	}

	/**
	 * ジョブテーブルの進捗率を更新する
	 *
	 * @param factory
	 * @param sql
	 * @param dto
	 * @param progress
	 * @param tm
	 * @throws PhotonModuleException
	 */
	protected void updateProgress(	SqlUpdateQueryFactory factory,
									String sql,
									ActionDto dto,
									int progress,
									TransactionManager tm)
			throws PhotonModuleException {

		Map<String, Object> srcParams = new HashMap<>();
		PhotonSqlUpdateQuery updateQuery = factory.createUpdateQuery(srcParams,
				dto, sql);

		updateQuery.addParameter("status", String.class, "1");
		updateQuery.addParameter("isEnd", String.class, "0");
		updateQuery.addParameter("progress", String.class,
				StringUtil.defaultString(progress, ""));
		updateQuery.addParameter("tenantId", String.class, dto.getTenantId());
		updateQuery.addParameter("scheduleId", String.class, getScheduleId());

		updateQuery.prepare();

		// 更新クエリ実行
		int sqlResult = tm.requiresNew(() -> {
			PhotonUpdateCommand command = new PhotonUpdateCommand(updateQuery);
			return command.execute();
		});
	}

	/**
	 * 更新用クエリオブジェクトを生成
	 *
	 * @param context
	 * @return
	 * @throws PhotonModuleException
	 */
	protected SqlSelectQuery createQuery(ModuleContext context)
			throws PhotonModuleException {

		Map<String, Object> params = createSqlParams(
				getSqlParamMap().getParamMap(), context);

		// SQLを読み込んでDomaのSQLParserで解析
		SqlParser parser = createSqlParser();

		SqlSelectQuery query = new SqlSelectQuery();

		query.setConfig(DomaConfig.singleton());
		query.setCallerClassName(getClass().getName());
		query.setCallerMethodName("execute");
		query.setSqlLogType(SqlLogType.FORMATTED);
		query.setSqlNode(parser.parse());

		// パラメータをクエリオブジェクトにセットする
		for (Map.Entry<String, Object> entry : params.entrySet()) {
			if (entry.getValue() == null) {
				query.addParameter(entry.getKey(), String.class, "");
			} else if (entry.getValue() instanceof Map) {
				query.addParameter(entry.getKey(), Map.class, entry.getValue());
			} else if (entry.getValue() instanceof List) {
				query.addParameter(entry.getKey(), List.class,
						entry.getValue());
			} else {
				query.addParameter(entry.getKey(), String.class,
						StringUtil.defaultString(entry.getValue(), ""));
			}
		}
		query.addParameter("_params", Map.class, params);

		return query;
	}

	/**
	 * SQLファイルのパラメータデータを生成する
	 *
	 * @param srcParams
	 * @param context
	 * @return
	 */
	@SuppressWarnings({ "rawtypes" })
	protected Map<String, Object> createSqlParams(	Map<String, Object> srcParams,
													ModuleContext context) {
		// パラメータデータを取得
		Map<String, Object> params = new LinkedHashMap<String, Object>();
		for (Map.Entry<String, Object> param : srcParams.entrySet()) {

			String type = StringUtil
					.defaultString(((Map) param.getValue()).get("type"), "");
			params.put((String) param.getKey(),
					ParamUtil.getParamValueByType(type,
							((Map) param.getValue()).get("val"),
							context.getDto()));
		}

		return params;
	}

	/**
	 * SQLファイルを解析してSQLパーサーオブジェクトを生成
	 *
	 * @return
	 * @throws PhotonModuleException
	 */
	protected SqlParser createSqlParser() throws PhotonModuleException {
		// SQLファイル存在チェック
		Path path = Paths.get(getSqlFileDirPath(), getSqlFilePath());
		if (!Files.exists(path)) {
			throw new PhotonModuleException("SQLファイルが存在しません:" + path.toString(),
					null);
		}

		// SQLを読み込んでDomaのSQLParserで解析
		SqlParser parser;
		try {
			parser = new SqlParser(
					Files.readAllLines(path, Charset.forName("UTF-8")).stream()
							.collect(Collectors.joining("\n")));
		} catch (IOException e) {
			throw new PhotonModuleException("SQLの解析に失敗しました:" + path.toString(),
					e);
		}

		return parser;
	}

	/**
	 * CSV行データをカラムをキーとしたマップに変換
	 *
	 * @param line
	 * @param columnSettingList
	 * @return
	 */
	protected Map<String, Object> convertLineToMap(	String line,
													List<String> headerColumnList,
													Map<String, Object> csvFieldMap) {

		Map<String, Object> dataMap = new LinkedHashMap<>();

		// CSVフィールド設定を元に空文字データを格納しておく。
		for (Map.Entry<String, Object> entry : csvFieldMap.entrySet()) {
			dataMap.put(entry.getKey(), "");
		}

		List<String> chunks = csvLineToList(line);
		int chunksSize = chunks.size();
		int index = 0;
		Map<String, Object> csvField = null;
		for (String columnName : headerColumnList) {
			if (csvFieldMap.containsKey(columnName)) {
				csvField = (Map<String, Object>) csvFieldMap.get(columnName);
				// インポート時はCSVカラムの桁数考慮しないが、考慮する場合はここで切る（csvField.field_lengthを使用）
				if (chunksSize < index + 1) {
					dataMap.put(columnName, "");
				} else {
					dataMap.put(columnName,
							StringUtil.defaultString(chunks.get(index), ""));
				}
			}
			index++;
		}

		return dataMap;
	}

	/**
	 * 1データが全て読み込まれたかチェック。 （CSVが複数行にまたがっている場合の考慮）
	 *
	 * @param line
	 * @return
	 */
	protected boolean isEndOfLine(String line) {
		String s = StringUtil.defaultString(line, "");
		if (s == "") {
			return true;
		}

		if (StringUtil.defaultString(getEncloser(), "").length() == 0) {
			return true;
		}

		char t = getEncloser().charAt(0);
		int count = 0;
		for (char x : s.toCharArray()) {
			if (x == t) {
				count++;
			}
		}

		return count % 2 == 1 ? false : true;
	}

	/**
	 * フォーム変換設定に基づいて行データを変換
	 *
	 * @param dataMap
	 * @return
	 */
	protected Map<String, Object> convertCsv2Form(	Map<String, Object> dataMap,
													List<Map<String, Object>> convertSettings,
													Map<String, Object> formFieldMap) {

		Map<String, Map<String, Object>> convertedMapMap = CsvConverter
				.convertCsv2Form(dataMap, formFieldMap, convertSettings);

		Map<String, Object> newDataMap = new HashMap<>();
		Map<String, Object> convertedMap = null;
		for (Map.Entry<String, Map<String, Object>> entry : convertedMapMap
				.entrySet()) {
			convertedMap = entry.getValue();
			newDataMap.put(entry.getKey(), convertedMap.get("value"));
		}

		return newDataMap;
	}

	/**
	 * CSV行データをListに変換
	 *
	 * @param line
	 * @return
	 */
	protected List<String> csvLineToList(String line) {
		String unescaped = StringUtil.excludeUTF8Bom(line);
		List<String> dataList = CsvUtil.csvTextToList(getDelimiter(),
				getEncloser(), unescaped);
		return dataList;
	}

	/**
	 * 改行コードを返す
	 *
	 * @return
	 */
	protected String getLineBreakString() {
		if (getLineBreak().equals("CRLF")) {
			return "\r\n";
		} else if (getLineBreak().equals("CR")) {
			return "\r";
		} else {
			return "\n";
		}
	}

	/**
	 * エラーログを保存
	 *
	 * @param errors
	 */
	protected void writeErrorLogs(	TransactionManager tm,
									int lineIndex,
									Map<String, Object> errors,
									ActionDto dto) {
		List<ErrorLog> errorLogs = new ArrayList<>();

		ErrorLog log = null;
		List<ActionDtoMessage> errorList = null;
		String messageText = null;
		for (Map.Entry<String, Object> entry : errors.entrySet()) {
			errorList = (List<ActionDtoMessage>) entry.getValue();
			for (ActionDtoMessage message : errorList) {

				if (message.getMessage() != null) {
					messageText = message.getMessage();
				} else {
					messageText = message.getMessageId();
				}

				log = new ErrorLog();
				log.setErrorTypeCd("0");
				log.setTenantId(dto.getTenantId());
				log.setAppId(dto.getAppId());
				log.setActionId(dto.getActionId());
				log.setScheduleId(getScheduleId());
				log.setRowNo(lineIndex + 1);
				log.setKeyName(entry.getKey());
				log.setMesg(messageText);
				log.setStrErrorInfo("{}");
				errorLogs.add(log);
			}
		}

		MetaObjectDao dao = new MetaObjectDaoImpl();
		tm.requiresNew(() -> {
			return dao.insertErrorLogs(errorLogs);
		});
	}

	public String getSqlFileDirPath() {
		if (sqlFileDirPath == null) {
			sqlFileDirPath = "";
		}
		return sqlFileDirPath;
	}

	public void setSqlFileDirPath(String sqlFileDirPath) {
		this.sqlFileDirPath = sqlFileDirPath;
	}

	public String getSqlFilePath() {
		return sqlFilePath;
	}

	public void setSqlFilePath(String sqlFilePath) {
		this.sqlFilePath = sqlFilePath;
	}

	public String getCsvImportDirPath() {
		if (csvImportDirPath == null) {
			csvImportDirPath = "";
		}
		return csvImportDirPath;
	}

	public void setCsvImportDirPath(String csvImportDirPath) {
		this.csvImportDirPath = csvImportDirPath;
	}

	public String getCsvImportFilePath() {
		return csvImportFilePath;
	}

	public void setCsvImportFilePath(String csvImportFilePath) {
		this.csvImportFilePath = csvImportFilePath;
	}

	public String getEncoding() {
		return encoding;
	}

	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	public String getLineBreak() {
		return lineBreak;
	}

	public void setLineBreak(String lineBreak) {
		this.lineBreak = lineBreak;
	}

	public String getDelimiter() {
		return delimiter;
	}

	public void setDelimiter(String delimiter) {
		this.delimiter = delimiter;
	}

	public String getEncloser() {
		return encloser;
	}

	public void setEncloser(String encloser) {
		this.encloser = encloser;
	}

	public MapParam getSqlParamMap() {
		return sqlParamMap;
	}

	public String getProgressSqlFilePath() {
		return progressSqlFilePath;
	}

	public void setProgressSqlFilePath(String progressSqlFilePath) {
		this.progressSqlFilePath = progressSqlFilePath;
	}

	public String getScheduleId() {
		return scheduleId;
	}

	public void setScheduleId(String scheduleId) {
		this.scheduleId = scheduleId;
	}

	public void setSqlParamMap(MapParam sqlParamMap) {
		this.sqlParamMap = sqlParamMap;
	}

	public MapListParam getCsvFieldList() {
		return csvFieldList;
	}

	public void setCsvFieldList(MapListParam csvFieldList) {
		this.csvFieldList = csvFieldList;
	}

	public MapListParam getFormFieldList() {
		return formFieldList;
	}

	public void setFormFieldList(MapListParam formFieldList) {
		this.formFieldList = formFieldList;
	}

	public MapListParam getConvertSettings() {
		return convertSettings;
	}

	public void setConvertSettings(MapListParam convertSettings) {
		this.convertSettings = convertSettings;
	}

	public MapListParam getDeleteFormFieldList() {
		return deleteFormFieldList;
	}

	public void setDeleteFormFieldList(MapListParam deleteFormFieldList) {
		this.deleteFormFieldList = deleteFormFieldList;
	}

	public String getBulkNum() {
		return bulkNum;
	}

	public void setBulkNum(String bulkNum) {
		this.bulkNum = bulkNum;
	}

	public String getIsBulkUpdatable() {
		if (isBulkUpdatable == null || isBulkUpdatable.isEmpty()) {
			isBulkUpdatable = "0";
		}
		return isBulkUpdatable;
	}

	public void setIsBulkUpdatable(String isBulkUpdatable) {
		this.isBulkUpdatable = isBulkUpdatable;
	}
}
