package jp.ill.photon.module.common;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.castor.core.util.StringUtil;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.annotation.ModuleReturn;
import jp.ill.photon.annotation.ModuleVersion;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.EscapeChars;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;

/**
 * [variable] 複数の値をDTOにまとめて設定するモジュール.
 *
 * <p>
 * DTO、パラメータ、固定値等のデータを1まとめにしてDTOに設定する。<br/>
 * 複数の値をフラットにマップとしてDTOに追加するか、JSONでマップのレイアウトを指定して追加するかを選択することができる。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 *
 * <h3>JSONレイアウトを指定した場合</h3>
 * <dl>
 * <dt>json</dt>
 * <dd>jsonLayoutパラメータに従って構成されたMap/Listオブジェクト</dd>
 * <dt>json_str</dt>
 * <dd>jsonをJSON文字列化したStringオブジェクト</dd>
 * </dl>
 *
 * <h3>JSONレイアウトを指定しない場合</h3>
 * <dl>
 * <dt>モジュールパラメータvariablesで指定したキー</dt>
 * <dd>モジュールパラメータvariablesで指定した値の実値</dd>
 * </dl>
 *
 * </p>
 *
 * @author h_ota
 *
 */
@ModuleVersion("1.0.0")
public class VariableModule implements PhotonModule {

	/**
	 * まとめたい値の指定用モジュールパラメータ.
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 *
	 * <pre>
	 * {
	 *   "tenantId": {
	 *     "type": "param",
	 *     "val": "_init.tenant_id"
	 *   },
	 *   "formCd": {
	 *     "type": "static",
	 *     "val": "test_val"
	 *   }
	 * }
	 * </pre>
	 *
	 * </dd>
	 * </dl>
	 * </p>
	 *
	 */
	@ModuleParam(required = true)
	@ModuleReturn()
	private Map<String, Object> variables;

	/**
	 * 返却データのレイアウト指定用モジュールパラメータ.
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static</dd>
	 * <dt>transfer_val</dt>
	 * <dd>{\"param1\": {\"tenant_id\": tenant_id}, \"param2\": {\"form_cd\":
	 * formCd}}</dd>
	 * </dl>
	 *
	 * variablesパラメータのキーと一致する箇所がvariablesパラメータの値に置換される。
	 * </p>
	 *
	 */
	@ModuleParam(required = false)
	private String jsonLayout;

	/**
	 * 動的後処理設定用モジュールパラメータ.
	 *
	 * <p>
	 * モジュール実行後処理を動的に指定したい場合に設定する。<br/>
	 * 後処理を設定したい結果コード毎に後処理パラメータを設定する。
	 * </p>
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 *
	 * <pre>
	 * {
	 *   "validate_ng": {
	 *     "result_type": "forward",
	 *     "next_path": "/user/shohin_list",
	 *     "params": {
	 *       "redisplay_flg": {
	 *         "transfer_type": "static",
	 *         "transfer_val": "1"
	 *       }
	 *     }
	 *   }
	 * }
	 * </pre>
	 *
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = false)
	private Map<String, Object> after;

	@Override
	public ModuleResult execute(ModuleContext context) {
		ModuleResult result = new ModuleResult();



		// jsonLayoutに値が設定されていない：各variablesの値を評価し、それを戻り値とする
		// jsonLayoutに値が設定されている：各variablesの値を評価し、jsonLayout内のvariablesのキーを置換した後JsonUtilで変換

		if (StringUtils.isEmpty(jsonLayout)) {

			// パラメータをDTOのパラメータとしてセットし直す
			for (String key : variables.keySet()) {
				Object obj = ParamUtil.getParamValueByType(
						String.valueOf(((Map) variables.get(key))
								.get("transfer_type")),
						((Map) variables.get(key)).get("transfer_val"),
						context.getDto());
				result.getReturnData().put(key, obj);
			}

		} else {

			String ret = jsonLayout;

			for (String key : variables.keySet()) {

				Object obj = ParamUtil.getParamValueByType(
						String.valueOf(((Map) variables.get(key))
								.get("transfer_type")),
						((Map) variables.get(key)).get("transfer_val"),
						context.getDto());

				String val = "";
				if (obj != null) {

					if (obj instanceof List) {

						val = JsonUtil.mapToJson(obj);

					} else if (obj instanceof Map) {

						val = JsonUtil.mapToJson(obj);

					} else if (obj instanceof String[]) {

						val = JsonUtil.mapToJson(Arrays.asList((String[]) obj));

					} else {

						//val = val.replaceAll("\"", "\\\\\"");
						val = EscapeChars.forJSON(obj.toString());

					}

				} else {

					val = "null";

				}

				ret = StringUtil.replaceAll(ret, key, val);

			}

			Object setValue = null;
			if (jsonLayout.startsWith("[")) {
				setValue = JsonUtil.jsonToList(ret);
			} else if (jsonLayout.startsWith("{")) {
				setValue = JsonUtil.jsonToMap(ret);
			}

			result.getReturnData().put("json", setValue);
			result.getReturnData().put("json_str", ret);

		}

		result.setResultCode("set");

		// モジュールのパラメータ"after"に値がセットされていたとき
		if (after != null) {
			Map<String, Object> mp = (Map<String, Object>) after
					.get(result.getResultCode());
			if (mp != null) {
				result.getReturnData().put("after_params", mp.get("params"));
				result.setNextPath((String) mp.get("next_path"));
				result.setResultType((String) mp.get("result_type"));
			}
		}

		return result;
	}

	/**
	 * variablesを取得します。
	 *
	 * @return variables
	 */
	public Map<String, Object> getVariables() {
		return variables;
	}

	/**
	 * variablesを設定します。
	 *
	 * @param variables
	 */
	public void setVariables(Map<String, Object> variables) {
		this.variables = variables;
	}

	/**
	 * jsonLayoutを取得します。
	 *
	 * @return jsonLayout
	 */
	public String getJsonLayout() {
		return jsonLayout;
	}

	/**
	 * jsonLayoutを設定します。
	 *
	 * @param jsonLayout
	 */
	public void setJsonLayout(String jsonLayout) {
		this.jsonLayout = jsonLayout;
	}

	/**
	 * afterを取得します。
	 *
	 * @return after
	 */
	public Map<String, Object> getAfter() {
		return after;
	}

	/**
	 * afterを設定します。
	 *
	 * @param after
	 */
	public void setAfter(Map<String, Object> after) {
		this.after = after;
	}

}
