package jp.ill.photon.module.common;

import java.util.HashMap;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.MapParam;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ParamUtil;

/**
 * [transfer_param_convert] タイプと値のパラメータマップを実値のマップに変換するモジュール
 * 
 * <p>
 * パラメータとして受け取った値（param,static,dto）を実値に変換する。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>converted</dt>
 * <dd>param_mapに渡されたパラメータ値の実値</dd>
 * </dl>
 * </p>
 * 
 * @author h_tanaka
 *
 */
public class TransferParamConvertModule implements PhotonModule {

	/**
	 * パラメータマップ用モジュールパラメータ.
	 * 
	 */
	@ModuleParam(required = false)
	private MapParam paramMap;

	/**
	 * パラメータタイプキー指定用モジュールパラメータ.
	 * 
	 * <p>
	 * パラメータのタイプを指定するキーを指定することができる。<br/>
	 * 指定されない場合は、「type」をキーとして使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String typeKeyName;

	/**
	 * パラメータ値キー指定用モジュールパラメータ.
	 * 
	 * <p>
	 * パラメータの値を指定するキーを指定することができる。<br/>
	 * 指定されない場合は、「val」をキーとして使用する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String valueKeyName;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {
		ModuleResult result = new ModuleResult();
		Map<String, Object> convertedParams = new HashMap<>();

		if (getParamMap() == null) {
			result.getReturnData().put("converted", convertedParams);
			return result;
		}

		if (getParamMap().getParamMap() == null) {
			result.getReturnData().put("converted", convertedParams);
			return result;
		}

		Map<String, Object> paramSettings = getParamMap().getParamMap();

		for (Map.Entry<String, Object> paramSetting : paramSettings
				.entrySet()) {
			convertedParams.put((String) paramSetting.getKey(),
					ParamUtil.getParamValueByType(
							String.valueOf(((Map) paramSetting.getValue())
									.get(getTypeKeyName())),
							((Map) paramSetting.getValue())
									.get(getValueKeyName()),
							context.getDto()));
		}

		result.getReturnData().put("converted", convertedParams);
		return result;
	}

	public MapParam getParamMap() {
		return paramMap;
	}

	public void setParamMap(MapParam paramMap) {
		this.paramMap = paramMap;
	}

	public String getValueKeyName() {
		if (valueKeyName == null || valueKeyName.isEmpty()) {
			valueKeyName = "val";
		}
		return valueKeyName;
	}

	public void setValueKeyName(String valueKeyName) {
		this.valueKeyName = valueKeyName;
	}

	public String getTypeKeyName() {
		if (typeKeyName == null || typeKeyName.isEmpty()) {
			typeKeyName = "type";
		}
		return typeKeyName;
	}

	public void setTypeKeyName(String typeKeyName) {
		this.typeKeyName = typeKeyName;
	}

}
