package jp.ill.photon.module.common;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ParamUtil;

public class ReplaceStringModule implements PhotonModule {

	@ModuleParam(required = true)
	private Map<String, Object> input;

	private ActionDto dto;

	private final static String MARKER_FRONT = "%%%";
	private final static String MARKER_REAR = "%%%";

	private final static class InputParams {
		private final static String BASE = "base";
		private final static String REPLACE = "replace";
	}
	private final static class ReplaceParams {
		private final static String TYPE = "type";
		private final static String VAL = "val";
	}
	private final static class ReplaceType {
		private final static String DTO = "dto";
		private final static String DTO_REF = "dto_ref";
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		dto = context.getDto();

		ModuleResult result = new ModuleResult();
		Map<String, Object> valMap = new LinkedHashMap<String, Object>();

		for (Map.Entry<String, Object> e : input.entrySet()) {

			Object value = getReplacedObj((Map)e.getValue());
			valMap.put(e.getKey(), value);

		}

		result.getReturnData().put("returns", valMap);

		return result;

	}


	/**
	 * 置換されたデータを返却
	 *
	 * @param 設定情報
	 * @return 置換されたデータ(設定情報の"base"の形で返却される)
	 *
	 * */
	@SuppressWarnings({ "unchecked" })
	private Object getReplacedObj(Map<String, Object> map) {

		List<String> replace = makeReplaceList((List<Map<String, String>>)map.get(InputParams.REPLACE));
		return replaceProc(map.get(InputParams.BASE), replace);

	}

	/**
	 * 置換リストを返却
	 *
	 * @param 設定情報
	 * @return 置換リスト
	 *
	 * */
	private List<String> makeReplaceList(List<Map<String, String>> list) {

		List<String> ret = new ArrayList<String>();

		for (Map<String, String> item : list) {

			// "dto_ref"のときは、"val"プロパティから取得された値(文字列を想定)
			// で、もう一度dto内を走査する
			String type = (String)item.get(ReplaceParams.TYPE);
			boolean isRef = ReplaceType.DTO_REF.equals(type);

			if (isRef) {
				type = ReplaceType.DTO;
			}

			String val = (String)ParamUtil.getParamValueByType(
										type,
										item.get(ReplaceParams.VAL),
										dto);

			if (isRef) {
				if (val != null) {
					val = (String)ParamUtil.getParamValueByType(
							ReplaceType.DTO,
							val,
							dto);
				}
			}

			if (val != null) {
				ret.add(val);
			}
		}

		return ret;

	}

	/**
	 * 置換処理
	 *
	 * @param 置換していないオブジェクト
	 * @param 設定情報
	 * @return 置換済みオブジェクト
	 *
	 * */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private Object replaceProc(Object src, List<String> replace) {

		if (src instanceof List){

			List list = (List)src;
			List ret = new ArrayList<Object>();
			for (Object o : list) {
				ret.add(replaceProc(o, replace));
			}

			return ret;

		} else if (src instanceof Map){

			Map<String, Object> map = (Map<String, Object>)src;
			Map<String, Object> ret = new LinkedHashMap<String, Object>();
			for (Map.Entry<String, Object> e : map.entrySet()) {
				String key = e.getKey();
				key = replaceAllMarker(key, replace);
				ret.put(key, replaceProc(e.getValue(), replace));
			}

			return ret;

		} else {

			return replaceAllMarker((String)src, replace);

		}

	}

	/**
	 * 置換処理
	 *
	 * @param 置換していない文字列
	 * @param 設定情報
	 * @return 置換済み文字列
	 *
	 * */
	private String replaceAllMarker(String src, List<String> replace) {

		String ret = src;
		for (int i = 0; i < replace.size(); i++) {
			ret = ret.replaceAll(MARKER_FRONT + Integer.toString(i+1) + MARKER_REAR, replace.get(i));
		}

		return ret;

	}

	/**
	 * inputを取得します。
	 * @return input
	 */
	public Map<String, Object> getInput() {
		return input;
	}

	/**
	 * inputを設定します。
	 * @param input
	 */
	public void setInput(Map<String, Object> input) {
		this.input = input;
	}


}
