package jp.ill.photon.module.common;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.model.MapParam;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ParamUtil;

/**
 * [regex_replace] 指定されたデータの内容を置換するモジュール.
 *
 * <p>
 * テキストを正規表現置換する。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>replaced</dt>
 * <dd>置換後のテキスト</dd>
 * </dl>
 * </p>
 *
 * @author h_tanaka
 *
 */
public class RegexReplaceModule implements PhotonModule {

	/**
	 * 変換元となるソーステキスト用モジュールパラメータ.
	 */
	@ModuleParam(required = false)
	private String srcText;

	/**
	 * 検索するパターン文字列用モジュールパラメータ.
	 *
	 * <p>
	 * 置換対象が1つだけの場合に使用する。<br/>
	 * replaceMode=noregexの場合は文字列として置換し、<br/>
	 * それ以外の場合は正規表現パターンとして置換する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String searchPattern;

	/**
	 * 置換テキスト用モジュールパラメータ.
	 *
	 * <p>
	 * 置換対象が1つだけの場合に使用する。<br/>
	 * searchPatternでマッチした箇所を置換する文字列を指定する。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String replaceText;

	/**
	 * 複数置換用モジュールパラメータ.
	 *
	 * <p>
	 * 置換対象を複数指定したい場合に使用する。<br/>
	 * replaceMode=noregexの場合は文字列として置換し、<br/>
	 * それ以外の場合は正規表現パターンとして置換する。
	 * </p>
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 *
	 * <pre>
	 * {
	 *   "#timestamp#": {
	 *     "type": "static",
	 *     "val": "20170614000000"
	 *   },
	 *   "#filename#": {
	 *     "type": "static",
	 *     "val": "test.txt"
	 *   }
	 * }
	 * </pre>
	 *
	 * 上記の内容を指定すると、srcText内の「#timestamp#」が「20170614000000」に置換され、<br/>
	 * 「#filename#」が「test.txt」に置換される。</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = false)
	private MapParam replacePatternSet;

	/**
	 * 置換モード指定用モジュールパラメータ.
	 *
	 * <p>
	 * 「noregex」を指定した場合は「文字列置換モード」となり、<br/>
	 * それ以外の場合は「正規表現置換モード」となる。
	 * </p>
	 */
	@ModuleParam(required = false)
	private String replaceMode;

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException {

		ModuleResult result = new ModuleResult();

		String resultText = getSrcText();
		if (StringUtils.isEmpty(resultText)) {
			result.getReturnData().put("replaced", resultText);
			return result;
		}

		String replaceMode = getReplaceMode();
		if (replaceMode != null && replaceMode.equals("noregex")) {

			if (!StringUtils.isEmpty(getSearchPattern())
					&& !StringUtils.isEmpty(getReplaceText())) {
				resultText = resultText.replace(getSearchPattern(),
						getReplaceText());
			}

			if (getReplacePatternSet() != null) {
				Map<String, Object> patterns = createReplacePatterns(
						getReplacePatternSet().getParamMap(), context);
				for (Map.Entry<String, Object> entry : patterns.entrySet()) {
					resultText = resultText.replace(entry.getKey(),
							String.valueOf(entry.getValue()));
				}
			}
		} else {
			if (!StringUtils.isEmpty(getSearchPattern())
					&& !StringUtils.isEmpty(getReplaceText())) {
				Pattern p = Pattern.compile(getSearchPattern());
				Matcher m = p.matcher(resultText);
				resultText = m.replaceAll(getReplaceText());
			}

			if (getReplacePatternSet() != null) {
				Map<String, Object> patterns = createReplacePatterns(
						getReplacePatternSet().getParamMap(), context);
				Pattern p = null;
				Matcher m = null;
				for (Map.Entry<String, Object> entry : patterns.entrySet()) {
					p = Pattern.compile(entry.getKey());
					m = p.matcher(resultText);
					resultText = m.replaceAll(String.valueOf(entry.getValue()));
				}
			}
		}

		// 置換後テキストを返す
		result.getReturnData().put("replaced", resultText);
		return result;
	}

	/**
	 * 置換セットのパラメータデータを生成する
	 *
	 * @param srcParams
	 * @param context
	 * @return
	 */
	@SuppressWarnings({ "rawtypes" })
	protected Map<String, Object> createReplacePatterns(Map<String, Object> srcParams,
														ModuleContext context) {
		// パラメータデータを取得
		Map<String, Object> params = new LinkedHashMap<String, Object>();
		for (Map.Entry<String, Object> param : srcParams.entrySet()) {

			String type = String.valueOf(((Map) param.getValue()).get("type"));
			params.put((String) param.getKey(),
					ParamUtil.getParamValueByType(type,
							((Map) param.getValue()).get("val"),
							context.getDto()));
		}

		return params;
	}

	public String getSrcText() {
		return srcText;
	}

	public void setSrcText(String srcText) {
		this.srcText = srcText;
	}

	public String getSearchPattern() {
		return searchPattern;
	}

	public void setSearchPattern(String searchPattern) {
		this.searchPattern = searchPattern;
	}

	public String getReplaceText() {
		if (replaceText == null) {
			replaceText = "";
		}
		return replaceText;
	}

	public void setReplaceText(String replaceText) {
		this.replaceText = replaceText;
	}

	public String getReplaceMode() {
		return replaceMode;
	}

	public void setReplaceMode(String replaceMode) {
		this.replaceMode = replaceMode;
	}

	public MapParam getReplacePatternSet() {
		return replacePatternSet;
	}

	public void setReplacePatternSet(MapParam replacePatternSet) {
		this.replacePatternSet = replacePatternSet;
	}

}
