package jp.ill.photon.module.common;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.annotation.ModuleVersion;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;

/**
 * [pager] 汎用ページャーを作成するモジュール
 * 
 * <p>
 * Doma用SQLテンプレートファイルのパスと、そのSQL内で使用したいパラメータを渡すと、SQLを実行して結果を返す。</br>
 * 実行されるSQLはSelect文である必要はないが、必ず値を返す必要がある。<br/>
 * UPDATE文等を実行する時は、RETURNING句等で値を返す。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>page_list</dt>
 * <dd>ページャーに表示するページ番号のリスト</dd>
 * <dt>page</dt>
 * <dd>現在のページ番号</dd>
 * <dt>totalCount</dt>
 * <dd>データ全件数</dd>
 * <dt>pageMaxCount</dt>
 * <dd>ページあたりの件数</dd>
 * <dt>maxPage</dt>
 * <dd>最終ページ番号</dd>
 * <dt>page_start_no</dt>
 * <dd>ページの開始番号</dd>
 * <dt>page_end_no</dt>
 * <dd>ページの終了番号</dd>
 * </dl>
 * </p>
 *
 * @author h_tanaka
 *
 */
@ModuleVersion("1.0.0")
public class PagerModule implements PhotonModule {

	/**
	 * 現在のページ指定用モジュールパラメータ.
	 * 
	 */
	@ModuleParam(required = false)
	private int page = 1;

	/**
	 * データ全件数指定用モジュールパラメータ.
	 * 
	 */
	@ModuleParam(required = false)
	private int totalCount = 0;

	/**
	 * ページあたりの件数指定用モジュールパラメータ.
	 * 
	 */
	@ModuleParam(required = false)
	private int pageMaxCount = 20;

	public int getPage() {
		return page;
	}

	public void setPage(int page) {
		this.page = page;
	}

	public int getTotalCount() {
		return totalCount;
	}

	public void setTotalCount(int totalCount) {
		this.totalCount = totalCount;
	}

	public int getPageMaxCount() {
		return pageMaxCount;
	}

	public void setPageMaxCount(int pageMaxCount) {
		this.pageMaxCount = pageMaxCount;
	}

	/**
	 * ページ番号、合計件数、ページ毎表示件数を元にページャーデータを作成して返す
	 */
	@Override
	public ModuleResult execute(ModuleContext context) {

		List<String> pageList = getPageList(page, totalCount, pageMaxCount);

		ModuleResult result = new ModuleResult();
		result.getReturnData().put("page_list", pageList);
		result.getReturnData().put("page", page);
		result.getReturnData().put("totalCount", totalCount);
		result.getReturnData().put("pageMaxCount", pageMaxCount);
		result.getReturnData().put("maxPage",
				getMaxPage(totalCount, pageMaxCount));

		Map<String, Integer> pageInfo = getPageStartEndInfo(page, totalCount,
				pageMaxCount);
		result.getReturnData().put("page_start_no", pageInfo.get("start"));
		result.getReturnData().put("page_end_no", pageInfo.get("end"));

		return result;
	}

	/**
	 * 画面表示用ページリストを返却する。<br />
	 * jspからの直接呼出しを想定
	 *
	 * @param ページリスト
	 */
	protected List<String> getPageList(	int page,
										int totalCount,
										int pageMaxCount) {
		// 戻り値格納用
		List<String> list = new ArrayList<String>();

		int endNo = 5;

		// 開始番号
		int start = 1;
		// 終了番号
		int end = 0;

		if (page - 4 > 0) {
			start = page - 4;
		} else {
			endNo += 1 - (page - 4);
		}

		int maxPage = getMaxPage(totalCount, pageMaxCount);
		if (page + endNo >= maxPage) {
			end = (int) Math.ceil(maxPage);
			start = start - (page + endNo - maxPage) <= 0 ? 1
					: start - (page + endNo - maxPage);
		} else {
			end = page + endNo;
		}

		// 開始番号から終了番号までを繰り返す
		for (int i = start; i <= end; i++) {
			list.add(Integer.toString(i));
		}

		return list;

	}

	protected Map<String, Integer> getPageStartEndInfo(	int page,
														int totalCount,
														int pageMaxCount) {
		Map<String, Integer> pageInfo = new HashMap<>();
		if (totalCount == 0) {
			pageInfo.put("start", 0);
		} else {
			pageInfo.put("start", 1);
		}
		pageInfo.put("end", Math.min(pageMaxCount, totalCount));

		if (page > 1) {
			pageInfo.put("start", pageMaxCount * (page - 1) + 1);
			pageInfo.put("end", Math.min(totalCount, pageMaxCount * page));
		}

		return pageInfo;
	}

	/**
	 * 最大ページ番号を取得します。
	 *
	 * @return 最大ページ番号
	 */
	protected int getMaxPage(int totalCount, int pageMaxCount) {
		return (int) Math.ceil((double) totalCount / (double) pageMaxCount);
	}
}
