package jp.ill.photon.module.common;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.action.ActionDispatcher;
import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.annotation.ModuleVersion;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.castor.core.util.StringUtil;

/**
 * パラメータで指定された条件を作り出し、判定するモジュール
 * 返却するReturnDataの値も指定できる。
 *
 * @author m_fukukawa
 *
 */
@ModuleVersion("1.0.0")
public class MultiComparatorModule implements PhotonModule {

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "is_equal")
	private String isEqualRetCode;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "is_not_equal")
	private String isNotEqualRetCode;

	@ModuleParam(required=true)
	private String conditionLayout;

	@ModuleParam(required=true)
	private Map<String, Object> conditionValues;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(ActionDispatcher.class);

	// dto
	private ActionDto dto;

	/* 評価対象文字列格納、走査クラス */
	public class Source {

		private final String str;
		private int pos;

		public Source(String str) {
			this.str = str;
		}

		public final int peek() {
			if (pos < str.length()) {
				return str.charAt(pos);
			}
			return -1;
		}

		public final void next() {
			++pos;
		}

	}

	/* 評価対象文字列パースクラス */
	public class Parser extends Source {

		private static final String specifiedValueChars = "truefals";

		public Parser(String str) {
			super(str);
		}

		/**
		 *
		 * 指定された文字列が評価式に使用可能な文字であるかを返却
		 *
		 * @return 指定された文字列が評価式に使用可能な文字であるか
		 *
		 * */
		public final boolean isSpecifiedChar(int ch) {
			return (specifiedValueChars.indexOf(ch) >= 0);
		}

		/**
		 *
		 * 文字列中から"true"、もしくは"false"の値を取り出す。
		 *
		 * @return 評価結果
		 *
		 * */
		public final boolean bool() throws PhotonModuleException {

			StringBuilder sb = new StringBuilder();

			int ch;

			while ((ch = peek()) >= 0 && isSpecifiedChar(ch)) {
				sb.append((char) ch);
				next();
			}

			// "true"でも"false"でもない文字列ができてしまっていたらException
			if ( !"true".equals(sb.toString().toLowerCase()) &&  !"false".equals(sb.toString().toLowerCase()) ) {
				throw new PhotonModuleException("評価済みの文字列で、\"true\"でも\"false\"でもない文字列が混入しています。", null);
			}

			return Boolean.parseBoolean(sb.toString());
		}

		/**
		 *
		 * メイン処理
		 *
		 * @return 評価結果
		 *
		 * */
		public final boolean expr() throws PhotonModuleException {
			boolean x = factor();
			for (;;) {
				StringBuffer sb = new StringBuffer();
				switch (peek()) {
					case 'a':
						sb.append((char)peek());
						for (int i = 0; i < 2; i++) {
							next();
							sb.append((char)peek());
						}
						// andだったら実行
						if ("and".equals(sb.toString())) {
							next();
							boolean r = factor();
							x = x && r;
						}
						continue;
					case 'o':
						sb.append((char)peek());
						for (int i = 0; i < 1; i++) {
							next();
							sb.append((char)peek());
						}
						// orだったら実行
						if ("or".equals(sb.toString())) {
							next();
							boolean r = factor();
							x = x || r;
						}
						continue;
				}
				break;
			}
			return x;
		}

		/**
		 *
		 * 括弧の処理
		 *
		 * @return 評価結果
		 *
		 * */
		public final boolean factor() throws PhotonModuleException {
			boolean ret;
			spaces();
			if (peek() == '(') {
				next();
				ret = expr();
				if (peek() == ')') {
					next();
				}
			} else {
				ret = bool();
			}
			spaces();
			return ret;
		}

		/**
		 *
		 * スペース読み飛ばし
		 *
		 * */
		public void spaces() {
			while (peek() == ' ') {
				next();
			}
		}

	}

	protected static class SettingValueType {
		public static final String OPERATOR = "operator";
		public static final String VALUE = "value";
	}

	protected static class OperatorType {
		public static final String EQUALS = "==";
		public static final String NOT_EQUALS = "!=";
		public static final String IS_EMPTY = "isempty";
	}

	@Override
	public ModuleResult execute( ModuleContext context ) throws PhotonModuleException {

		dto = context.getDto();

		logger.info(String.format("---->[condition_layout]:[%s]", conditionLayout));

		// conditionValuesをすべて評価
		Map<String, Boolean> cond = evaluateConditionValues();

		// conditionLayoutに評価結果をすべて埋め込み
		String embededCond = embedValuesToLayout(cond);

		logger.info(String.format("---->[embedValuesToLayout]:[%s]", embededCond));

		// 評価
		boolean ret = new Parser(embededCond).expr();

		ModuleResult result = new ModuleResult();

		if (ret) {
			logger.info("---->[is_equals]");
			result.getReturnData().put("result_code", getIsEqualRetCode());
			result.setResultCode("is_equal");
		} else {
			logger.info("---->[is_not_equal]");
			result.getReturnData().put("result_code", getIsNotEqualRetCode());
			result.setResultCode("is_not_equal");
		}

		return result;


	}

	/**
	 *
	 * conditionValuesをすべて評価
	 *
	 * @return 条件ごとの結果マップ
	 *
	 * */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private Map<String, Boolean> evaluateConditionValues() {

		Map<String, Boolean> ret = new HashMap<String, Boolean>();

		// conditionValuesをすべて評価
		for (Map.Entry<String, Object> e : conditionValues.entrySet()) {

			String key = e.getKey();
			List<Map<String, Object>> list = (List<Map<String, Object>>)e.getValue();

			Object src = null;
			boolean srcSet = false;

			String op = null;
			boolean opSet = false;

			Object dst = null;
			boolean dstSet = false;

			for (Map<String, Object> mp :list) {

				String operateType  = (String)mp.get("operate_type");
				Object content  = mp.get("content");

				if (SettingValueType.OPERATOR.equals(operateType)) {

					// "operator"のときは、比較演算子をセット
					if (!opSet) {
						op = (String)content;
						opSet = true;
					}
					// operatorが"isempty"のときは、destの値は無視する。
					if (OperatorType.IS_EMPTY.equals(content)) {
						dstSet = true;
					}

				} else if(SettingValueType.VALUE.equals(operateType)) {

					// "value"のときは、値をセット
					if (!srcSet) {
						src = ParamUtil.getParamObjectValueByType(content, dto);
						srcSet = true;
					} else if (!dstSet) {
						dst = ParamUtil.getParamObjectValueByType(content, dto);
						dstSet = true;
					}

				}

			}

			// src、op、dstがすべてセットされていれば、判定処理を行う
			if (srcSet && opSet && dstSet) {

				if (OperatorType.EQUALS.equals(op)) {

					ret.put(key, java.util.Objects.equals(src, dst));

				} else if (OperatorType.NOT_EQUALS.equals(op)) {

					ret.put(key, !( java.util.Objects.equals(src, dst) ));

				} else if (OperatorType.IS_EMPTY.equals(op)) {

					boolean retVal = true;

					if (src != null) {

						if (src instanceof List) {

							retVal = CollectionUtils.isEmpty((List)src);

						} else if (src instanceof Map) {

							retVal = MapUtils.isEmpty((Map)src);

						} else if (src instanceof String) {

							retVal = StringUtils.isEmpty((String)src);

						}

					}

					ret.put(key, retVal);

				}

			}

		}

		return ret;

	}

	/**
	 *
	 * conditionLayoutに評価結果を埋め込む
	 *
	 * @param values 評価結果
	 * @return 評価結果が埋め込まれた文字列
	 *
	 * */
	private String embedValuesToLayout(Map<String, Boolean> values) {

		String ret = conditionLayout;

		for (Map.Entry<String, Boolean> e : values.entrySet()) {
			ret = StringUtil.replaceAll(ret, e.getKey(), e.getValue().toString());
		}

		return ret;

	}

	/**
	 * isEqualRetCodeを取得します。
	 * @return isEqualRetCode
	 */
	public String getIsEqualRetCode() {
		return isEqualRetCode;
	}

	/**
	 * isEqualRetCodeを設定します。
	 * @param isEqualRetCode
	 */
	public void setIsEqualRetCode(String isEqualRetCode) {
		this.isEqualRetCode = isEqualRetCode;
	}

	/**
	 * isNotEqualRetCodeを取得します。
	 * @return isNotEqualRetCode
	 */
	public String getIsNotEqualRetCode() {
		return isNotEqualRetCode;
	}

	/**
	 * isNotEqualRetCodeを設定します。
	 * @param isNotEqualRetCode
	 */
	public void setIsNotEqualRetCode(String isNotEqualRetCode) {
		this.isNotEqualRetCode = isNotEqualRetCode;
	}

	/**
	 * conditionLayoutを取得します。
	 * @return conditionLayout
	 */
	public String getConditionLayout() {
		return conditionLayout;
	}

	/**
	 * conditionLayoutを設定します。
	 * @param conditionLayout
	 */
	public void setConditionLayout(String conditionLayout) {
		this.conditionLayout = conditionLayout;
	}

	/**
	 * conditionValuesを取得します。
	 * @return conditionValues
	 */
	public Map<String, Object> getConditionValues() {
		return conditionValues;
	}

	/**
	 * conditionValuesを設定します。
	 * @param conditionValues
	 */
	public void setConditionValues(Map<String, Object> conditionValues) {
		this.conditionValues = conditionValues;
	}

}
