package jp.ill.photon.module.common;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.action.ActionDispatcher;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.JudgeExecutableModuleUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.seasar.doma.FetchType;
import org.seasar.doma.internal.jdbc.sql.SqlParser;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.query.SqlSelectQuery;

/**
 *
 * targetListをループして、
 * conditionで指定された条件に合致したもののみを新たなリストとして返却する
 *
 * @author m_fukukawa
 *
 */
public class FilterMapListModule implements PhotonModule {

	@ModuleParam(required = true)
	private String conditionLayout; // 条件文字列、Domaの条件式と同じ書式で記述

	@ModuleParam(required = true)
	private List<Map<String, Object>> targetList; // フィルタ対象リスト（

	@ModuleParam(required = false)
	private Map<String, Object> conditionValue; // 比較対象の値をDTOから持ってきたいときに使用

	// モジュール内で使用する変数
	/** DTO */
	ActionDto dto;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(ActionDispatcher.class);

	/** 1レコード前の要素 */
	private Map<String, Object> before;

	@Override
	public ModuleResult execute( ModuleContext context ) {

		dto = context.getDto();

		List<Map<String, Object>> ret = new ArrayList<Map<String, Object>>();

		logger.info("入力リスト：" + targetList);

		if (!CollectionUtils.isEmpty(targetList)) {

			for (Map<String, Object> mp : targetList) {

				if (isMatchCondition(mp)) {
					ret.add(new HashMap<String, Object>(mp));
				}

				before = new HashMap<String, Object>(mp);

			}

		}

		ModuleResult result = new ModuleResult();
		int cnt = 0;
		if (!ret.isEmpty()) {
			if (ret.get(0).get("count") != null) {
				cnt = Integer.parseInt(ret.get(0).get("count").toString());
			} else {
				cnt = ret.size();
			}
		}

		logger.info("結果リスト：" + ret);

		result.getReturnData().put("first", (ret.isEmpty())? null : ret.get(0));
		result.getReturnData().put("list", ret);
		result.getReturnData().put("cnt", cnt);

		return result;

	}

	/**
	 *
	 * 条件に合致しているか？
	 *
	 * */
	public boolean isMatchCondition(Map<String, Object> mp) {

		// SQLを読み込んでDomaのSQLParserで解析
		SqlParser sp = new SqlParser("/*%if " + conditionLayout + "  */ true /*%else*/ false /*%end*/");
		SqlSelectQuery sq = new SqlSelectQuery();
		sq.setConfig(DomaConfig.singleton());
		sq.setCallerClassName(JudgeExecutableModuleUtil.class.getName());
		sq.setCallerMethodName("isExecutable");
		sq.setFetchType(FetchType.LAZY);
		sq.setSqlLogType(SqlLogType.FORMATTED);
		sq.setSqlNode(sp.parse());

		// パラメータ設定
		sq.addParameter("before", Map.class, before);
		sq.addParameter("current", Map.class, mp);
		if (!MapUtils.isEmpty(conditionValue)) {
			for (Map.Entry<String, Object> e : conditionValue.entrySet()) {
				Object val = ParamUtil.getParamObjectValueByType(e.getValue(), dto);
				sq.addParameter(e.getKey(), val.getClass(), val);
			}
		}

		try {
			sq.prepare();
		} catch (Exception e) {
			logger.info(String.format("---->[error]:[%s]", e.getMessage()));
			return false;
		}

		return Boolean.parseBoolean(sq.getSql().toString());

	}

	/**
	 * conditionLayoutを取得します。
	 * @return conditionLayout
	 */
	public String getConditionLayout() {
		return conditionLayout;
	}

	/**
	 * conditionLayoutを設定します。
	 * @param conditionLayout
	 */
	public void setConditionLayout(String conditionLayout) {
		this.conditionLayout = conditionLayout;
	}

	/**
	 * targetListを取得します。
	 * @return targetList
	 */
	public List<Map<String, Object>> getTargetList() {
		return targetList;
	}

	/**
	 * targetListを設定します。
	 * @param targetList
	 */
	public void setTargetList(List<Map<String, Object>> targetList) {
		this.targetList = targetList;
	}

	/**
	 * conditionValueを取得します。
	 * @return conditionValue
	 */
	public Map<String, Object> getConditionValue() {
		return conditionValue;
	}

	/**
	 * conditionValueを設定します。
	 * @param conditionValue
	 */
	public void setConditionValue(Map<String, Object> conditionValue) {
		this.conditionValue = conditionValue;
	}


}
