package jp.ill.photon.module.common;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpSession;

import org.apache.commons.lang3.StringUtils;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.annotation.ModuleVersion;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.ParamUtil;

/**
 *
 * [add_message] クライアント出力用メッセージを追加するモジュール.
 *
 * <p>
 * パラメータで指定された内容でアクションが出力するメッセージを追加する。
 * </p>
 *
 * <p>
 * <h2>DTOへの設定値：</h2>
 * <dl>
 * <dt>なし</dt>
 * <dd>メッセージ追加のみ</dd>
 * </dl>
 * </p>
 *
 * @author m_fukukawa
 *
 */
@ModuleVersion("1.0.0")
public class AddMessageModule implements PhotonModule {

	/**
	 * メッセージ内容指定用モジュールパラメータ.
	 *
	 * <p>
	 * 追加するメッセージ情報のリストを受け取る。
	 * </p>
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 *
	 * <pre>
	 * [
	 *   {
	 *     "message_id": {
	 *       "type": "static"
	 *       "val": "doneInsert"
	 *     },
	 *     "message_type": {
	 *       "type": "static",
	 *       "val": "info_field"
	 *     },
	 *     "form_field_cd": {
	 *       "type": "static",
	 *       "val": "mail"
	 *     },
	 *     "message_params": {
	 *       "type": "static_json",
	 *       "val": {
	 *         "%1$s": {
	 *           "type": "static",
	 *           "val": "メールアドレス"
	 *         }
	 *       }
	 *     }
	 *   }
	 * ]
	 * </pre>
	 *
	 * </dl>
	 *
	 * 上記の例では、フィールド「mail」のメッセージタイプ「info_field」に対して、<br/>
	 * メッセージID「doneInsert」をテンプレートとして<br/>
	 * 「%1$s」を「商品」という文字列で置換したメッセージを追加している。
	 * </p>
	 *
	 * <p>
	 * message_idはmesgエンティティに登録されているデータを指定する。<br/>
	 * message_typeは下記の4種類から選択可能：
	 *
	 * <dl>
	 * <dt>info_page</dt>
	 * <dd>ページ上部表示用通常メッセージ</dd>
	 * <dt>error_page</dt>
	 * <dd>ページ上部表示用エラーメッセージ</dd>
	 * <dt>info_field</dt>
	 * <dd>指定フォームフィールド用通常メッセージ</dd>
	 * <dt>error_field</dt>
	 * <dd>指定フォームフィールド用エラーメッセージ</dd>
	 * </dl>
	 * </p>
	 *
	 * パラメータ「form_field_cd」はフォームフィールド用メッセージ（info_field、error_field）でのみ必要となる。
	 */
	@ModuleParam(required = true)
	private List<Map<String, Object>> addMessages;

	/**
	 * 動的後処理設定用モジュールパラメータ.
	 *
	 * <p>
	 * モジュール実行後処理を動的に指定したい場合に設定する。<br/>
	 * 後処理を設定したい結果コード毎に後処理パラメータを設定する。
	 * </p>
	 *
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 *
	 * <pre>
	 * {
	 *   "validate_ng": {
	 *     "result_type": "forward",
	 *     "next_path": "/user/shohin_list",
	 *     "params": {
	 *       "redisplay_flg": {
	 *         "transfer_type": "static",
	 *         "transfer_val": "1"
	 *       }
	 *     }
	 *   }
	 * }
	 * </pre>
	 *
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = false)
	private Map<String, Object> after;

	/**
	 * メッセージデータをセッションに追加するかしないかパラメータ.
	 * 1の時に追加する。デフォルトは0。
	 */
	@ModuleParam(required = false)
    @DefaultParamSetting(transferType = "static", transferVal = "0")
	private int useSession;
	/**
	 * メッセージデータをセッションに追加するときのKeyに使用
	 * セッションに追加する場合は必須
	 */
	@ModuleParam(required = false)
	private String appId;
	/**
	 * メッセージデータをセッションに追加するときのKeyに使用
	 * セッションに追加する場合は必須
	 */
	@ModuleParam(required = false)
	private String tenantId;
	/**
	 * メッセージデータをセッションに追加するときのKeyに使用
	 *
	 */
	private static final String messages = "messages";


	@SuppressWarnings("unchecked")
	@Override
	public ModuleResult execute(ModuleContext context) {

		ActionDto dto = context.getDto();

		ModuleResult result = new ModuleResult();

		Map<String, Object> actionModule = context.getActionModule();
		String returnPrefix = (String) actionModule.get("return_prefix");

		Map<String, Object> returnMessages = new HashMap<String, Object>();

		for (Map<String, Object> message : addMessages) {

			String messageType = ParamUtil
					.getParamStrValueByType(message.get("message_type"), dto);

			List<ActionDtoMessage> list = (List<ActionDtoMessage>) returnMessages
					.get(messageType);
			if (list == null) {
				list = new ArrayList<ActionDtoMessage>();
			}

			ActionDtoMessage adm = new ActionDtoMessage();

			String messageId = ParamUtil
					.getParamStrValueByType(message.get("message_id"), dto);
			String formFieldCd = ParamUtil
					.getParamStrValueByType(message.get("form_field_cd"), dto);

			Map<String, Object> messageParams = (Map<String, Object>) message
					.get("message_params");

			if (messageParams != null) {

				String messageParamType = (String) messageParams.get("type");

				Map<String, Object> messageParamVal = (Map<String, Object>) messageParams
						.get("val");

				if (!StringUtils.isEmpty(messageParamType)
						&& messageParamVal != null) {

					Map<String, Object> setParams = new HashMap<String, Object>();

					for (Map.Entry<String, Object> e : messageParamVal
							.entrySet()) {

						String key = e.getKey();
						Map<String, Object> mp = (Map<String, Object>) e
								.getValue();
						String paramType = (String) mp.get("type");
						String paramVal = (String) mp.get("val");

						if ("local".equals(paramType)) {
							Map<String, Object> localParam = (Map<String, Object>) message
									.get("local_params");
							Map<String, Object> setParamItem = new HashMap<String, Object>() {
								{
									put("type", "static");
									put("val", localParam.get(paramVal));
								}
							};
							setParams.put(key, setParamItem);
						} else {
							setParams.put(key, mp);
						}

					}

					adm.setParams(setParams);

				}

			}

			adm.setReturnPrefix(returnPrefix);
			adm.setMessageId(messageId);
			if (!StringUtils.isEmpty(formFieldCd)) {
				adm.setFormFieldCd(formFieldCd);
			}

			list.add(adm);

			returnMessages.put(messageType, list);

		}

		result.setMessages(returnMessages);
		result.setResultCode("add");
		// result.getReturnData().put(dtoKey, setListMap);

		// useSessionが１の時にセッションにセットを行う
		if(useSession == 1) {
			HttpSession session = context.getDto().getSession();
			// パラメータ情報をセッションに保持する
			session.setAttribute(messages + "-" + getTenantId() + "-" + getAppId(), returnMessages);
		}

		// モジュールのパラメータ"after"に値がセットされていたとき
		if (after != null) {
			Map<String, Object> mp = (Map<String, Object>) after
					.get(result.getResultCode());
			if (mp != null) {
				result.getReturnData().put("after_params", mp.get("params"));
				result.setNextPath((String) mp.get("next_path"));
				result.setResultType((String) mp.get("result_type"));
			}
		}

		return result;

	}

	/**
	 * addMessagesを取得します。
	 *
	 * @return addMessages
	 */
	public List<Map<String, Object>> getAddMessages() {
		return addMessages;
	}

	/**
	 * addMessagesを設定します。
	 *
	 * @param addMessages
	 */
	public void setAddMessages(List<Map<String, Object>> addMessages) {
		this.addMessages = addMessages;
	}

	/**
	 * afterを取得します。
	 *
	 * @return after
	 */
	public Map<String, Object> getAfter() {
		return after;
	}

	/**
	 * afterを設定します。
	 *
	 * @param after
	 */
	public void setAfter(Map<String, Object> after) {
		this.after = after;
	}

	/**
	 * useSessionを取得します。
	 *
	 * @return useSession
	 */
	public int getUseSession() {
		return useSession;
	}

	/**
	 * useSessionを設定します。
	 *
	 * @param useSession
	 */
	public void setUseSession(int useSession) {
		this.useSession = useSession;
	}

	/**
	 * tenantIdを取得します。
	 *
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 *
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	/**
	 * appIdを取得します。
	 *
	 * @return appId
	 */
	public String getAppId() {
		return appId;
	}

	/**
	 * appIdを設定します。
	 *
	 * @param appId
	 */
	public void setAppId(String appId) {
		this.appId = appId;
	}


}
