package jp.ill.photon.module.cart;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.UtilTools;

import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.seasar.doma.jdbc.tx.TransactionManager;

public class GetHolidaySettingsModule implements PhotonModule {

    @ModuleParam(required=true)
    @DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.inputBusinessDays.note")
    private String inputBusinessDays;

    @ModuleParam(required=true)
    @DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.wishingShipmentLimitTime.note")
    private String wishingShipmentLimitTime;

    public String getInputBusinessDays() {
        return inputBusinessDays;
    }

    public void setInputBusinessDays(String inputBusinessDays) {
        this.inputBusinessDays = inputBusinessDays;
    }

    public void setWishingShipmentLimitTime(String wishingShipmentLimitTime) {
        this.wishingShipmentLimitTime = wishingShipmentLimitTime;
    }

    @Override
    public ModuleResult execute(ModuleContext context)
            throws PhotonModuleException {
        ModuleResult result = new ModuleResult();

        TransactionManager tm = DomaConfig.singleton().getTransactionManager();
        JsonDataDao dao = new JsonDataDaoImpl();

        tm.required(() -> {

            // 休業日リストを取得する
            List<Map<String, Object>> holidayList = dao.getHolidayList(context.getDto().getTenantId());
            String scriptHoliday = getScriptHoliday(holidayList);

            result.getReturnData().put("script_holiday", scriptHoliday);

            // 休業曜日リストを取得する
            List<Map<String, Object>> weeklyData = dao.getWeekHolidayList(context.getDto().getTenantId());
            if (weeklyData == null) {
                weeklyData = new ArrayList<Map<String, Object>>();
            }
            if (weeklyData.size() <= 0) {
                weeklyData.add(new HashMap<String, Object>());
            }
            HashMap<String, String> weeklyMap = new HashMap<String, String>();
                weeklyMap.put(String.valueOf(Calendar.SUNDAY),
                        StringUtils.defaultIfEmpty((String)weeklyData.get(0).get("sunday_holiday_div"), "0"));
                weeklyMap.put(String.valueOf(Calendar.MONDAY),
                        StringUtils.defaultIfEmpty((String)weeklyData.get(0).get("monday_holiday_div"), "0"));
                weeklyMap.put(String.valueOf(Calendar.TUESDAY),
                        StringUtils.defaultIfEmpty((String)weeklyData.get(0).get("tuesday_holiday_div"), "0"));
                weeklyMap.put(String.valueOf(Calendar.WEDNESDAY),
                        StringUtils.defaultIfEmpty((String)weeklyData.get(0).get("wednesday_holiday_div"), "0"));
                weeklyMap.put(String.valueOf(Calendar.THURSDAY),
                        StringUtils.defaultIfEmpty((String)weeklyData.get(0).get("thursday_holiday_div"), "0"));
                weeklyMap.put(String.valueOf(Calendar.FRIDAY),
                        StringUtils.defaultIfEmpty((String)weeklyData.get(0).get("friday_holiday_div"), "0"));
                weeklyMap.put(String.valueOf(Calendar.SATURDAY),
                        StringUtils.defaultIfEmpty((String)weeklyData.get(0).get("saturday_holiday_div"), "0"));


            String scriptWeekHoliday = getScriptWeekHoliday(weeklyMap);

            result.getReturnData().put("script_week_holiday", scriptWeekHoliday);

            result.getReturnData().put("input_business_day", getInputBusinessDay(holidayList, weeklyMap));

        });

        return result;
    }

    public int getInputBusinessDay(List<Map<String, Object>> holiday, Map<String, String> weeklyMap) {

        Map<String, Integer> holidayMap = new HashMap<String, Integer>();
        for (Map<String, Object> m : holiday) {
            holidayMap.put((String)m.get("date"), 0);
        }

        // 全ての曜日が休業日かチェック
        boolean allWeekHoliday = false;
        if ("1".equals(weeklyMap.get(String
                .valueOf(Calendar.SUNDAY)))
                && "1".equals(weeklyMap
                        .get(String.valueOf(Calendar.MONDAY)))
                && "1".equals(weeklyMap
                        .get(String.valueOf(Calendar.TUESDAY)))
                && "1".equals(weeklyMap
                        .get(String.valueOf(Calendar.WEDNESDAY)))
                && "1".equals(weeklyMap
                        .get(String.valueOf(Calendar.THURSDAY)))
                && "1".equals(weeklyMap
                        .get(String.valueOf(Calendar.FRIDAY)))
                && "1".equals(weeklyMap
                        .get(String.valueOf(Calendar.SATURDAY)))) {
            allWeekHoliday = true;
        }

        // 到着希望日入力可能営業日数
        String days = inputBusinessDays;

        // 算出
        int numDays = 0;
        if (!UtilTools.isEmpty(days)) {
            numDays = Integer.parseInt(days);
        }
        int businessCnt = 0;
        Calendar businessDay = Calendar.getInstance();
        SimpleDateFormat fmt = new SimpleDateFormat("yyyyMMdd");
        // 全ての曜日が休日の場合は処理を行わない
        if (!allWeekHoliday) {
            // ループ
            // Rev.172>>>
            // 出荷希望日制限時間から当日指定可能かどうかチェック
            // 当日可の場合は1日プラスしない
            if (!getSendTodayFlg()) {
                // 当日指定不可 == 翌日からなので、1日プラス
                numDays++;
            }
            // <<<Rev.172
            while (businessCnt <= numDays) {
                // 休日判定
                boolean holFlg = false;
                String str = fmt.format(businessDay.getTime());
                if (holidayMap.keySet().contains(str)) {
                    holFlg = true;

                } else if ("1"
                        .equals(weeklyMap.get(String.valueOf(businessDay
                                .get(Calendar.DAY_OF_WEEK))))) {
                    holFlg = true;
                }
                if (!holFlg) {
                    // 営業日
                    businessCnt++;
                }
                if (businessCnt > numDays) {
                    // 日付加算前にループ終了
                    break;
                }
                // 日付加算
                businessDay = plusDay(businessDay, 1);
            }
        }

        // Rev.172>>>
        // // 出荷希望日制限時間から当日指定可能かどうかチェック
        // if (getSendTodayFlg()) {
        // // 当日可の場合は1日戻す
        // businessDay = DateUtil.plusDay(businessDay, -1);
        // }
        // <<<Rev.172

        fmt.applyPattern("yyyy/MM/dd");
        Calendar today = Calendar.getInstance();
        String inputBusinessDay =  fmt.format(businessDay.getTime());

        int count = termCalc(today.get(Calendar.YEAR),
                today.get(Calendar.MONTH) + 1, today.get(Calendar.DATE),
                Integer.parseInt(inputBusinessDay.substring(0, 4)),
                Integer.parseInt(inputBusinessDay.substring(5, 7)),
                Integer.parseInt(inputBusinessDay.substring(8, 10)));
        // Rev.172>>>
        return count;
        // <<<Rev.172
    }

    private static int termCalc(int ys, int ms, int ds, int ye, int me, int de) {

        int term = 0;

        Calendar objAfter = Calendar.getInstance();
        Calendar objBefore = Calendar.getInstance();

        // Calenderクラスでは1月は0、2月は1となっているため、取得月-1する。
        objAfter.set(ye, me - 1, de);
        objBefore.set(ys, ms - 1, ds);

        long timeDiff = objAfter.getTimeInMillis()
                - objBefore.getTimeInMillis();
        long dayDif = timeDiff / 1000 / 60 / 60 / 24;

        term = (int) (dayDif);

        return term;
    }

    private static Calendar plusDay(Calendar cal, int plus) {
        // 日付の加算
        cal.add(Calendar.DATE, plus);

        return cal;

    }

    /**
     * 到着希望日の当日選択可否を取得
     *
     * @return 当日可/不可(true/false)
     */
    public boolean getSendTodayFlg() {
        try {
            // 時間と分に分割
            String[] times = wishingShipmentLimitTime.split(":", -1);

            DateTime dt1 = new DateTime(); // 現在日付用

            // 抑制時間を設定
            DateTime dt2 = dt1.withTime(Integer.parseInt(times[0]),
                    Integer.parseInt(times[1]), 0, 0);

            // dt1(現在日時)はdt2(抑制時間)より過去か
            return dt1.isBefore(dt2);

        } catch (Exception e) {
            return false;
        }
    }

    /**
     * javaScript用休業曜日リスト
     *
     * @return
     */
    public String getScriptWeekHoliday(Map<String, String> weeklyMap) {

        StringBuffer returnValue = new StringBuffer();
        for (String key : weeklyMap.keySet()) {
            if ("1".equals(weeklyMap
                    .get(key))) {
                // 休日
                if (returnValue.length() > 0) {
                    returnValue.append(",");
                }
                // 日曜日：0～土曜：6
                returnValue.append(Integer.parseInt(key) - 1);
            }
        }

        return returnValue.toString();
    }

    public String getScriptHoliday(List<Map<String, Object>> holidayList) {

        StringBuffer returnValue = new StringBuffer();
        for (Map<String, Object> map : holidayList) {
            String date = (String)map.get("date");

            if (returnValue.length() > 0) {
                returnValue.append(",");
            }
            String year = date.substring(0, 4);
            String month = date.substring(4, 6);
            String day = date.substring(6, 8);

            returnValue.append(year + "-" + month + "-" + day);
        }
        return returnValue.toString();
    }

}
