package jp.ill.photon.module.cart;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.action.ActionDispatcher;
import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.UtilTools;

public class GetHeaderCartModule implements PhotonModule {

	@ModuleParam(required=true)
	private String tenantId;

	@ModuleParam(required=true)
	private String userCd;

	@ModuleParam(required=true)
	private String userChargeId;

	@ModuleParam(required=true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.caseNumDiv.note")
	private String caseNumDiv;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.aladdinsetting.売上単価整数桁.value")
	private String salesUnitPriceDigitNum;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.aladdinsetting.売上単価小数点桁.value")
	private String salesUnitPriceDecimalPointNum;

	@ModuleParam(required=false)
	private String salesUnitPriceRoundingDiv; // 得意先のprice_fraction_div

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.aladdinsetting.売上金額整数桁.value")
	private String salesPriceDigitNum;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "static", transferVal = "0")
	private String salesPriceDecimalPointNum;

	@ModuleParam(required=false)
	private String salesPriceRoundingDiv; // 得意先のunit_price_fraction_div

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.aladdinsetting.売上数量整数桁.value")
	private String salesQuantityDigitNum;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.aladdinsetting.売上数量小数点桁.value")
	private String salesQuantityDecimalPointNum;

	@ModuleParam(required=false)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.aladdinsetting.売上数量端数処理区分.value")
	private String salesQuantityRoundingDiv;


	/** ケース数区分 */
	public static class CaseNumDiv {
		/** 表示 */
		private static final String ON = "1";
		/** 非表示 */
		private static final String OFF = "0";
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(ActionDispatcher.class);

	// dao
	private JsonDataDao dao;

	@Override
	public ModuleResult execute(ModuleContext context) throws PhotonModuleException {

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		dao = new JsonDataDaoImpl();

		ModuleResult result = new ModuleResult();

		String wholesalePriceVal = "0";
		int itemCnt = 0;

		List<Map<String, Object>> tmpList = tm.required(() -> {
			return dao.getListByUserChargeId(tenantId, userCd, userChargeId);
		});

		if (!CollectionUtils.isEmpty(tmpList)) {

			for (Map<String, Object> tmp : tmpList) {

				BigDecimal orderCaseQuantityBdVal = new BigDecimal( StringUtils.defaultIfEmpty((String)tmp.get("order_case_quantity"), "0") );
				if (CaseNumDiv.OFF.equals(caseNumDiv) &&
					orderCaseQuantityBdVal.compareTo(BigDecimal.ZERO) != 0) {

					// ケース入力を使用しないモードのときには、カート内のケースデータは勘定に入れない。
					continue;

				}

				itemCnt++;

				// 単価の端数処理
				String unitPrice = AoUtil.convAoRoundingSalesUnitPrice(
					StringUtils.defaultIfEmpty((String)tmp.get("wholesale_price"), "0") ,
					AoUtil.DispPattern.PATTERN_NO_COMMA,
					salesUnitPriceDigitNum,
					salesUnitPriceDecimalPointNum,
					salesUnitPriceRoundingDiv
				);

				// 金額の端数処理
				String price = AoUtil.convAoRoundingSalesPrice(
						StringUtils.defaultIfEmpty((String)tmp.get("order_quantity"), "0") ,
						unitPrice,
						AoUtil.DispPattern.PATTERN_NO_COMMA,
						salesPriceDigitNum,
						salesPriceRoundingDiv
				);

				// 合計
				wholesalePriceVal = UtilTools.addBigDecimal(wholesalePriceVal, price);

			}

		}

		// 返却する値
		String wholesalePriceFormatted = AoUtil.convAoRoundingSalesPrice(
				wholesalePriceVal,
				salesPriceDigitNum,
				salesPriceRoundingDiv);

		logger.info(String.format("---->[count]:[%s]", itemCnt));
		logger.info(String.format("---->[wholesale_price]:[%s]", wholesalePriceFormatted));

		result.getReturnData().put("count",  itemCnt);
		result.getReturnData().put("wholesale_price", wholesalePriceFormatted);

		result.setResultCode("get");

		return result;

	}

	/**
	 * tenantIdを取得します。
	 * @return tenantId
	 */
	public String getTenantId() {
		return tenantId;
	}

	/**
	 * tenantIdを設定します。
	 * @param tenantId
	 */
	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	/**
	 * userCdを取得します。
	 * @return userCd
	 */
	public String getUserCd() {
		return userCd;
	}

	/**
	 * userCdを設定します。
	 * @param userCd
	 */
	public void setUserCd(String userCd) {
		this.userCd = userCd;
	}

	/**
	 * userChargeIdを取得します。
	 * @return userChargeId
	 */
	public String getUserChargeId() {
		return userChargeId;
	}

	/**
	 * userChargeIdを設定します。
	 * @param userChargeId
	 */
	public void setUserChargeId(String userChargeId) {
		this.userChargeId = userChargeId;
	}

	/**
	 * caseNumDivを取得します。
	 * @return caseNumDiv
	 */
	public String getCaseNumDiv() {
		return caseNumDiv;
	}

	/**
	 * caseNumDivを設定します。
	 * @param caseNumDiv
	 */
	public void setCaseNumDiv(String caseNumDiv) {
		this.caseNumDiv = caseNumDiv;
	}

	/**
	 * salesUnitPriceDigitNumを取得します。
	 * @return salesUnitPriceDigitNum
	 */
	public String getSalesUnitPriceDigitNum() {
		return salesUnitPriceDigitNum;
	}

	/**
	 * salesUnitPriceDigitNumを設定します。
	 * @param salesUnitPriceDigitNum
	 */
	public void setSalesUnitPriceDigitNum(String salesUnitPriceDigitNum) {
		this.salesUnitPriceDigitNum = salesUnitPriceDigitNum;
	}

	/**
	 * salesUnitPriceDecimalPointNumを取得します。
	 * @return salesUnitPriceDecimalPointNum
	 */
	public String getSalesUnitPriceDecimalPointNum() {
		return salesUnitPriceDecimalPointNum;
	}

	/**
	 * salesUnitPriceDecimalPointNumを設定します。
	 * @param salesUnitPriceDecimalPointNum
	 */
	public void setSalesUnitPriceDecimalPointNum(String salesUnitPriceDecimalPointNum) {
		this.salesUnitPriceDecimalPointNum = salesUnitPriceDecimalPointNum;
	}

	/**
	 * salesUnitPriceRoundingDivを取得します。
	 * @return salesUnitPriceRoundingDiv
	 */
	public String getSalesUnitPriceRoundingDiv() {
		return salesUnitPriceRoundingDiv;
	}

	/**
	 * salesUnitPriceRoundingDivを設定します。
	 * @param salesUnitPriceRoundingDiv
	 */
	public void setSalesUnitPriceRoundingDiv(String salesUnitPriceRoundingDiv) {
		this.salesUnitPriceRoundingDiv = salesUnitPriceRoundingDiv;
	}

	/**
	 * salesPriceDigitNumを取得します。
	 * @return salesPriceDigitNum
	 */
	public String getSalesPriceDigitNum() {
		return salesPriceDigitNum;
	}

	/**
	 * salesPriceDigitNumを設定します。
	 * @param salesPriceDigitNum
	 */
	public void setSalesPriceDigitNum(String salesPriceDigitNum) {
		this.salesPriceDigitNum = salesPriceDigitNum;
	}

	/**
	 * salesPriceDecimalPointNumを取得します。
	 * @return salesPriceDecimalPointNum
	 */
	public String getSalesPriceDecimalPointNum() {
		return salesPriceDecimalPointNum;
	}

	/**
	 * salesPriceDecimalPointNumを設定します。
	 * @param salesPriceDecimalPointNum
	 */
	public void setSalesPriceDecimalPointNum(String salesPriceDecimalPointNum) {
		this.salesPriceDecimalPointNum = salesPriceDecimalPointNum;
	}

	/**
	 * salesPriceRoundingDivを取得します。
	 * @return salesPriceRoundingDiv
	 */
	public String getSalesPriceRoundingDiv() {
		return salesPriceRoundingDiv;
	}

	/**
	 * salesPriceRoundingDivを設定します。
	 * @param salesPriceRoundingDiv
	 */
	public void setSalesPriceRoundingDiv(String salesPriceRoundingDiv) {
		this.salesPriceRoundingDiv = salesPriceRoundingDiv;
	}

	/**
	 * salesQuantityDigitNumを取得します。
	 * @return salesQuantityDigitNum
	 */
	public String getSalesQuantityDigitNum() {
		return salesQuantityDigitNum;
	}

	/**
	 * salesQuantityDigitNumを設定します。
	 * @param salesQuantityDigitNum
	 */
	public void setSalesQuantityDigitNum(String salesQuantityDigitNum) {
		this.salesQuantityDigitNum = salesQuantityDigitNum;
	}

	/**
	 * salesQuantityDecimalPointNumを取得します。
	 * @return salesQuantityDecimalPointNum
	 */
	public String getSalesQuantityDecimalPointNum() {
		return salesQuantityDecimalPointNum;
	}

	/**
	 * salesQuantityDecimalPointNumを設定します。
	 * @param salesQuantityDecimalPointNum
	 */
	public void setSalesQuantityDecimalPointNum(String salesQuantityDecimalPointNum) {
		this.salesQuantityDecimalPointNum = salesQuantityDecimalPointNum;
	}

	/**
	 * salesQuantityRoundingDivを取得します。
	 * @return salesQuantityRoundingDiv
	 */
	public String getSalesQuantityRoundingDiv() {
		return salesQuantityRoundingDiv;
	}

	/**
	 * salesQuantityRoundingDivを設定します。
	 * @param salesQuantityRoundingDiv
	 */
	public void setSalesQuantityRoundingDiv(String salesQuantityRoundingDiv) {
		this.salesQuantityRoundingDiv = salesQuantityRoundingDiv;
	}



}
