package jp.ill.photon.module.cart;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.action.ActionDispatcher;
import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.model.AoSetting;
import jp.ill.photon.model.CartItem;
import jp.ill.photon.model.CommonFlg;
import jp.ill.photon.model.Item;
import jp.ill.photon.model.Item.CartDispFlg;
import jp.ill.photon.model.SystemSetting.CaseNumDiv;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.AoFormatCheckUtil;
import jp.ill.photon.util.AoUtil;
import jp.ill.photon.util.AoUtil.FractionDiv;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.StringUtil;
import jp.ill.photon.util.UtilTools;

public class CartInModule implements PhotonModule {

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "param_multi", transferVal = "item_cd")
	private String[] itemCd;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "param_multi", transferVal = "order_quantity")
	private String[] orderQuantity;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "param_multi", transferVal = "order_case_quantity")
	private String[] orderCaseQuantity;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "got_session.userForm.user_cd")
	private String userCd;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "got_session_charge.userChargeForm.user_charge_id")
	private int userChargeId;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType="dto", transferVal="common.aladdinsetting")
	private AoSetting aoSetting;

	@ModuleParam(required = true)
	@DefaultParamSetting(transferType = "dto", transferVal = "common.systemsetting.caseNumDiv.note")
	private String caseNumDiv;

	@ModuleParam(required = true)
	private List<Map<String, Object>> itemList;

	@ModuleParam(required = true)
    @DefaultParamSetting(transferType = "dto", transferVal = "user.first")
    private Map<String, Object> userForm;

	/** メッセージID（0件エラー） */
    public static final String CART_ERROR_MSG_ID = "cartInNoInput";

	/** メッセージID（商品無効） */
	public static final String INPUT_ERR_ITEM_NON_ACTIVE = "itemMesNoCartAdd";

	/** メッセージID（入力値無効） */
	public static final String INPUT_ERR = "inputErrMesNumber";

	/** メッセージID（入力値無効） */
	public static final String INPUT_ERR_NUMBER = "inputErrMesCartNumber";

	/** メッセージID（注文数量0以下） */
	public static final String INPUT_ERR_ORDER_NUM = "inputErrMesValueMinNotInclude";

	/** メッセージID（在庫なしエラー） */
	public static final String ORDER_ERR_MES_STOCK_OUT = "orderErrMesStockOut";


	private static String ERR_NUMBER_INVALID = "number invalid";

    public Map<String, Object> getUserForm() {
        return userForm;
    }

    public void setUserForm(Map<String, Object> userForm) {
        this.userForm = userForm;
    }

	public String getCaseNumDiv() {
		return caseNumDiv;
	}

	public void setCaseNumDiv(String caseNumDiv) {
		this.caseNumDiv = caseNumDiv;
	}

	public String[] getItemCd() {
		return itemCd;
	}

	public void setItemCd(String[] itemCd) {
		this.itemCd = itemCd;
	}

	public String[] getOrderQuantity() {
		return orderQuantity;
	}

	public void setOrderQuantity(String[] quantity) {
		this.orderQuantity = quantity;
	}

	public String[] getOrderCaseQuantity() {
		return orderCaseQuantity;
	}

	public void setOrderCaseQuantity(String[] orderCaseQuantity) {
		this.orderCaseQuantity = orderCaseQuantity;
	}

	public String getUserCd() {
		return userCd;
	}

	public void setUserCd(String userCd) {
		this.userCd = userCd;
	}

	public int getUserChargeId() {
		return userChargeId;
	}

	public void setUserChargeId(int userChargeId) {
		this.userChargeId = userChargeId;
	}

	public AoSetting getAoSetting() {
		return aoSetting;
	}

	public void setAoSetting(AoSetting aoSetting) {
		this.aoSetting = aoSetting;
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(ActionDispatcher.class);

	@Override
	public ModuleResult execute( ModuleContext context ) {

		ModuleResult result = new ModuleResult();
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		Map<String, List<String>> errorMessageListMap = new HashMap<String, List<String>>();
		Map<String, Object> retrunMessageListMap = new HashMap<String, Object>();
		List<ActionDtoMessage> tmpErrorMessage = new ArrayList<ActionDtoMessage>();

		if (itemCd.length == 0 || checkEmpty()) {
		    // 0件だったら、エラーにして返す
		    result.setResultCode("error");
		    result.getReturnData().put("result_code", "error");

		    List<ActionDtoMessage> errorMessages = new ArrayList<ActionDtoMessage>();
            ActionDtoMessage msg = new ActionDtoMessage();
            msg.setMessageId(CART_ERROR_MSG_ID);
            errorMessages.add(msg);

            result.getMessages().put("error_page", errorMessages);

			return result;
		}

		boolean isAllEmpty = true;

		for (int i = 0; i < itemCd.length; i++) {

			List<String> errorMessage = new ArrayList<String>();

			String itemCdStr = itemCd[i];
			String quantityStr = "0";
			if (orderQuantity != null && orderQuantity.length >= (i+1)) {
				quantityStr = orderQuantity[i];
			}
			String caseQuantityStr = "0";
			if (orderCaseQuantity != null && orderCaseQuantity.length >= (i+1)) {
				caseQuantityStr = orderCaseQuantity[i];
			}

			logger.info(String.format("---->[item_cd]:[%s]", itemCdStr));
			logger.info(String.format("---->[order_quantity]:[%s]", quantityStr));
			logger.info(String.format("---->[order_case_quantity]:[%s]", caseQuantityStr));

			if (StringUtils.isEmpty(itemCdStr)) {
				logger.info("---->---->商品コードが空なので、登録しない。");
				continue;
			}

			Item item = getItemData(itemCdStr);

			if (item == null || item.getCartDispFlg() == Integer.parseInt(CommonFlg.OFF)) {
				// エラーメッセージを積む
				logger.info("---->---->有効な商品ではないので、登録しない。");
				result.setResultCode("error");
				result.getReturnData().put("result_code", "error");

				ActionDtoMessage dtoMessage = new ActionDtoMessage();
				dtoMessage.setFormFieldCd(itemCdStr);
				dtoMessage.setMessageId(INPUT_ERR_ITEM_NON_ACTIVE);
				tmpErrorMessage.add(dtoMessage);
				continue;
			}

			// 入力フォーマットチェック
			boolean[] checkResult = checkCartInputFormat(item, quantityStr, caseQuantityStr, caseNumDiv, aoSetting, errorMessage, userForm);

			if (!errorMessage.isEmpty()) {
				if (ERR_NUMBER_INVALID.equals(errorMessage.get(0))) {
					addErrorMessage(errorMessageListMap, itemCdStr, errorMessage, result, tmpErrorMessage, INPUT_ERR_NUMBER);
					continue;
				} else {
					addErrorMessage(errorMessageListMap, itemCdStr, errorMessage, result, tmpErrorMessage, INPUT_ERR);
					continue;
				}
			}

			// 商品単品のカート投入で、注文数量が0以下の場合
			BigDecimal tmpQuantity = StringUtils.defaultString(quantityStr, "0").length() > 0 ? new BigDecimal(
																												StringUtils.defaultString(
																														quantityStr,
																														"0")) : BigDecimal.ZERO;
			BigDecimal tmpCaseQuantity = StringUtils.defaultString(caseQuantityStr, "0").length() > 0 ? new BigDecimal(
																														StringUtils.defaultString(
																																caseQuantityStr,
																																"0"))
					: BigDecimal.ZERO;
			if ((itemCd.length == 1 && (BigDecimal.ZERO.compareTo(tmpQuantity) >= 0 && BigDecimal.ZERO.compareTo(tmpCaseQuantity) >= 0))) {
				ActionDtoMessage dtoMessage = new ActionDtoMessage();
				dtoMessage.setFormFieldCd(itemCdStr);
				dtoMessage.setMessageId(INPUT_ERR_ORDER_NUM);
				Map<String, Object> messageMap = new HashMap<String, Object>();
				Map<String, Object> messageParamMap1 = new HashMap<String, Object>();
				messageParamMap1.put("val", "注文数量");
				messageParamMap1.put("type", "static_json");
				messageMap.put("%1$s", messageParamMap1);
				Map<String, Object> messageParamMap2 = new HashMap<String, Object>();
				messageParamMap2.put("val", "0");
				messageParamMap2.put("type", "static_json");
				messageMap.put("%2$s", messageParamMap2);
				dtoMessage.setParams(messageMap);
				tmpErrorMessage.add(dtoMessage);
				continue;
			}

			// 商品在庫チェックの結果がNGの場合
			@SuppressWarnings("unchecked")
			List<Map<String, Object>> check_query =  (List<Map<String, Object>>) context.getDto().get("check_query", "list");
			boolean stockNumCheckErr = false;
			if(check_query != null){
				for(Map<String, Object> record: check_query){
					String stockCheckResult = (String) record.get("result_of_stock_check"); // 在庫チェック結果
					String itemName = (String) record.get("item_name"); // 商品名
					String reqCaseDiv = StringUtil.defaultString((String) record.get("req_case_div"), "");   // リクエストケース区分
					String dbCaseDiv = StringUtil.defaultString((String) record.get("db_case_div"), "");   // DBケース区分
					String quantity = (String) record.get("quantity");  // 入力数量
					String stockNum = (String) record.get("stock_num"); // 在庫数量

					if("OK".equals(stockCheckResult)){
						continue;
					}

					// 在庫チェック結果がNG かつ、数量が空でない、かつ、
					// リクエストしたケース区分がカートにないか、リクエストしたケース区分がカートと一致
					if("NG".equals(stockCheckResult) && !UtilTools.isEmpty(quantity) &&
							(UtilTools.isEmpty(dbCaseDiv)) || (reqCaseDiv.equals(dbCaseDiv))){
						ActionDtoMessage dtoMessage = new ActionDtoMessage();
						dtoMessage.setFormFieldCd(itemCdStr);
						dtoMessage.setMessageId(ORDER_ERR_MES_STOCK_OUT);
						Map<String, Object> messageMap = new HashMap<String, Object>();
						Map<String, Object> messageParamMap1 = new HashMap<String, Object>();
						messageParamMap1.put("val", StringUtil.defaultString(itemName, ""));
						messageParamMap1.put("type", "static_json");
						messageMap.put("%1$s", messageParamMap1);
						Map<String, Object> messageParamMap2 = new HashMap<String, Object>();
						messageParamMap2.put("val", StringUtil.defaultString(quantity, ""));
						messageParamMap2.put("type", "static_json");
						messageMap.put("%2$s", messageParamMap2);
						Map<String, Object> messageParamMap3 = new HashMap<String, Object>();
						messageParamMap3.put("val", StringUtil.defaultString(stockNum, ""));
						messageParamMap3.put("type", "static_json");
						messageMap.put("%3$s", messageParamMap3);
						dtoMessage.setParams(messageMap);
						tmpErrorMessage.add(dtoMessage);

						stockNumCheckErr = true;
						continue;
					}
				}
			}

			if (tmpQuantity.compareTo(BigDecimal.ZERO) > 0 || tmpCaseQuantity.compareTo(BigDecimal.ZERO) > 0) {
			    isAllEmpty = false;
			}

			boolean orderQuantityIsBlank = checkResult[0];
			boolean orderCaseQuantityIsBlank = checkResult[1];
			boolean orderQuantityFormatErr = checkResult[2];

			// 数量チェック、または商品在庫チェックがエラーのときには登録しない
			if (!orderQuantityFormatErr && !stockNumCheckErr) {

			    if (!orderCaseQuantityIsBlank) {
	                String quantityPerCase = AoUtil.convAoRoundingUnitQuantity(
	                        item.getQuantityPerCase().toString(),
	                        AoUtil.DispPattern.PATTERN_NO_COMMA,
	                        aoSetting.getUnitQuantityDigitNum(),
	                        aoSetting.getUnitQuantityDecimalPointNum(),
	                        FractionDiv.DOWN
	                );
	                String orderQuantityByCase = CartInModule.quantityCalc(caseQuantityStr,
	                        quantityPerCase,
	                        aoSetting
	                );

	                logger.info("---->---->ケースとしてカートデータに登録。");
	                // カート更新（ケース）
	                addCartData(context, item, orderQuantityByCase, caseQuantityStr);
	            }
	            if (!orderQuantityIsBlank) {
	                logger.info("---->---->バラとしてカートデータに登録。");
	                // カート更新（バラ）
	                addCartData(context, item, quantityStr, null);
	            }
			}
		}

		if (itemCd.length > 1 && isAllEmpty) {
		    ActionDtoMessage dtoMessage = new ActionDtoMessage();
            dtoMessage.setMessageId(INPUT_ERR_ORDER_NUM);
            Map<String, Object> messageMap = new HashMap<String, Object>();
            Map<String, Object> messageParamMap1 = new HashMap<String, Object>();
            messageParamMap1.put("val", "注文数量");
            messageParamMap1.put("type", "static_json");
            messageMap.put("%1$s", messageParamMap1);
            Map<String, Object> messageParamMap2 = new HashMap<String, Object>();
            messageParamMap2.put("val", "0");
            messageParamMap2.put("type", "static_json");
            messageMap.put("%2$s", messageParamMap2);
            dtoMessage.setParams(messageMap);
            tmpErrorMessage.add(dtoMessage);
		}

		retrunMessageListMap.put("error_field", tmpErrorMessage);
		result.getReturnData().put("messages", errorMessageListMap);
		result.setMessages(retrunMessageListMap);

		return result;
	}

	private void addErrorMessage(Map<String, List<String>> errorMessageListMap,
			String itemCdStr, List<String> errorMessage, ModuleResult result, List<ActionDtoMessage> tmpErrorMessage, String errorMsg) {
		errorMessageListMap.put(itemCdStr, errorMessage);
		logger.info("---->---->入力フォーマットチェックでエラーが出たので、登録しない。");
		result.setResultCode("error");
		result.getReturnData().put("result_code", "error");

		ActionDtoMessage dtoMessage = new ActionDtoMessage();
		dtoMessage.setFormFieldCd(itemCdStr);
		dtoMessage.setMessageId(errorMsg);
		Map<String, Object> messageMap = new HashMap<String, Object>();
		Map<String, Object> messageParamMap = new HashMap<String, Object>();
		messageParamMap.put("val", "注文数量");
		messageParamMap.put("type", "static_json");
		messageMap.put("%1$s", messageParamMap);
		dtoMessage.setParams(messageMap);
		tmpErrorMessage.add(dtoMessage);
	}

	private boolean checkEmpty() {
	    boolean result = true;

	    for (int i = 0; i < itemCd.length; i++) {
            String itemCdStr = itemCd[i];

            if (itemCdStr != null && !"".equals(itemCdStr)) {
                result = false;
                return result;
            }
	    }

	    return result;
	}

	public static boolean[] checkCartInputFormat( Item item, String quantityStr, String caseQuantityStr, String caseNumDiv, AoSetting aoSetting, List<String> errorMessage, Map<String, Object> userForm) {
		return checkCartInputFormat( item, quantityStr, caseQuantityStr, caseNumDiv, aoSetting, errorMessage, "1", userForm);
	}

	public static boolean[] checkCartInputFormat( Item item, String quantityStr, String caseQuantityStr, String caseNumDiv, AoSetting aoSetting, List<String> errorMessage, String defValue, Map<String, Object> userForm) {
	    return checkCartInputFormat(item.getWholeSalePrice().toString(), String.valueOf(item.getItemDispFlg()), String.valueOf(item.getCartDispFlg()), item.getQuantityPerCase().toString(), quantityStr, caseQuantityStr, caseNumDiv, aoSetting, errorMessage, defValue, userForm);
	}

	public static boolean[] checkCartInputFormat( String price, String itemDispFlg, String cartDispFlg, String quantityPerCase,
	        String quantityStr, String caseQuantityStr, String caseNumDiv, AoSetting aoSetting, List<String> errorMessage, String defValue, Map<String, Object> userForm) {

		boolean[] result = new boolean[3];
		result[0] = false;
		result[1] = false;
		result[2] = false;

		if ("0".equals(String.valueOf(itemDispFlg))) {
			// エラーメッセージを積む
		    errorMessage.add("item invalid error.");
			return result;
		}

		if (CartDispFlg.NO.equals(String.valueOf(cartDispFlg))) {
			// エラーメッセージを積む
		    errorMessage.add("cart btn invalid error.");
			return result;
		}

		String unitPrice = price;

		// フォーマット違反フラグ
		boolean quantityFormatErrFlg = false;
		// ケース入力の数量
		String orderQuantityByCase = "0";

		// ケース数が入力されているか
		boolean orderCaseQuantityIsBlank = true;
		if ( !StringUtils.isEmpty(caseQuantityStr) &&
			 (BigDecimal.ZERO.compareTo(new BigDecimal(caseQuantityStr)) != 0) ) {
			orderCaseQuantityIsBlank = false;
		}

		// 数量が入力されているか
		boolean orderQuantityIsBlank = true;
		if ( !StringUtils.isEmpty(quantityStr) &&
			 (BigDecimal.ZERO.compareTo(new BigDecimal(quantityStr)) != 0) ) {
			orderQuantityIsBlank = false;
		}

		// ケース数チェック
		if (!orderCaseQuantityIsBlank) {
			if (caseNumDiv.equals(CaseNumDiv.ON)
					&& quantityPerCase != null
					&& new BigDecimal(quantityPerCase).compareTo(
							new BigDecimal(0)) > 0) {
				// ケース数の桁数チェック
				if (!AoFormatCheckUtil.checkMakeCaseQuantityFormat( caseQuantityStr, aoSetting )) {
					// エラーメッセージを積む
				    errorMessage.add(ERR_NUMBER_INVALID);
				}

				caseQuantityStr = AoFormatCheckUtil.makeCaseQuantity(caseQuantityStr, aoSetting);

				// 入数
				String quantityPerCaseFormat = AoUtil.convAoRoundingUnitQuantity(quantityPerCase, AoUtil.DispPattern.PATTERN_NO_COMMA,
						aoSetting.getUnitQuantityDigitNum(), aoSetting.getUnitQuantityDecimalPointNum(),FractionDiv.DOWN);

				orderQuantityByCase = CartInModule.quantityCalc(caseQuantityStr, quantityPerCaseFormat, aoSetting);

				// 数量桁数チェック
				if (!AoFormatCheckUtil.checkMakeQuantityFormat(orderQuantityByCase, aoSetting)) {
					// エラーメッセージを積む
				    errorMessage.add(ERR_NUMBER_INVALID);
					quantityFormatErrFlg = true;
				} else if (!AoFormatCheckUtil.checkMakeQuantity(orderQuantityByCase, unitPrice, aoSetting, userForm)) {
					// エラーメッセージを積む
				    errorMessage.add(ERR_NUMBER_INVALID);
					quantityFormatErrFlg = true;
				}

				// ケース数の数量フォーマットチェック（チェック違反なら1を設定）
				if (quantityFormatErrFlg) {
					caseQuantityStr = "1";
					orderQuantityByCase = CartInModule.quantityCalc(caseQuantityStr, quantityPerCaseFormat, aoSetting);
				}
			} else {
				// ケース数入力不可
				caseQuantityStr = "";
			}
		}

		// 数量チェック
		if (!orderQuantityIsBlank) {
			if (!AoFormatCheckUtil.checkMakeQuantityFormat(quantityStr, aoSetting)) {
				// エラーメッセージを積む
			    errorMessage.add(ERR_NUMBER_INVALID);
			    quantityFormatErrFlg = true;
			} else if (!AoFormatCheckUtil.checkMakeQuantity(quantityStr,  unitPrice, aoSetting, userForm)) {
				// エラーメッセージを積む
			    errorMessage.add(ERR_NUMBER_INVALID);
			    quantityFormatErrFlg = true;
			}

			// 注文数量のフォーマットチェック（チェック違反なら1を設定）
			quantityStr = AoFormatCheckUtil.makeQuantity(quantityStr, unitPrice, aoSetting, userForm);

		}

		result[0] = orderQuantityIsBlank;
		result[1] = orderCaseQuantityIsBlank;
		result[2] = quantityFormatErrFlg;

		return result;
	}

	private void addCartData(ModuleContext context, Item item, String quantityStr, String caseQuantityStr) {

		// 既にカートに存在するかチェック
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = new JsonDataDaoImpl();

		int result = tm.required(() -> {

			boolean caseFlg = false;

			if (!StringUtils.isEmpty(caseQuantityStr)) {
				caseFlg = true;
			}

			CartItem cart = null;
			List<Map<String, Object>> cartMap = dao.getCartByItemCd(context.getDto().getTenantId(), userCd, String.valueOf(userChargeId), item.getItemCd(), caseFlg);
			if (cartMap != null && !cartMap.isEmpty()) {
				cart = CartItem.valueOf(cartMap.get(0));
			}

			if (cart == null) {
				cart = new CartItem();
				cart.setItemCd(item.getItemCd());
				cart.setUserChargeId(String.valueOf(userChargeId));
				cart.setUserCd(userCd);
			}

			cart.setOrderQuantity(new BigDecimal(quantityStr));
			if (caseFlg) {
				cart.setOrderCaseQuantity(new BigDecimal(caseQuantityStr));
			} else {
				cart.setOrderCaseQuantity(new BigDecimal(0));
			}

			// この辺りはセットしなくなるはず
			cart.setWholesalePrice(item.getWholeSalePrice());
			cart.setRetailPrice(new BigDecimal(0)); // 暫定

			cart.setItemName1(item.getItemName1());
			cart.setItemName2(item.getItemName2());
			cart.setQuantityPerCase(item.getQuantityPerCase());

			if (item.getUnit() != null) {
				cart.setUnit(item.getUnit().toString());
			}

			if (cart.getCartId() == null) {
				// シーケンスを取得
				List<Map<String, Object>> seqMap = dao.getSeqSfNextVal(context.getDto().getTenantId(), "cart", "cart_id");
				if (seqMap == null || seqMap.isEmpty()) {
					return -1;
				}
				int cartId = Integer.valueOf(String.valueOf(seqMap.get(0).get("seq")));
				cart.setCartId(cartId);

				// 挿入
				try {
					dao.insertEditTableData(context.getDto().getTenantId(), "cart", userCd, JsonUtil.mapToJsonSnakeCaseKey(cart));
				} catch (Exception ex) {
					ex.printStackTrace();
				}

				return 0;
			} else {
				// 更新
				dao.updateCartItem(context.getDto().getTenantId(), userCd, String.valueOf(userChargeId), cart);
			}

			return -1;

		});

	}

	/**
	 * 数量計算.
	 *
	 * @param caseQuantity ケース数
	 * @param quantityPerCase 入数
	 * @return 数量
	 */
	public static String quantityCalc(String caseQuantity, String quantityPerCase,AoSetting aoSetting) {
		return AoUtil.convAoRoundingSalesQuantity(caseQuantity, quantityPerCase, AoUtil.DispPattern.PATTERN_NO_COMMA,
				aoSetting.getSalesQuantityDigitNum(), aoSetting.getSalesQuantityDecimalPointNum(), aoSetting.getSalesQuantityRoundingDiv());
	}

	/**
	 * itemListを取得します。
	 * @return itemList
	 */
	public List<Map<String,Object>> getItemList() {
	    return itemList;
	}

	/**
	 * itemListを設定します。
	 * @param itemList itemList
	 */
	public void setItemList(List<Map<String,Object>> itemList) {
	    this.itemList = itemList;
	}

	/**
	 * itemListを設定します。
	 * @param itemList itemList
	 */
	public Item getItemData(String itemCd) {
		Item rtnItem = null;
		for (Map<String, Object> itemData : itemList) {
			if (!itemCd.equals(itemData.get("item_cd"))) {
				continue;
			}
			rtnItem = new Item();
			rtnItem.setItemCd(String.valueOf(itemData.get("item_cd")));
			rtnItem.setWholeSalePrice(new BigDecimal(String.valueOf(itemData.get("price") == null ? "0" : itemData.get("price"))));
			rtnItem.setItemDispFlg(1); // 取得SQLの条件句がdisp_flg = 1 のため1固定でセット
			rtnItem.setCartDispFlg(Integer.valueOf(String.valueOf(itemData.get("cart_disp_flg") == null ? "0" : itemData.get("cart_disp_flg"))));
			rtnItem.setQuantityPerCase(
					new BigDecimal(String.valueOf(itemData.get("quantity_per_case") == null ? "0" : itemData.get("quantity_per_case"))));
			rtnItem.setItemName1(String.valueOf(itemData.get("item_name1")));
			rtnItem.setItemName2(String.valueOf(itemData.get("item_name2")));
			rtnItem.setUnit(String.valueOf(itemData.get("unit")));
			break;
		}

		return rtnItem;
	}
}
