package jp.ill.photon.module.auth;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.annotation.ModuleVersion;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.LabelUtil;
import jp.ill.photon.util.OutputParamUtil;

/**
 * 
 * 管理者の権限一覧を作成する<br />
 * <br />
 * 入力: <br/>
 * functionList: 権限一覧(List{@literal<Map<String, Object>>})<br/>
 * authorityList: 権限状況(List{@literal<Map<String, Object>>})<br/>
 * <br/>
 * 戻り値: <br />
 * column_list: 権限一覧(Map{@literal<String, Object>})<br />
 * row_list: 権限状況(List{@literal<}Map{@literal<String, Object>>})<br />
 * result_list_name: 権限表示欄タイトル(String)
 * 
 */
@ModuleVersion("1.0.0")
public class MakeAdminAuthorityListModule implements PhotonModule {

	/** 管理者ID */
	@ModuleParam(required = false)
	private String admin_user_id;

	/** 権限一覧 */
	@ModuleParam(required=false)
	private List<Map<String, Object>> functionList;

	/** 権限状況 */
	@ModuleParam(required=false)
	private List<Map<String, Object>> authorityList;
	
	/** 出力項目リンク設定 */
	@ModuleParam(required=false)
	private Map<String, Object> linkInfo;

	/** 出力項目入力形式設定 */
	@ModuleParam(required=false)
	private Map<String, Object> objectInfo;
	
	/** 出力項目表示形式設定 */
	@ModuleParam(required=false)
	private Map<String, Object> dispMap;

	@Override
	public ModuleResult execute(ModuleContext context) {

		// 返却用変数
		Map<String, Object> columnList = new HashMap<String, Object>();
		List<Map<String, Object>> rowList = new ArrayList<Map<String, Object>>();
		String resultListName = "詳細権限設定";
		List<Object> tmpFunctionArray = new ArrayList<Object>();
		Map<String, Object> tmpAuthorityMap = new HashMap<String, Object>();

		if (functionList != null && !functionList.isEmpty()) {
			for (Map<String, Object> adminFunction : functionList) {
				Map<Object, Object> tmpFunctionMap = new HashMap<Object, Object>();
				tmpFunctionMap.put("key", adminFunction.get("admin_function_cd"));
				tmpFunctionMap.put("name", LabelUtil.replaceKeyPattern(adminFunction.get("admin_function_name").toString(), context.getDto()));
				if (admin_user_id == null || admin_user_id.isEmpty()) {
					// 編集可
					tmpFunctionMap.put("object", adminFunction.get("admin_function_cd"));

				} else {
					// 編集不可
					tmpFunctionMap.put("disp_map", dispMap.get("disp_map"));

				}
				tmpFunctionArray.add(tmpFunctionMap);

				if (admin_user_id == null || admin_user_id.isEmpty()) {
					// 新規登録時の権限区分のデフォルトを設定
					tmpAuthorityMap.put(adminFunction.get("admin_function_cd").toString(), "2");

				} else {
					// 新規登録時以外
					tmpAuthorityMap.put(adminFunction.get("admin_function_cd").toString(), "0");

				}

				if (authorityList != null && !authorityList.isEmpty()) {
					for (Map<String, Object> tmpAuthority : authorityList) {
						if (adminFunction.get("admin_function_cd").equals(tmpAuthority.get("admin_function_cd"))) {
							// 権限区分の設定が存在すれば、使用する
							tmpAuthorityMap.replace(tmpAuthority.get("admin_function_cd").toString(), tmpAuthority.get("authority_div"));
							break;
						}
					}
				}
			}
		}

		if (tmpFunctionArray != null && !tmpFunctionArray.isEmpty()) {
			columnList.put("list", tmpFunctionArray);
		}

		if (tmpAuthorityMap != null && !tmpAuthorityMap.isEmpty()) {
			rowList.add(tmpAuthorityMap);
		}

		// 結果を返す
		ModuleResult result = new ModuleResult();
		ArrayList<Object> rtnList = new ArrayList<Object>();
		List<Map<String, Object>> colList = OutputParamUtil.makeColList(columnList, context.getDto());
		if (linkInfo != null) {
			rowList = OutputParamUtil.addLinkInfoToRowList(rowList, linkInfo, context.getDto());
		}
		if (objectInfo != null) {
			rowList = OutputParamUtil.addObjectInfoToRowList(rowList, objectInfo, context.getDto());
		}
		rtnList.add(colList);
		rtnList.add(rowList);
		rtnList.add(resultListName);
		result.getReturnData().put("column_list", colList);
		result.getReturnData().put("row_list", rowList);
		result.getReturnData().put("result_list_name", resultListName);
		result.getReturnData().put("result_list", rtnList);

		return result;
	}

	/**
	 * 権限一覧を取得します。
	 * @return 権限一覧
	 */
	public List<Map<String,Object>> getFunctionList() {
	    return functionList;
	}

	/**
	 * 権限一覧を設定します。
	 * @param functionList 権限一覧
	 */
	public void setFunctionList(List<Map<String,Object>> functionList) {
	    this.functionList = functionList;
	}

	/**
	 * 権限状況を取得します。
	 * @return 権限状況
	 */
	public List<Map<String,Object>> getAuthorityList() {
	    return authorityList;
	}

	/**
	 * 権限状況を設定します。
	 * @param authorityList 権限状況
	 */
	public void setAuthorityList(List<Map<String,Object>> authorityList) {
	    this.authorityList = authorityList;
	}

	/**
	 * 出力項目リンク設定を取得します。
	 * @return 出力項目リンク設定
	 */
	public Map<String,Object> getLinkInfo() {
	    return linkInfo;
	}

	/**
	 * 出力項目リンク設定を設定します。
	 * @param linkInfo 出力項目リンク設定
	 */
	public void setLinkInfo(Map<String,Object> linkInfo) {
	    this.linkInfo = linkInfo;
	}

	/**
	 * 出力項目入力形式設定を取得します。
	 * @return 出力項目入力形式設定
	 */
	public Map<String,Object> getObjectInfo() {
	    return objectInfo;
	}

	/**
	 * 出力項目入力形式設定を設定します。
	 * @param objectInfo 出力項目入力形式設定
	 */
	public void setObjectInfo(Map<String,Object> objectInfo) {
	    this.objectInfo = objectInfo;
	}

	/**
	 * 出力項目表示形式設定を取得します。
	 * @return 出力項目表示形式設定
	 */
	public Map<String,Object> getDispMap() {
	    return dispMap;
	}

	/**
	 * 出力項目表示形式設定を設定します。
	 * @param dispMap 出力項目表示形式設定
	 */
	public void setDispMap(Map<String,Object> dispMap) {
	    this.dispMap = dispMap;
	}

}
