package jp.ill.photon.module.action;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.postgresql.util.PGobject;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.MetaObjectDao;
import jp.ill.photon.dao.MetaObjectDaoImpl;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonFrameworkException;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.exception.PhotonPageNotFoundException;
import jp.ill.photon.model.MapParam;
import jp.ill.photon.module.ModuleContext;
import jp.ill.photon.module.ModuleProcessor;
import jp.ill.photon.module.ModuleResult;
import jp.ill.photon.module.PhotonModule;
import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.ParamUtil;

/**
 * [action_call] 指定したアクションのモジュールリストを取得し、実行して結果dtoを返すモジュール.
 * 
 * <p>
 * 複数アクションで同じような処理（モジュールの組み合わせ）を実行したい場合に、<br/>
 * 別アクションとして定義された共通処理を実行するためのモジュール。
 * </p>
 * 
 * <p>
 * <h2>DTOへの設定値：</h2>
 * 
 * 子アクションが返したdtoのキーと値のマップ。
 * </p>
 * 
 * @author h_tanaka
 *
 */
public class ActionCallModule implements PhotonModule {

	/**
	 * 子アクションのテナントID指定用モジュールパラメータ.
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>param</dd>
	 * <dt>transfer_val</dt>
	 * <dd>_init.tenant_id</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = true)
	private String tenantId;

	/**
	 * 子アクションのアプリケーションID指定用モジュールパラメータ.
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static</dd>
	 * <dt>transfer_val</dt>
	 * <dd>ecuser</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = true)
	private String appId;

	/**
	 * 子アクションのアクションID指定用モジュールパラメータ.
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static</dd>
	 * <dt>transfer_val</dt>
	 * <dd>shohin_list</dd>
	 * </dl>
	 * </p>
	 */
	@ModuleParam(required = true)
	private String actionId;

	/**
	 * 親アクションから子アクションにdtoとして渡すデータ用モジュールパラメータ.
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static_json</dd>
	 * <dt>transfer_val</dt>
	 * <dd>
	 * 
	 * <pre>
	 * {
	 *   "tenantId": {
	 *     "type": "param",
	 *     "val": "_init.tenant_id"
	 *   },
	 *   "formCd": {
	 *     "type": "static",
	 *     "val": "test_val"
	 *   },
	 *   "check_list": {
	 *     "type": "param",
	 *     "val": "checks",
	 *     "data_type": "object_list"
	 *   }
	 * }
	 * </pre>
	 * 
	 * </dd>
	 * </dl>
	 * </p>
	 * 
	 * <p>
	 * 上記の内容を指定すると、SQLファイル内で「tenantId」「formCd」「check_list」というパラメータを「dtoReturnPrefix」で指定したreturn_prefux経由で
	 * dtoから呼び出して使用することができる。
	 * </p>
	 */
	@ModuleParam(required = false)
	private MapParam dataMapValues;

	/**
	 * 子アクションで親アクションから渡されたdtoデータを取得する際に使用するreturn_prefix用モジュールパラメータ.
	 * 
	 * デフォルト値は「parent」.
	 * 
	 * <p>
	 * パラメータ指定例：<br/>
	 * <dl>
	 * <dt>transfer_type</dt>
	 * <dd>static</dd>
	 * <dt>transfer_val</dt>
	 * <dd>parent</dd>
	 * </dl>
	 * 
	 */
	@ModuleParam(required = false)
	private String dataMapReturnPrefix;

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(ActionCallModule.class);

	@Override
	public ModuleResult execute(ModuleContext context)
			throws PhotonModuleException, PhotonPageNotFoundException {
		ModuleResult result = new ModuleResult();

		MetaObjectDao dao = new MetaObjectDaoImpl();
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		List<Map<String, Object>> actionList = tm.required(() -> {
			return dao.selectActionByActionId(tenantId, appId, actionId);
		});

		if (actionList.size() == 0) {
			logger.error("子アクションが取得できませんでした。");
			return result;
		}

		Map<String, Object> action = actionList.get(0);
		List modules = JsonUtil
				.jsonToList(((PGobject) action.get("modules")).getValue());

		ActionDto parentDto = context.getDto();

		Map<String, Object> childAction = new HashMap<>();
		childAction.putAll(parentDto.getAction());
		childAction.put("modules", modules);

		ActionDto dto = new ActionDto();
		dto.setAction(childAction);
		dto.setSession(parentDto.getSession());
		dto.setInitParams(parentDto.getInitParams());
		dto.setRawParams(parentDto.getRawParams());
		dto.setParams(parentDto.getParams());

		// dataParamMapとして受け取ったデータを実値に変換し、dataMapReturnPrefixをキーとしてdataMapを作成する。
		Map<String, Object> dataMapData = createDataMapData(
				getDataMapValues().getParamMap(), parentDto);
		Map<String, Object> dataMap = new HashMap<>();
		dataMap.put(getDataMapReturnPrefix(), dataMapData);

		// commonモジュールのみ、使い勝手のためにそのまま利用可能とする。
		dataMap.put("common", parentDto.getDataMap().getOrDefault("common",
				new HashMap<String, Object>()));

		dto.setDataMap(dataMap);

		ModuleProcessor processor = new ModuleProcessor();
		try {
			dto = processor.processModules(dto, new HashMap<>());
			Map<String, Object> resultDataMap = dto.getDataMap();

			// 親アクションでのDTOの重複や上書きを避けるため、受け取ったDTOの内容は削除する。
			resultDataMap.remove(getDataMapReturnPrefix());
			resultDataMap.remove("common");

			result.getReturnData().putAll(resultDataMap);
			result.setMessages(dto.getMessages());
		} catch (PhotonFrameworkException e) {
			logger.error("子アクションでエラーが発生しました", e);
		}

		return result;
	}

	protected Map<String, Object> createDataMapData(Map<String, Object> srcParams,
													ActionDto dto) {
		Map<String, Object> params = new LinkedHashMap<String, Object>();
		for (Map.Entry<String, Object> param : srcParams.entrySet()) {

			String type = String.valueOf(((Map) param.getValue()).get("type"));
			params.put((String) param.getKey(), ParamUtil.getParamValueByType(
					type, ((Map) param.getValue()).get("val"), dto));
		}

		return params;
	}

	public String getTenantId() {
		return tenantId;
	}

	public void setTenantId(String tenantId) {
		this.tenantId = tenantId;
	}

	public String getAppId() {
		return appId;
	}

	public void setAppId(String appId) {
		this.appId = appId;
	}

	public String getActionId() {
		return actionId;
	}

	public void setActionId(String actionId) {
		this.actionId = actionId;
	}

	public MapParam getDataMapValues() {
		if (dataMapValues == null) {
			dataMapValues = new MapParam();
		}
		return dataMapValues;
	}

	public void setDataMapValues(MapParam dataMapValues) {
		this.dataMapValues = dataMapValues;
	}

	public String getDataMapReturnPrefix() {
		if (dataMapReturnPrefix == null) {
			dataMapReturnPrefix = "parent";
		}
		return dataMapReturnPrefix;
	}

	public void setDataMapReturnPrefix(String dataMapReturnPrefix) {
		this.dataMapReturnPrefix = dataMapReturnPrefix;
	}
}