package jp.ill.photon.module;

import java.io.File;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.annotation.DefaultParamSetting;
import jp.ill.photon.annotation.ModuleParam;
import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.MetaObjectDao;
import jp.ill.photon.dao.MetaObjectDaoImpl;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonFrameworkException;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.exception.PhotonPageNotFoundException;
import jp.ill.photon.util.LogUtil;
import jp.ill.photon.util.MessageUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

/**
 * DTOを元にモジュールを処理するクラス
 * 
 * @author h_tanaka
 *
 */
public class ModuleProcessor {

	/**
	 * 特殊なreturn_prefix管理クラス.
	 * 
	 * @author h_tanaka
	 *
	 */
	public static final class SpecialReturnPrefix {
		/**
		 * returnDataをマップとして扱い、マップの内容とdtoのdataMapをマージする。
		 * （マップ内のキーがreturn_prefixとして扱われる）
		 */
		public static final String EXTEND = "!extend";
	}

	/**
	 * AECActionのモジュールを順次処理する
	 *
	 * @param action
	 * @param params
	 * @param session
	 * @param remoteAddress
	 * @return
	 */
	public ActionDto processModules(ActionDto dto,
									Map<String, List<Map<String, Object>>> fookActions)
			throws PhotonFrameworkException, PhotonModuleException,
			PhotonPageNotFoundException {

		List<Map<String, Object>> actionModules = new ArrayList<>();

		if (fookActions.containsKey("before_process_modules")) {
			List<Map<String, Object>> fookModules = fookActions
					.get("before_process_modules");
			actionModules.addAll(fookModules);
		}

		actionModules.addAll((List) dto.getAction().get("modules"));

		if (fookActions.containsKey("after_process_modules")) {
			List<Map<String, Object>> fookModules = fookActions
					.get("after_process_modules");
			actionModules.addAll(fookModules);
		}

		List<String> moduleIdList = actionModules.stream()
				.map(s -> (String) s.get("module_id"))
				.collect(Collectors.toList());

		Map<String, Object> moduleSettings = ModuleRepository.newInstance()
				.getModuleSettings(moduleIdList);

		Map<String, Object> moduleSetting = null;
		PhotonModule module = null;
		Map<String, Object> convertedParams = null;
		ModuleResult moduleResult = null;
		Map<String, Object> convertedAfterEvent = null;
		String returnPrefix = null;
		for (Map<String, Object> actionModule : actionModules) {
			moduleSetting = (Map) moduleSettings
					.get((String) actionModule.get("module_id"));

			// モジュールオブジェクトを生成
			module = ModuleFactory
					.get((String) moduleSetting.get("module_class"));

			ModuleContext context = new ModuleContext(dto, actionModule);

			// モジュールが実行可能かを判定
			if (module.isExecutable(context,
					(Map) actionModule.get("params"))) {

				// モジュールに渡すパラメータ情報を生成
				convertedParams = convertModuleParams(
						(Map) actionModule.get("params"), dto);

				// ここでモジュールにプロパティを勝手にセット
				try {
					setModuleParams(module, (Map) actionModule.get("params"),
							dto);
				} catch (Exception ex) {
					logger.error("モジュールパラメータセットエラー", ex);
				}

				logger.info(String.format("[MODULE][%s:%s] start",
						StringUtil.defaultString(actionModule.get("module_id"),
								""),
						StringUtil.defaultString(actionModule.get("name"),
								"")));

				// モジュールを実行して結果を取得
				moduleResult = module.execute(context);

				logger.info(String.format("[MODULE][%s:%s] end",
						StringUtil.defaultString(actionModule.get("module_id"),
								""),
						StringUtil.defaultString(actionModule.get("name"),
								"")));

				// モジュールの戻り値をモジュールの戻り値設定に応じて変換してdtoに追加
				returnPrefix = StringUtil
						.defaultString(actionModule.get("return_prefix"), "");
				switch (returnPrefix) {
				case SpecialReturnPrefix.EXTEND:
					dto.putAll(moduleResult.getReturnData());
					break;
				default:
					dto.put(returnPrefix, moduleResult.getReturnData());
				}

				// モジュールでセットされたメッセージを変換してActionDtoに格納
				MessageUtil.buildMessage(moduleResult.getMessages(), dto);
				dto.mergeMessages(moduleResult.getMessages());

				// モジュールの処理後イベント情報に応じて処理後イベントをdtoに設定してbreak|continue。
				convertedAfterEvent = convertModuleAfterEvent(
						(Map) actionModule.get("after"), moduleResult, dto);

				dto.setResultType(
						String.valueOf(convertedAfterEvent.get("result_type")));
				// next_methodプロパティが"get"のときは、パラメータ名をすべてクエリ文字列にして遷移先に渡す
				dto.setNextPath(
						String.valueOf(convertedAfterEvent.get("next_path")));
				dto.setAfterParams(
						(Map) convertedAfterEvent.get("after_params"));
				if ("get".equals(
						convertedAfterEvent.getOrDefault("next_method", ""))) {
					String queryString = dto.getAfterParamQuery();
					dto.setNextPath(dto.getNextPath() + "?" + queryString);
					dto.setAfterParams(null);
				}

				// 結果コードがcontinue以外の場合は以降のモジュールを実行しない
				if (!dto.getResultType().equals("continue")) {
					break;
				}
			}

		}
		return dto;
	}

	/**
	 * アクションの出力モジュールを処理する
	 *
	 * @param dto
	 * @return
	 */
	public ActionDto processOutputModule(ActionDto dto)
			throws PhotonFrameworkException, PhotonModuleException,
			PhotonPageNotFoundException {
		MetaObjectDao dao = new MetaObjectDaoImpl();
		TransactionManager tm = DomaConfig.singleton().getTransactionManager();

		Map<String, Object> returnModule = tm.required(() -> {
			return dao.getModule(
					(String) dto.getAction().get("output_module_id"));
		});
		PhotonModule module = ModuleFactory
				.get((String) returnModule.get("module_class"));
		Map<String, Object> actionOutputParams = (Map) dto.getAction()
				.get("output_module_params");
		Map<String, Object> convertedParams = convertModuleParams(
				actionOutputParams, dto);

		// ここでモジュールにプロパティを勝手にセット
		setModuleParams(module, actionOutputParams, dto);

		ModuleContext context = new ModuleContext(dto);
		dto.put("output", module.execute(context).getReturnData());

		return dto;
	}

	protected void setModuleParams(	PhotonModule module,
									Map<String, Object> moduleParamSettings,
									ActionDto dto)
			throws PhotonFrameworkException {
		Field[] fs = module.getClass().getDeclaredFields();
		for (Field f : fs) {
			ModuleParam param = f.getAnnotation(ModuleParam.class);
			if (param != null) {
				String fieldName = f.getName();
				String fieldNameSnake = ParamUtil.camelToSnake(fieldName);

				if (param.paramGroup()) {
					Field[] paramFs = f.getType().getDeclaredFields();
					for (Field pF : paramFs) {
						String paramFieldName = pF.getName();
						String paramValueName = fieldNameSnake + "."
								+ ParamUtil.camelToSnake(paramFieldName);

						DefaultParamSetting defSetting = pF
								.getAnnotation(DefaultParamSetting.class);
						Object setting = moduleParamSettings
								.get(paramValueName);
						if (setting != null || defSetting != null) {

							// 設定があれば、DefaultParamSettingよりも優先する
							String trasferType = setting != null
									? StringUtil.defaultString(((Map) setting)
											.get("transfer_type"), "")
									: defSetting.transferType();
							Object transferVal = setting != null
									? ((Map) setting).get("transfer_val")
									: defSetting.transferVal();

							Object obj = ParamUtil.getParamValueByType(
									trasferType, transferVal, dto);

							if (obj != null && !"".equals(obj)) {
								Object fieldValue = convertParamObject(obj, pF);

								try {
									String getterMethodName = ParamUtil
											.getGetterName(fieldName);
									Method getterMethod = module.getClass()
											.getMethod(getterMethodName,
													new Class[] {});
									Object paramValue = getterMethod
											.invoke(module);

									String setterMethodName = ParamUtil
											.getSetterName(paramFieldName);
									Method setterMethod = paramValue.getClass()
											.getMethod(setterMethodName,
													new Class[] {
															pF.getType() });
									setterMethod.invoke(paramValue, fieldValue);
								} catch (Exception ex) {
									// ここでエラーが起きるときには、渡されたデータの型がおかしい。違う例外でthrowし直す？
									logger.error("モジュールパラメータグループコンバートエラー", ex);
									throw new PhotonFrameworkException(
											"Type of parameter or name is failed.",
											ex);
								}
							}

						}
					}
				} else {
					DefaultParamSetting defSetting = f
							.getAnnotation(DefaultParamSetting.class);
					Object setting = moduleParamSettings.get(fieldNameSnake);
					if (setting != null || defSetting != null) {

						String trasferType = setting != null
								? StringUtil.defaultString(
										((Map) setting).get("transfer_type"),
										"")
								: defSetting.transferType();
						Object transferVal = setting != null
								? ((Map) setting).get("transfer_val")
								: defSetting.transferVal();

						Object obj = ParamUtil.getParamValueByType(trasferType,
								transferVal, dto);

						if (obj != null && !"".equals(obj)) {
							Object fieldValue = convertParamObject(obj, f);

							try {
								String setterMethodName = ParamUtil
										.getSetterName(fieldName);
								Method setterMethod = module.getClass()
										.getMethod(setterMethodName,
												new Class[] { f.getType() });
								setterMethod.invoke(module, fieldValue);
							} catch (Exception ex) {
								// ここでエラーが起きるときには、渡されたデータの型がおかしい。違う例外でthrowし直す？
								logger.error("モジュールパラメータコンバートエラー", ex);
								throw new PhotonFrameworkException(
										"Type of parameter or name is failed.",
										ex);
							}

						} else {
							// requiredのチェック
							if (param.required()) {
								throw new PhotonFrameworkException(
										"Required parameter is not found."
												+ module.getClass().getName()
												+ ":" + fieldName,
										null);
							}
						}
					}
				}
			}
		}

	}

	private Object convertParamObject(Object obj, Field f)
			throws PhotonFrameworkException {
		Object fieldValue = obj;
		Class clazz = f.getType();

		try {
			// プリミティブ型の変換
			switch (clazz.getName()) {
			case "int":
				fieldValue = Integer.valueOf(StringUtil.defaultString(obj, ""));
				return fieldValue;
			case "java.lang.Integer":
				fieldValue = Integer.valueOf(
						StringUtil.defaultString(obj, "").replace(",", ""));
				return fieldValue;
			case "boolean":
				fieldValue = Boolean.valueOf(StringUtil.defaultString(obj, ""));
				return fieldValue;
			case "java.math.BigDecimal":
				fieldValue = new BigDecimal(
						StringUtil.defaultString(obj, "").replace(",", ""));
				return fieldValue;
			default:
				fieldValue = obj;
			}

		} catch (Exception ex) {
			throw new PhotonFrameworkException(
					"Converting domain object is failed." + ":"
							+ clazz.getName() + ":" + f.getName() + ":" + obj,
					ex);
		}

		if (clazz.getName().startsWith("java.lang.")
				|| clazz.getName().startsWith("java.util.")
				|| clazz.getName().startsWith("[Ljava.lang.")
				|| clazz.getName().startsWith("[B")) {
			return fieldValue;
		}

		// ドメインオブジェクトの変換
		try {
			Method valueOfMethod = null;
			if (obj instanceof List) {
				valueOfMethod = f.getType().getMethod("valueOf",
						new Class[] { List.class });
			} else {
				valueOfMethod = f.getType().getMethod("valueOf",
						new Class[] { Map.class });
			}
			fieldValue = valueOfMethod.invoke(null, obj);
		} catch (Exception ex) {
			throw new PhotonFrameworkException(
					"Converting domain object is failed." + ":"
							+ clazz.getName() + ":" + f.getName(),
					ex);
		}

		return fieldValue;
	}

	/**
	 * モジュールのパラメータ設定に合わせてdtoのデータキー名を変換したパラメータを生成する 設定のないキーは無視する
	 *
	 * @param moduleParams
	 * @param dto
	 * @return
	 */
	protected Map<String, Object> convertModuleParams(	Map<String, Object> moduleParamSettings,
														ActionDto dto) {

		Map<String, Object> convertedParams = new HashMap<>();
		for (Map.Entry<String, Object> moduleParamSetting : moduleParamSettings
				.entrySet()) {
			convertedParams
					.put((String) moduleParamSetting.getKey(),
							ParamUtil.getParamValueByType(
									StringUtil.defaultString(
											((Map) moduleParamSetting
													.getValue())
															.get("transfer_type"),
											""),
									((Map) moduleParamSetting.getValue())
											.get("transfer_val"),
									dto));
		}
		return convertedParams;
	}

	/**
	 * モジュールの処理後イベントを処理後イベント設定とモジュールの処理結果から生成
	 *
	 * @param moduleAfterEventSettings
	 * @param result
	 * @return
	 */
	protected Map<String, Object> convertModuleAfterEvent(	Map<String, Object> moduleAfterEventSettings,
															ModuleResult result,
															ActionDto dto) {
		Map<String, Object> convertedResult = new HashMap<>();

		// モジュールの結果returnCodeが処理後イベント設定に含まれている場合は処理後イベント設定のreturnTypeとnextPathを優先する
		if (moduleAfterEventSettings.containsKey(result.getResultCode())) {
			Map<String, Object> afterSetting = (Map<String, Object>) moduleAfterEventSettings
					.get(result.getResultCode());
			convertedResult.put("result_type",
					(String) afterSetting.get("result_type"));
			convertedResult.put("next_path",
					(String) afterSetting.get("next_path"));
			convertedResult.put("next_method",
					(String) afterSetting.get("next_method"));

			if (afterSetting.containsKey("params")) {
				Map<String, Object> params = convertModuleParams(
						(Map) afterSetting.get("params"), dto);

				Map<String, String[]> convertedParams = new HashMap<String, String[]>();
				for (Map.Entry<String, Object> e : params.entrySet()) {
					if (e.getValue() == null) {
						convertedParams.put(e.getKey(),
								new String[] { String.valueOf(e.getValue()) });
					} else {
						Class clazz = e.getValue().getClass();
						if (clazz.isArray()) {
							convertedParams.put(e.getKey(),
									(String[]) e.getValue());
						} else {
							convertedParams.put(e.getKey(), new String[] {
									String.valueOf(e.getValue()) });
						}
					}
				}

				convertedResult.put("after_params", convertedParams);
			} else {
				convertedResult.put("after_params", null);
			}
		} else {
			// 含まれていない場合はモジュールの結果(returnTypeとnextPath)を採用する
			convertedResult.put("result_type", result.getResultType());
			convertedResult.put("next_path", result.getNextPath());
			Map<String, Object> mp = (Map) result.getReturnData()
					.get("after_params");
			if (mp != null) {
				Map<String, String[]> convertedParams = new HashMap<String, String[]>();
				for (Map.Entry<String, Object> e : mp.entrySet()) {
					if (e.getValue() == null) {
						convertedParams.put(e.getKey(), new String[] { null });
					} else {
						if (e.getValue() instanceof File[]) {
							convertedParams.put(e.getKey(),
									new String[] { null });
						} else {
							Class clazz = e.getValue().getClass();
							if (clazz.isArray()) {
								convertedParams.put(e.getKey(),
										(String[]) e.getValue());
							} else {
								convertedParams.put(e.getKey(), new String[] {
										String.valueOf(e.getValue()) });
							}
						}
					}
				}
				convertedResult.put("after_params", convertedParams);
			} else {
				convertedResult.put("after_params", null);
			}
		}

		return convertedResult;
	}

	/** ログ用変数 */
	protected final LogUtil logger = new LogUtil(ModuleProcessor.class);
}
