package jp.ill.photon.dto;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.servlet.http.HttpSession;

import jp.ill.photon.action.ActionParamMap;
import jp.ill.photon.message.ActionDtoMessage;
import jp.ill.photon.module.ModuleResult;

public class ActionDto {

	private HttpSession session;

	private Map<String, Object> action;

	private Map<String, Object> initParams;

	private Map<String, Object> rawParams;

	private ActionParamMap params;

	private Map<String, Object> dataMap;

	private String resultType;

	private String nextPath;

	private Map<String, Object> afterParams;

	private Map<String, Object> messages;

	/**
	 * 指定したパス（search.shohins.shohin等）のキーが存在するかチェックする
	 *
	 * @param key
	 * @return
	 */
	public boolean containsKey(String key) {
		return pathContainsKey(key);
	}

	/**
	 * 指定したマップパス（キーの1つ上までのパス）にキーが存在するかチェックする
	 *
	 * @param path
	 * @param key
	 * @return
	 */
	public boolean containsKey(String path, String key) {
		Map<String, Object> map = getMapFromPath(path);
		if (!map.containsKey(key)) {
			return false;
		}

		return true;
	}

	/**
	 * 指定したパスに値をセットする
	 *
	 * @param key
	 * @param val
	 */
	public void put(String key, Object val) {
		putValueToPath(key, val);
	}

	/**
	 * マップの内容をデータマップにマージする
	 * 
	 * @param map
	 */
	public void putAll(Map<String, Object> map) {
		map.forEach((k, v) -> {
			putValueToPath(k, v);
		});
	}

	/**
	 * 指定したパスから値を取得する
	 *
	 * @param key
	 * @return
	 */
	public Object get(String key) {
		return getValueFromPath(key);
	}

	/**
	 * 指定したパスから値を取得し、Stringに変換して返す
	 *
	 * @param key
	 * @return
	 */
	public String getString(String key) {
		return String.valueOf(getValueFromPath(key));
	}

	/**
	 * 指定したマップパスのキーに値をセットする
	 *
	 * @param path
	 * @param key
	 * @param val
	 */
	public void put(String path, String key, Object val) {
		Map<String, Object> map = getMapFromPath(path);
		map.put(key, val);
	}

	/**
	 * 指定したマップパスのキーから値を取得する
	 *
	 * @param path
	 * @param key
	 * @return
	 */
	public Object get(String path, String key) {
		Map<String, Object> map = getMapFromPath(path);
		return map.get(key);
	}

	/**
	 * パスを解析してdataMapにキーが存在するかチェックする
	 *
	 * @param path
	 * @return
	 */
	protected boolean pathContainsKey(String path) {
		if (path == null) {
			return false;
		}

		String[] keyTree = path.split("\\.");
		String lastKey = null;
		if (keyTree.length == 0) {
			lastKey = path;
		} else {
			lastKey = keyTree[keyTree.length - 1];
			keyTree = Arrays.copyOf(keyTree, keyTree.length - 1);
		}

		if (lastKey == null) {
			return false;
		}

		Map<String, Object> targetMap = this.getDataMap();
		boolean result = true;
		for (String key : keyTree) {
			if (targetMap == null || !targetMap.containsKey(key)) {
				result = false;
				break;
			}

			targetMap = (Map<String, Object>) targetMap.get(key);
		}

		if (result && !targetMap.containsKey(lastKey)) {
			result = false;
		}

		return result;
	}

	/**
	 * パスを解析して該当のパスに値をセットする
	 *
	 * @param path
	 * @param value
	 */
	protected void putValueToPath(String path, Object value) {
		if (path == null) {
			return;
		}

		String[] keyTree = path.split("\\.");
		String lastKey = null;
		if (keyTree.length == 0) {
			lastKey = path;
		} else {
			lastKey = keyTree[keyTree.length - 1];
			keyTree = Arrays.copyOf(keyTree, keyTree.length - 1);
		}

		if (lastKey == null) {
			return;
		}

		Map<String, Object> targetMap = this.getDataMap();
		for (String key : keyTree) {
			if (targetMap == null) {
				targetMap = new LinkedHashMap<>();
			}

			if (!targetMap.containsKey(key)) {
				targetMap.put(key, new LinkedHashMap<>());
			}

			targetMap = (Map<String, Object>) targetMap.get(key);
		}

		targetMap.put(lastKey, value);
	}

	/**
	 * パスを解析してパスから値を取得する
	 *
	 * @param path
	 * @return
	 */
	protected Object getValueFromPath(String path) {
		if (path == null) {
			return null;
		}

		String[] keyTree = path.split("\\.");
		String lastKey = null;
		if (keyTree.length == 0) {
			lastKey = path;
		} else {
			lastKey = keyTree[keyTree.length - 1];
			keyTree = Arrays.copyOf(keyTree, keyTree.length - 1);
		}

		if (lastKey == null) {
			return null;
		}

		Map<String, Object> targetMap = this.getDataMap();
		for (String key : keyTree) {
			if (targetMap == null) {
				targetMap = new LinkedHashMap<>();
			}

			if (!targetMap.containsKey(key)) {
				targetMap.put(key, new LinkedHashMap<>());
			}

			targetMap = (Map<String, Object>) targetMap.get(key);
		}

		if (targetMap != null && targetMap.containsKey(lastKey)) {
			return targetMap.get(lastKey);
		} else {
			return null;
		}
	}

	/**
	 * マップパスを解析してマップを取得する
	 *
	 * @param path
	 * @return
	 */
	protected Map<String, Object> getMapFromPath(String path) {
		if (path == null) {
			return null;
		}

		String[] keyTree = path.split("\\.");
		if (keyTree.length == 0) {
			keyTree = new String[] { path };
		}
		Map<String, Object> targetMap = this.getDataMap();
		for (String key : keyTree) {
			if (targetMap == null) {
				targetMap = new LinkedHashMap<>();
			}

			if (!targetMap.containsKey(key)) {
				targetMap.put(key, new LinkedHashMap<>());
			}

			targetMap = (Map<String, Object>) targetMap.get(key);
		}

		return targetMap;
	}

	/**
	 * データマップを取得する
	 *
	 * @return
	 */
	public Map<String, Object> getDataMap() {
		if (this.dataMap == null) {
			dataMap = new LinkedHashMap<>();
		}
		return dataMap;
	}

	public void setDataMap(Map<String, Object> dataMap) {
		this.dataMap = dataMap;
	}

	/**
	 * テナントIDを取得する
	 *
	 * @return
	 */
	public String getTenantId() {
		return (String) action.get("tenant_id");
	}

	/**
	 * アプリケーションIDを取得する
	 *
	 * @return
	 */
	public String getAppId() {
		return (String) action.get("app_id");
	}

	/**
	 * アクションIDを取得する
	 *
	 * @return
	 */
	public String getActionId() {
		return (String) action.get("action_id");
	}

	/**
	 * セッションを取得する
	 *
	 * @return
	 */
	public HttpSession getSession() {
		return session;
	}

	/**
	 * セッションをセットする
	 *
	 * @param session
	 */
	public void setSession(HttpSession session) {
		this.session = session;
	}

	public Map<String, Object> getAction() {
		return action;
	}

	public void setAction(Map<String, Object> action) {
		this.action = action;
	}

	public ActionParamMap getParams() {
		return params;
	}

	public void setParams(ActionParamMap params) {
		this.params = params;
	}

	public String getResultType() {
		if (resultType == null) {
			resultType = ModuleResult.ResultTypes.CONTINUE;
		}
		return resultType;
	}

	public void setResultType(String resultType) {
		this.resultType = resultType;
	}

	public String getNextPath() {
		return nextPath;
	}

	public void setNextPath(String nextPath) {
		this.nextPath = nextPath;
	}

	public Map<String, Object> getInitParams() {
		return initParams;
	}

	public void setInitParams(Map<String, Object> initParams) {
		this.initParams = initParams;
	}

	public Map<String, Object> getRawParams() {
		return rawParams;
	}

	public void setRawParams(Map<String, Object> rawParams) {
		this.rawParams = rawParams;
	}

	public Map<String, Object> getAfterParams() {
		if (afterParams == null) {
			afterParams = new LinkedHashMap<>();
		}
		return afterParams;
	}

	public void setAfterParams(Map<String, Object> afterParams) {
		this.afterParams = afterParams;
	}

	public Map<String, Object> getMessages() {
		return messages;
	}

	public void setMessages(Map<String, Object> messages) {
		this.messages = messages;
	}

	/**
	 * 処理後パラメータを元にリダイレクト用クエリ文字列を生成して返す
	 *
	 * @return
	 */
	public String getAfterParamQuery() {
		Map<String, Object> paramMap = getAfterParams();

		List<String> paramQuery = new ArrayList<>();
		for (Map.Entry<String, Object> param : paramMap.entrySet()) {
			if (param.getValue() instanceof String[]) {
				paramQuery.addAll(Arrays.stream((String[]) param.getValue())
						.map(s -> param.getKey() + "=" + s)
						.collect(Collectors.toList()));
			} else {
				paramQuery.add(param.getKey() + "="
						+ String.valueOf(param.getValue()));
			}
		}

		return paramQuery.stream().collect(Collectors.joining("&"));
	}

	/**
	 * リダイレクト・フォワード用の絶対URLを生成して返す
	 *
	 * @return
	 */
	public String getNextFullUrl() {
		String nextUrl = getNextPath();
		if (nextUrl != null && !nextUrl.startsWith("http")) {
			StringBuffer u = new StringBuffer();
			u.append(getParams().get("_init.context_url"));
			u.append(nextUrl);
			nextUrl = u.toString();
		}

		return nextUrl;
	}

	/**
	 * メッセージをマージする
	 *
	 * @param messages
	 *
	 */
	@SuppressWarnings("unchecked")
	public void mergeMessages(Map<String, Object> messages) {

		Map<String, Object> newMap = new HashMap<String, Object>();

		if (this.messages != null) {
			for (Map.Entry<String, Object> e : this.messages.entrySet()) {
				newMap.put(e.getKey(), e.getValue());
			}
		}
		if (messages != null) {
			for (Map.Entry<String, Object> e : messages.entrySet()) {
				List<ActionDtoMessage> messageList = (List<ActionDtoMessage>) newMap
						.get(e.getKey());
				if (messageList == null) {
					messageList = new ArrayList<ActionDtoMessage>();
				}
				List<ActionDtoMessage> addMessageList = (List<ActionDtoMessage>) e
						.getValue();
				if (addMessageList != null) {
					messageList.addAll(addMessageList);
				}
				newMap.put(e.getKey(), messageList);
			}
		}

		this.messages = newMap;

	}

	/**
	 *
	 * 指定されたキーを持つメッセージリストを取得
	 *
	 * @param key
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public List<ActionDtoMessage> getSpecifiedMessageList(String key) {
		List<ActionDtoMessage> ret = new ArrayList<ActionDtoMessage>();
		if (this.messages != null) {
			for (Map.Entry<String, Object> e : this.messages.entrySet()) {
				if (key.equals(e.getKey())) {
					ret.addAll((List<ActionDtoMessage>) this.messages.get(key));
				}
			}
		}
		return ret;
	}

	/**
	 *
	 * 指定されたキーとフィールドコードを持つメッセージリストを取得
	 *
	 * @param key
	 * @param fieldCd
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public List<ActionDtoMessage> getSpecifiedMessageList(	String key,
															String fieldCd) {
		List<ActionDtoMessage> ret = new ArrayList<ActionDtoMessage>();
		if (this.messages != null) {
			for (Map.Entry<String, Object> e : this.messages.entrySet()) {
				if (key.equals(e.getKey())) {
					List<ActionDtoMessage> list = (List<ActionDtoMessage>) this.messages
							.get(key);
					if (list != null) {
						for (ActionDtoMessage item : list) {
							if (fieldCd != null
									&& fieldCd.equals(item.getFormFieldCd())) {
								ret.add(item);
							}
						}
					}
				}
			}
		}
		return ret;
	}

	/**
	 *
	 * 指定されたキーとフィールドコードを持つメッセージリストを取得
	 *
	 * @param key
	 * @param fieldCd
	 * @return
	 */
	public List<ActionDtoMessage> getSpecifiedMessageListByMultiFieldCd(String key,
																		List<String> fieldCdList) {

		List<ActionDtoMessage> ret = new ArrayList<ActionDtoMessage>();

		if (fieldCdList != null) {

			for (String fieldCd : fieldCdList) {

				ret.addAll(getSpecifiedMessageList(key, fieldCd));

			}

		}

		return ret;

	}

	public Map<String, Object> getMappedMessages() {
		Map<String, Object> mappedMessages = new HashMap<>();
		if (this.messages != null) {
			List<ActionDtoMessage> list = null;
			List<Map<String, Object>> newList = null;
			for (Map.Entry<String, Object> e : this.messages.entrySet()) {
				newList = new ArrayList<>();
				mappedMessages.put(e.getKey(), newList);
				list = (List<ActionDtoMessage>) this.messages.get(e.getKey());
				if (list != null) {
					for (ActionDtoMessage item : list) {
						newList.add(item.toMap());
					}
				}
			}
		}

		return mappedMessages;
	}
}
