package jp.ill.photon.doma;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.map.HashedMap;
import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.FetchType;
import org.seasar.doma.internal.jdbc.sql.SqlParser;
import org.seasar.doma.internal.jdbc.sql.node.BindVariableNode;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.SqlNode;
import org.seasar.doma.jdbc.query.SqlSelectQuery;

import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.util.StringUtil;

/**
 * SQLとパラメータを元にDomaのSqlSelectQueryオブジェクトを生成するファクトリクラス
 *
 * @author h_tanaka
 *
 */
public class SqlSelectQueryFactory extends AbstractSqlQueryFactory {

	public static SqlSelectQueryFactory newInstance() {
		return new SqlSelectQueryFactory();
	}

	/**
	 * SQLファイルパスとパラメータを元にDomaのSqlSelectQueryオブジェクトを生成する
	 *
	 * @param srcParams
	 * @param dto
	 * @param sqlFileDirPath
	 * @param sqlFilePath
	 * @return
	 * @throws PhotonModuleException
	 */
	public SqlSelectQuery createSelectQueryFromFile(Map<String, Object> srcParams,
													ActionDto dto,
													String sqlFileDirPath,
													String sqlFilePath)
			throws PhotonModuleException {
		String sql = readSqlFile(sqlFileDirPath, sqlFilePath);
		return createSelectQuery(srcParams, dto, sql);
	}

	/**
	 * SQLテキストとパラメータを元にDomaのSqlSelectQueryオブジェクトを生成する
	 *
	 * @param srcParams
	 * @param dto
	 * @param sql
	 * @return
	 * @throws PhotonModuleException
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public SqlSelectQuery createSelectQuery(Map<String, Object> srcParams,
											ActionDto dto,
											String sql)
			throws PhotonModuleException {
		// パラメータデータを取得
		Map<String, Object> params = createSqlParams(srcParams, dto);
		Map<String, Object> paramTypes = createSqlParamTypes(srcParams);

		// SQLを読み込んでDomaのSQLParserで解析
		SqlParser parser = createSqlParser(sql);

		// DomaのクエリオブジェクトにSQLをセットする
		SqlSelectQuery selectQuery = new SqlSelectQuery();
		selectQuery.setConfig(DomaConfig.singleton());
		selectQuery.setCallerClassName(getClass().getName());
		selectQuery.setCallerMethodName("execute");
		selectQuery.setFetchType(FetchType.LAZY);
		selectQuery.setSqlLogType(SqlLogType.FORMATTED);
		// selectQuery.setSqlNode(parser.parse());
		SqlNode sn = parser.parse();
		selectQuery.setSqlNode(sn);

		// SQL文の中からパラメータを取得
		List<String> paramFromSqlFile = new ArrayList<String>();
		getVariableName(sn, paramFromSqlFile);

		// paramに不足分をマージする
		for (String p : paramFromSqlFile) {
			if (!params.containsKey(p)) {
				params.put(p, null);
				paramTypes.put(p, new HashedMap() {
					{
						put("type", "");
						put("data_type", ""); // データタイプは、強制的に単一テキストとなる
					}
				});
			}
		}

		// パラメータをクエリオブジェクトにセットする
		for (Map.Entry<String, Object> entry : params.entrySet()) {

			Map<String, String> typeDataTypeMap = (Map<String, String>) paramTypes
					.get(entry.getKey());
			String type = typeDataTypeMap.get("type");
			String dataType = typeDataTypeMap.get("data_type");

			Class clazz = null;
			Object defValue = null;
			Object setValue = null;
			if ("param_multi".equals(type)) {

				clazz = List.class;
				String[] strArr = (String[]) entry.getValue();

				if (DataTypes.NUMBER.equals(dataType)) {
					List<Integer> list = new ArrayList<Integer>();
					for (String item : strArr) {
						list.add(Integer.parseInt(item));
					}
					setValue = list;
				} else {
					List<String> list = new ArrayList<String>();
					for (String item : strArr) {
						list.add(item);
					}
					setValue = list;
				}

			} else {

				if (DataTypes.NUMBER.equals(dataType)) {
					clazz = Integer.class;
					setValue = (entry.getValue() == null) ? null
							: Integer.parseInt(
									StringUtil.defaultString(entry.getValue()));
				} else if (DataTypes.TEXT_LIST.equals(dataType)) {
					clazz = List.class;
					List<String> list = new ArrayList<String>();
					if (entry.getValue() != null) {
						for (String item : (List<String>) entry.getValue()) {
							list.add(item);
						}
					}
					setValue = list;
				} else if (DataTypes.OBJECT_LIST.equals(dataType)) {
					clazz = List.class;
					List<Object> list = new ArrayList<Object>();
					if (entry.getValue() != null) {
						for (Object item : (List<Object>) entry.getValue()) {
							list.add(item);
						}
					}
					setValue = list;
				} else if (DataTypes.TEXT_MAP.equals(dataType)) {
					clazz = Map.class;
					Map<String, String> map = new LinkedHashMap<>();
					if (entry.getValue() != null) {
						for (Map.Entry<String, String> item : ((Map<String, String>) entry
								.getValue()).entrySet()) {
							map.put(item.getKey(), item.getValue());
						}
					}
					setValue = map;
				} else if (DataTypes.OBJECT_MAP.equals(dataType)) {
					clazz = Map.class;
					Map<String, Object> map = new LinkedHashMap<>();
					if (entry.getValue() != null) {
						for (Map.Entry<String, Object> item : ((Map<String, Object>) entry
								.getValue()).entrySet()) {
							map.put(item.getKey(), item.getValue());
						}
					}
					setValue = map;
				} else {
					defValue = "";
					clazz = String.class;
					setValue = StringUtil.defaultString(entry.getValue(), "");
				}

			}

			selectQuery.addParameter(entry.getKey(), clazz,
					(entry.getValue() == null ? defValue : setValue));

		}

		return selectQuery;

	}

	/***
	 *
	 * SQL文の中に埋め込まれた変数名を取得
	 *
	 * @param sn sqlノード
	 * @param ret SQL文の中に埋め込まれた変数名のリスト
	 *
	 */
	public void getVariableName(SqlNode sn, List<String> ret) {
		if (BindVariableNode.class.equals(sn.getClass())) {
			BindVariableNode bvn = (BindVariableNode) sn;
			String name = bvn.getVariableName();
			if (!StringUtils.isEmpty(bvn.getVariableName())) {
				if (!ret.contains(name)) {
					ret.add(name);
				}
			}
		}
		List<SqlNode> list = sn.getChildren();
		if (list.size() >= 1) {
			for (SqlNode child : list) {
				getVariableName(child, ret);
			}
		}

	}

}
