package jp.ill.photon.doma;

import java.sql.SQLException;
import java.util.function.Supplier;

import org.seasar.doma.internal.util.AssertionUtil;
import org.seasar.doma.jdbc.JdbcLogger;
import org.seasar.doma.jdbc.Sql;
import org.seasar.doma.jdbc.SqlExecutionSkipCause;
import org.seasar.doma.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Slf4jJdbcLogger implements JdbcLogger {

	/**
	 * Loggerインスタンス。
	 */
	protected Logger log;
	private String defaultLevel = "info";

	public Slf4jJdbcLogger() {
		log = LoggerFactory.getLogger(Slf4jJdbcLogger.class);
	}

	protected void log(	String level,
						String callerClassName,
						String callerMethodName,
						Throwable throwable,
						Supplier<String> messageSupplier) {
		StringBuilder logMessage = new StringBuilder();
		logMessage.append(callerClassName).append(" ");
		logMessage.append(callerMethodName).append(" ");
		logMessage.append(messageSupplier.get());
		switch (level) {
		case "debug":
			log.debug(logMessage.toString());
			break;
		case "info":
			log.info(logMessage.toString());
			break;
		case "warn":
			log.warn(logMessage.toString());
			break;
		case "error":
			log.error(logMessage.toString());
			break;
		default:
			log.info(logMessage.toString());
			break;
		}

	}

	@Override
	public void logDaoMethodEntering(	String callerClassName,
										String callerMethodName,
										Object... args) {
		log("debug", callerClassName, callerMethodName, null,
				() -> Message.DOMA2220.getMessage(callerClassName,
						callerMethodName));
	}

	@Override
	public void logDaoMethodExiting(String callerClassName,
									String callerMethodName,
									Object result) {
		log("debug", callerClassName, callerMethodName, null,
				() -> Message.DOMA2221.getMessage(callerClassName,
						callerMethodName));
	}

	@Override
	public void logDaoMethodThrowing(	String callerClassName,
										String callerMethodName,
										RuntimeException e) {
		log(defaultLevel, callerClassName, callerMethodName, null,
				() -> Message.DOMA2222.getMessage(callerClassName,
						callerMethodName, e.getClass().getName()));
	}

	@Override
	public void logSqlExecutionSkipping(String callerClassName,
										String callerMethodName,
										SqlExecutionSkipCause cause) {
		log(defaultLevel, callerClassName, callerMethodName, null,
				() -> Message.DOMA2223.getMessage(callerClassName,
						callerMethodName, cause.name()));
	}

	@Override
	public void logSql(	String callerClassName,
						String callerMethodName,
						Sql<?> sql) {
		log("debug", callerClassName, callerMethodName, null,
				() -> Message.DOMA2076.getMessage(sql.getSqlFilePath(),
						getSqlText(sql)));
	}

	protected String getSqlText(Sql<?> sql) {
		switch (sql.getSqlLogType()) {
		case RAW:
			return sql.getRawSql();
		case FORMATTED:
			return sql.getFormattedSql();
		case NONE:
			return "";
		}
		return AssertionUtil.assertUnreachable();
	}

	@Override
	public void logTransactionBegun(String callerClassName,
									String callerMethodName,
									String transactionId) {
		log("debug", callerClassName, callerMethodName, null,
				() -> Message.DOMA2063.getMessage(transactionId));
	}

	@Override
	public void logTransactionEnded(String callerClassName,
									String callerMethodName,
									String transactionId) {
		log("debug", callerClassName, callerMethodName, null,
				() -> Message.DOMA2064.getMessage(transactionId));
	}

	@Override
	public void logTransactionSavepointCreated(	String callerClassName,
												String callerMethodName,
												String transactionId,
												String savepointName) {
		log("debug", callerClassName, callerMethodName, null,
				() -> Message.DOMA2065.getMessage(transactionId,
						savepointName));
	}

	@Override
	public void logTransactionCommitted(String callerClassName,
										String callerMethodName,
										String transactionId) {
		log("debug", callerClassName, callerMethodName, null,
				() -> Message.DOMA2067.getMessage(transactionId));
	}

	@Override
	public void logTransactionRolledback(	String callerClassName,
											String callerMethodName,
											String transactionId) {
		log("warn", callerClassName, callerMethodName, null,
				() -> Message.DOMA2068.getMessage(transactionId));
	}

	@Override
	public void logTransactionSavepointRolledback(	String callerClassName,
													String callerMethodName,
													String transactionId,
													String savepointName) {
		log("warn", callerClassName, callerMethodName, null,
				() -> Message.DOMA2069.getMessage(transactionId,
						savepointName));
	}

	@Override
	public void logTransactionRollbackFailure(	String callerClassName,
												String callerMethodName,
												String transactionId,
												SQLException e) {
		log("warn", callerClassName, callerMethodName, null,
				() -> Message.DOMA2070.getMessage(transactionId));
	}

	@Override
	public void logAutoCommitEnablingFailure(	String callerClassName,
												String callerMethodName,
												SQLException e) {
		log("warn", callerClassName, callerMethodName, null,
				() -> Message.DOMA2071.getMessage());
	}

	@Override
	public void logTransactionIsolationSettingFailure(	String callerClassName,
														String callerMethodName,
														int transactionIsolationLevel,
														SQLException e) {
		log("warn", callerClassName, callerMethodName, null,
				() -> Message.DOMA2072.getMessage(transactionIsolationLevel));
	}

	@Override
	public void logConnectionClosingFailure(String callerClassName,
											String callerMethodName,
											SQLException e) {
		log("warn", callerClassName, callerMethodName, null,
				() -> Message.DOMA2073.getMessage());
	}

	@Override
	public void logStatementClosingFailure(	String callerClassName,
											String callerMethodName,
											SQLException e) {
		log("warn", callerClassName, callerMethodName, null,
				() -> Message.DOMA2074.getMessage());
	}

	@Override
	public void logResultSetClosingFailure(	String callerClassName,
											String callerMethodName,
											SQLException e) {
		log("warn", callerClassName, callerMethodName, null,
				() -> Message.DOMA2075.getMessage());
	}
}
