package jp.ill.photon.doma;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.seasar.doma.jdbc.dialect.PostgresDialect.PostgresExpressionFunctions;

import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ListUtil;
import jp.ill.photon.util.StringUtil;

/**
 * Doma SQL内で利用可能な関数群
 *
 * @author h_tanaka
 *
 */
public class CustomExpressions extends PostgresExpressionFunctions {

	/**
	 * オブジェクトをString型に変換する
	 *
	 * @param o
	 * @return
	 */
	public String toStr(Object o) {
		if (o == null) {
			return null;
		}

		return StringUtil.defaultString(o, "");
	}

	/**
	 * 文字列をInteger型に変換
	 *
	 * @param obj オブジェクト
	 * @param defaultDcm デフォルト値
	 * @return Integer(数値以外:デフォルト値)
	 */
	public Integer toInt(Object obj, Integer defaultValue) {
		if (obj == null) {
			return defaultValue;
		}
		try {
			return Integer.parseInt((String) obj);
		} catch (Exception e) {
			return defaultValue;
		}
	}

	public List toList(Object obj) {
		return (List) obj;
	}

	/**
	 * 文字列をInteger型に変換
	 *
	 * @param obj オブジェクト
	 * @return Integer(数値以外:0)
	 */
	public Integer toInt(Object obj) {
		return toInt(obj, 0);
	}

	/**
	 * 文字列をDouble型に変換
	 *
	 * @param obj オブジェクト
	 * @param defaultDcm デフォルト値
	 * @return Integer(数値以外:デフォルト値)
	 */
	public Double toDouble(Object obj, Double defaultValue) {
		if (obj == null) {
			return defaultValue;
		}
		try {
			return Double.parseDouble((String) obj);
		} catch (Exception e) {
			return defaultValue;
		}
	}

	/**
	 * 文字列をDouble型に変換
	 *
	 * @param obj オブジェクト
	 * @return Double(数値以外:0)
	 */
	public Double toDouble(Object obj) {
		return toDouble(obj, 0.0);
	}

	/**
	 * Like検索用に前後に%を追加
	 *
	 * @param src
	 * @return
	 */
	public String infix(Object src) {
		StringBuffer sb = new StringBuffer();
		sb.append("%");
		sb.append(src);
		sb.append("%");

		return sb.toString();
	}

	/**
	 * クラス型をチェック
	 *
	 * @param src
	 * @param type
	 * @return
	 */
	public boolean isInstanceOf(Object src, String type) {
		Class<?> clazz = null;
		try {
			clazz = Class.forName(type);
		} catch (ClassNotFoundException e) {
		}

		if (clazz == null) {
			return false;
		}

		return clazz.isInstance(src);
	}

	/**
	 * カンマ区切りテキストに値が含まれるかチェック
	 *
	 * @param src
	 * @param dataInputTypes
	 * @return
	 */
	public boolean isIn(Object src, String dataInputTypes) {

		if (dataInputTypes == null || dataInputTypes.length() == 0) {
			return false;
		}

		String[] dataInputTypeList = dataInputTypes.split(",");
		if (Arrays.asList(dataInputTypeList).contains(src)) {
			return true;
		}

		return false;
	}

	/**
	 * 浮動小数チェック
	 *
	 * @param str
	 * @return boolean (true:doubleに変換できる false:doubleに変換できない)
	 **/
	public boolean isDouble(Object obj) {
		boolean result = false;
		try {
			Double.parseDouble(String.valueOf(obj));
			result = true;
		} catch (Exception e) {
		}
		return result;
	}

	/**
	 * 正規表現置換する
	 *
	 * @param obj
	 * @param regex
	 * @param replacement
	 * @return
	 */
	public String replace(Object obj, String regex, String replacement) {
		if (obj == null) {
			return null;
		}

		String str = String.valueOf(obj);
		return str.replaceAll(regex, replacement);
	}

	/**
	 * JavaのマップをJSONテキストに変換して返す
	 *
	 */
	public String mapToJson(Object obj) {
		if (obj == null) {
			return "{}";
		}

		return JsonUtil.mapToJson(obj);
	}

	/**
	 * JavaのリストをJSONテキストに変換して返す
	 *
	 */
	public String listToJson(Object obj) {
		if (obj == null) {
			return "[]";
		}

		return JsonUtil.mapToJson(obj);
	}

	public List<Integer> rangeClosed(int start, int end) {
		return ListUtil.rangeClosed(start, end);
	}

	private Map<String, String> varMap;

	public boolean setVar(String name, String val) {
		if (varMap == null) {
			varMap = new HashMap<>();
		}
		varMap.put(name, val);
		return true;
	}

	public String getVar(String name) {
		if (varMap == null) {
			varMap = new HashMap<>();
		}

		return varMap.getOrDefault(name, "");
	}

	public List<String> listMapToList(	List<Map<String, String>> arg,
										String key) {
		return arg.stream()
				.map(item -> item.get(key))
				.collect(Collectors.toList());
	}
}
