package jp.ill.photon.doma;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.seasar.doma.internal.jdbc.sql.SqlParser;

import jp.ill.photon.dto.ActionDto;
import jp.ill.photon.exception.PhotonModuleException;
import jp.ill.photon.util.ParamUtil;

public abstract class AbstractSqlQueryFactory {

	/** データタイプ */
	public static final class DataTypes {
		public static final String TEXT = "text";
		public static final String NUMBER = "number";
		public static final String TEXT_LIST = "text_list";
		public static final String OBJECT_LIST = "object_list";
		public static final String TEXT_MAP = "text_map";
		public static final String OBJECT_MAP = "object_map";
	}

	/**
	 * SQLファイルを解析してSQLパーサーオブジェクトを生成
	 *
	 * @return
	 * @throws PhotonModuleException
	 */
	protected SqlParser createSqlParser(String sqlFileDir, String sqlFilePath)
			throws PhotonModuleException {
		String sql = readSqlFile(sqlFileDir, sqlFilePath);

		// SQLを読み込んでDomaのSQLParserで解析
		return new SqlParser(sql);
	}

	/**
	 * SQLテキストを元にSQLパーサーオブジェクトを生成
	 *
	 * @param sql
	 * @return
	 */
	protected SqlParser createSqlParser(String sql) {
		return new SqlParser(sql);
	}

	/**
	 * SQLファイルの内容を読み込んで返す
	 *
	 * @param sqlFileDir
	 * @param sqlFilePath
	 * @return
	 * @throws PhotonModuleException
	 */
	public String readSqlFile(String sqlFileDir, String sqlFilePath)
			throws PhotonModuleException {
		if (sqlFileDir == null || sqlFilePath == null) {
			throw new PhotonModuleException("SQLファイルが指定されていません", null);
		}

		// SQLファイル存在チェック
		Path path = Paths.get(sqlFileDir, sqlFilePath);
		if (!Files.exists(path)) {
			throw new PhotonModuleException("SQLファイルが存在しません:" + path.toString(),
					null);
		}

		// SQLを読み込み
		String sql;
		try {
			sql = Files.readAllLines(path, Charset.forName("UTF-8")).stream()
					.collect(Collectors.joining("\n"));
		} catch (IOException e) {
			throw new PhotonModuleException("SQLの読込に失敗しました:" + path.toString(),
					e);
		}

		return sql;
	}

	/**
	 * SQLファイルのパラメータデータを生成する
	 *
	 * @param srcParams
	 * @param context
	 * @return
	 */
	@SuppressWarnings({ "rawtypes" })
	protected Map<String, Object> createSqlParams(	Map<String, Object> srcParams,
													ActionDto dto) {
		// パラメータデータを取得
		Map<String, Object> params = new LinkedHashMap<String, Object>();
		for (Map.Entry<String, Object> param : srcParams.entrySet()) {

			String type = String.valueOf(((Map) param.getValue()).get("type"));
			params.put((String) param.getKey(), ParamUtil.getParamValueByType(
					type, ((Map) param.getValue()).get("val"), dto));
		}

		return params;
	}

	/**
	 * SQLファイルのパラメータ別タイプのマップを生成する
	 *
	 * @param srcParams
	 * @param context
	 * @return
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	protected Map<String, Object> createSqlParamTypes(Map<String, Object> srcParams) {
		Map<String, Object> paramTypes = new LinkedHashMap<String, Object>();
		String type = null;
		String dataType = null;
		Map<String, String> typeDataTypeMap = null;
		for (Map.Entry<String, Object> param : srcParams.entrySet()) {

			type = String.valueOf(((Map) param.getValue()).get("type"));
			dataType = (String) ((Map) param.getValue())
					.getOrDefault("data_type", "");
			if (StringUtils.isEmpty(dataType)) {
				dataType = DataTypes.TEXT;
			}

			typeDataTypeMap = new HashMap<String, String>();
			typeDataTypeMap.put("type", type);
			typeDataTypeMap.put("data_type", dataType);
			paramTypes.put((String) param.getKey(), typeDataTypeMap);

		}

		return paramTypes;
	}
}
