package jp.ill.photon.dao.builder;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.postgresql.util.PGobject;
import org.seasar.doma.jdbc.tx.TransactionManager;

import jp.ill.photon.dao.DomaConfig;
import jp.ill.photon.dao.JsonDataDao;
import jp.ill.photon.dao.JsonDataDaoImpl;
import jp.ill.photon.util.JsonUtil;

/**
 * シンプルなSQLを生成して実行するヘルパクラス
 * 
 * @author h_tanaka
 *
 */
public class SimpleJSONSelectSQLBuilder implements SimpleSQLBuilder {

	private JsonDataDao dao = null;

	/**
	 * Domaクエリビルダ用にSQLデータを生成する
	 * 
	 * @param tenantId
	 * @param editTableCd
	 * @param where
	 * @param order
	 * @return
	 */
	protected List<QueryPiece> build(	String tenantId,
										String editTableCd,
										List<Map<String, Object>> where,
										List<Map<String, String>> order) {
		if (editTableCd == null) {
			return new ArrayList<>();
		}

		List<QueryPiece> query = new ArrayList<>();
		query.add(new SqlQueryPiece("select count(*) OVER() as count, val"));
		query.add(new SqlQueryPiece("from entity_data"));
		query.add(new SqlQueryPiece("where"));
		query.add(new SqlQueryPiece("tenant_id ="));
		query.add(new ParamQueryPiece(String.class, tenantId));
		query.add(new SqlQueryPiece("and"));
		query.add(new SqlQueryPiece("edit_table_cd ="));
		query.add(new ParamQueryPiece(String.class, editTableCd));

		if (where != null && where.size() > 0) {
			for (Map<String, Object> q : where) {
				query.add(new SqlQueryPiece("and"));
				query.add(new SqlQueryPiece(createJsonColName(
						(String) q.get("name"), (Class<?>) q.get("class"))));
				if (q.containsKey("op")) {
					query.add(new SqlQueryPiece((String) q.get("op")));
				} else {
					query.add(new SqlQueryPiece("="));
				}
				query.add(new ParamQueryPiece((Class<?>) q.get("class"),
						q.get("value")));
			}
		}

		if (order != null && order.size() > 0) {
			query.add(new SqlQueryPiece("order by"));
			for (Map<String, String> o : order) {
				query.add(new SqlQueryPiece(
						createJsonColName(o.get("key"), null)));
				query.add(new SqlQueryPiece(o.get("dr")));
			}
		}

		return query;
	}

	/**
	 * 指定条件でSQLを生成して結果をマップのリストで返す
	 * 
	 * @param tenantId
	 * @param editTableCd
	 * @param where
	 * @param order
	 * @return
	 */
	public List<Map<String, Object>> getListResult(	String tenantId,
													String editTableCd,
													List<Map<String, Object>> where,
													List<Map<String, String>> order) {
		List<QueryPiece> query = build(tenantId, editTableCd, where, order);

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = getJsonDao();

		List<Map<String, Object>> result = tm.required(() -> {
			return dao.selectList(query);
		});

		List<Map<String, Object>> resultList = new ArrayList<>();
		Map<String, Object> jsonValue = null;
		for (Map<String, Object> row : result) {
			jsonValue = JsonUtil
					.jsonToMap(((PGobject) row.get("val")).getValue());
			resultList.add(jsonValue);
		}

		return resultList;
	}

	/**
	 * 指定条件でSQLを生成して結果を単一マップで返す
	 * 
	 * @param tenantId
	 * @param editTableCd
	 * @param where
	 * @param order
	 * @return
	 */
	public Map<String, Object> getFirstResult(	String tenantId,
												String editTableCd,
												List<Map<String, Object>> where,
												List<Map<String, String>> order) {
		List<QueryPiece> query = build(tenantId, editTableCd, where, order);

		TransactionManager tm = DomaConfig.singleton().getTransactionManager();
		JsonDataDao dao = getJsonDao();

		Map<String, Object> result = tm.required(() -> {
			return dao.selectOne(query);
		});

		Map<String, Object> jsonValue = null;
		if (result != null) {
			jsonValue = JsonUtil
					.jsonToMap(((PGobject) result.get("val")).getValue());
		}

		return jsonValue;
	}

	protected JsonDataDao getJsonDao() {
		if (this.dao == null) {
			dao = new JsonDataDaoImpl();
		}
		return dao;
	}

}
