package jp.ill.photon.dao.builder;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

import jp.ill.photon.dto.ActionDto;

public abstract class AbstractDomaSQLQueryBuilder {

	public abstract List<Map<String, Object>> getListResult(String query,
															Map<String, Object> params,
															ActionDto dto);

	public abstract Map<String, Object> getFirstResult(	String query,
														Map<String, Object> params,
														ActionDto dto);

	public abstract int insert(	String query,
								Map<String, Object> params,
								ActionDto dto);

	public abstract int update(	String query,
								Map<String, Object> params,
								ActionDto dto);

	public abstract int delete(	String query,
								Map<String, Object> params,
								ActionDto dto);

	/**
	 * クエリ文字列をパースしてDoma用クエリに変換する
	 * 
	 * @param tenantId
	 * @param query
	 * @param params
	 * @return
	 */
	protected List<QueryPiece> build(	String query,
										Map<String, Object> params,
										ActionDto dto) {

		String convertedQuery = StringUtils.replaceEach(query,
				new String[] { "{#", "#}" }, new String[] { ":::#", ":::" });
		String[] queryPieces = convertedQuery.split(":::");

		List<QueryPiece> domaQuery = new ArrayList<>();
		for (String piece : queryPieces) {
			if (piece.startsWith("#")) {
				piece = piece.replaceFirst("#", "");
				domaQuery.addAll(createParamQueryPiece(piece, params, dto));
			} else {
				domaQuery.add(new SqlQueryPiece(piece));
			}
		}

		return domaQuery;
	}

	/**
	 * パラメータクエリ部品を生成する
	 * 
	 * @param queryPiece
	 * @param params
	 * @return
	 */
	protected List<QueryPiece> createParamQueryPiece(	String queryPiece,
														Map<String, Object> params,
														ActionDto dto) {
		String[] pieces = queryPiece.split(",");

		// String val = null;
		Object val = null;

		String qVal = null;
		String qType = null;
		if (pieces.length >= 3) {
			qVal = pieces[0].trim();
			qType = pieces[2].trim();
			if (qType.equals("param") && params.containsKey(qVal)) {
				// val = String.valueOf(params.get(qVal));
				val = params.get(qVal);
			} else if (qType.equals("dto") && dto.containsKey(qVal)) {
				// val = String.valueOf(dto.get(qVal));
				val = dto.get(qVal);
			} else if (qType.equals("action")
					&& dto.getAction().containsKey(qVal)) {
				// val = String.valueOf(dto.getAction().get(qVal));
				val = dto.getAction().get(qVal);
			}
		}
		if (val == null) {
			val = pieces[0].trim();
		}

		String typeName = "text";
		if (pieces.length >= 2) {
			typeName = pieces[1].trim();
		}

		// JSONB型用にキャスト構文を追加
		if (typeName.equals("jsonb")) {
			List<QueryPiece> query = new ArrayList<>();
			query.add(createParamQueryPiece(typeName, val));
			query.add(new SqlQueryPiece("::jsonb"));
			return query;
		}
		return Arrays.asList(createParamQueryPiece(typeName, val));
	}

	/**
	 * パラメータのタイプ名に合わせてパラメータクエリパーツを作成する
	 * 
	 * @param typeName
	 * @param val
	 * @return
	 */
	protected ParamQueryPiece createParamQueryPiece(String typeName,
													Object val) {
		Map<String, Class<?>> typeConvertMap = getTypeConvertMap();

		// パラメータのタイプ名からクラスを取得する
		Class<?> type = null;
		Object convertedVal = null;
		String typeNameVal = typeName;
		boolean likeFirst = false;
		boolean likeLast = false;
		if (typeNameVal.startsWith("%")) {
			typeNameVal = typeNameVal.substring("%".length());
			likeFirst = true;
		}
		if (typeNameVal.endsWith("%")) {
			typeNameVal = typeNameVal.substring(0,
					typeNameVal.length() - "%".length());
			likeLast = true;
		}

		if (typeNameVal != null && typeConvertMap.containsKey(typeNameVal)) {
			type = typeConvertMap.get(typeNameVal);
			switch (typeNameVal) {
			case "text":
				convertedVal = String.valueOf(val);
				break;
			case "text[]":
				if (val != null) {
					String[] strArr = (String[]) val;
					List<String> strList = new ArrayList<String>();
					for (String item : strArr) {
						strList.add(item);
					}
					convertedVal = strList.toArray(new String[0]);
				} else {
					convertedVal = null;
				}
				break;
			case "int":
				convertedVal = new Integer(String.valueOf(val));
				break;
			case "int[]":
				if (val != null) {
					String[] strArr = (String[]) val;
					List<Integer> intList = new ArrayList<Integer>();
					for (String item : strArr) {
						intList.add(new Integer(item));
					}
					convertedVal = intList.toArray(new Integer[0]);
				} else {
					convertedVal = null;
				}
				break;
			case "jsonb":
				convertedVal = String.valueOf(val);
				break;
			default:
				convertedVal = val;
			}
		} else {
			type = String.class;
			convertedVal = val;
		}

		if (likeFirst) {
			convertedVal = "%" + convertedVal;
		}
		if (likeLast) {
			convertedVal = convertedVal + "%";
		}

		return new ParamQueryPiece(type, convertedVal);
	}

	protected Map<String, Class<?>> getTypeConvertMap() {
		Map<String, Class<?>> typeConvertMap = new HashMap<>();
		typeConvertMap.put("int", Integer.class);
		typeConvertMap.put("int[]", Integer[].class);
		typeConvertMap.put("text", String.class);
		typeConvertMap.put("text[]", String[].class);
		typeConvertMap.put("jsonb", String.class);
		return typeConvertMap;
	}
}
