package jp.ill.photon.dao;

import java.lang.reflect.Array;
import java.util.List;
import java.util.Map;

import jp.ill.photon.dao.builder.QueryPiece;
import jp.ill.photon.util.JsonUtil;

import org.seasar.doma.Dao;
import org.seasar.doma.MapKeyNamingType;
import org.seasar.doma.jdbc.Config;
import org.seasar.doma.jdbc.builder.DeleteBuilder;
import org.seasar.doma.jdbc.builder.InsertBuilder;
import org.seasar.doma.jdbc.builder.SelectBuilder;
import org.seasar.doma.jdbc.builder.UpdateBuilder;

@Dao(config = DomaConfig.class)
public interface BaseDao {
	/**
	 * クエリ部品のリストからクエリを生成して実行する（リスト取得）
	 *
	 * @param queryPieces
	 * @return
	 */
	default List<Map<String, Object>> selectList(List<QueryPiece> queryPieces) {
		Config config = Config.get(this);
		SelectBuilder builder = SelectBuilder.newInstance(config);
		for (QueryPiece piece : queryPieces) {
			if (piece.getTypeName().equals("sql")) {
				builder.sql((String) piece.getValue());
			} else if (piece.getTypeName().equals("prm")) {
				// 配列のときには、要素を一つ一つ展開して
				// ( [要素1], [要素2], [要素3], ... [要素N] )
				// というパラメータを渡す
				if (piece.getValueType().isArray()) {
					Class clazz = piece.getValueType().getComponentType();
					if (piece.getValue() != null) {
						int max = Array.getLength(piece.getValue());
						// Object list = Array.newInstance(clazz, max);
						builder.sql("(");
						for(int i = 0; i < max; i++) {
							if (i > 0) {
								builder.sql(",");
							}
							builder.param(clazz, clazz.cast(Array.get(piece.getValue(), i)));
						}
						builder.sql(")");
					}
				} else {
					builder.param(piece.getValueType(), piece.getValue());
				}
			}
		}
		return builder.getMapResultList(MapKeyNamingType.LOWER_CASE);
	}

	/**
	 * クエリ部品のリストからクエリを生成して実行する（単一データ取得）
	 *
	 * @param queryPieces
	 * @return
	 */
	default Map<String, Object> selectOne(List<QueryPiece> queryPieces) {
		Config config = Config.get(this);
		SelectBuilder builder = SelectBuilder.newInstance(config);
		for (QueryPiece piece : queryPieces) {
			if (piece.getTypeName().equals("sql")) {
				builder.sql((String) piece.getValue());
			} else if (piece.getTypeName().equals("prm")) {
				builder.param(piece.getValueType(), piece.getValue());
			}
		}
		return builder.getMapSingleResult(MapKeyNamingType.LOWER_CASE);
	}

	/**
	 * クエリ部品のリストからクエリを生成して実行する
	 *
	 * @param queryPieces
	 * @return
	 */
	@SuppressWarnings("unchecked")
	default int insert(List<QueryPiece> queryPieces) {
		Config config = Config.get(this);
		InsertBuilder builder = InsertBuilder.newInstance(config);
		for (QueryPiece piece : queryPieces) {
			if (piece.getTypeName().equals("sql")) {
				builder.sql((String) piece.getValue());
			} else if (piece.getTypeName().equals("prm")) {
				// builder.param(piece.getValueType(), piece.getValue());
				if (piece.getValueType().isArray()) {
					if (piece.getValue() != null) {
						builder.sql(JsonUtil.mapToJson(piece.getValue()));
					}
				} else {
					builder.param(piece.getValueType(), piece.getValue());
				}
			}
		}
		return builder.execute();
	}

	/**
	 * クエリ部品のリストからクエリを生成して実行する（更新）
	 *
	 * @param queryPieces
	 * @return
	 */
	@SuppressWarnings("unchecked")
	default int update(List<QueryPiece> queryPieces) {
		Config config = Config.get(this);
		UpdateBuilder builder = UpdateBuilder.newInstance(config);
		for (QueryPiece piece : queryPieces) {
			if (piece.getTypeName().equals("sql")) {
				builder.sql((String) piece.getValue());
			} else if (piece.getTypeName().equals("prm")) {
				//builder.param(piece.getValueType(), piece.getValue());
				if (piece.getValueType().isArray()) {
					if (piece.getValue() != null) {
						builder.sql(JsonUtil.mapToJson(piece.getValue()));
					}
				} else {
					builder.param(piece.getValueType(), piece.getValue());
				}
			}
		}
		return builder.execute();
	}

	/**
	 * クエリ部品のリストからクエリを生成して実行する(削除)
	 *
	 * @param queryPieces
	 * @return
	 */
	default int delete(List<QueryPiece> queryPieces) {
		Config config = Config.get(this);
		DeleteBuilder builder = DeleteBuilder.newInstance(config);
		for (QueryPiece piece : queryPieces) {
			if (piece.getTypeName().equals("sql")) {
				builder.sql((String) piece.getValue());
			} else if (piece.getTypeName().equals("prm")) {
				builder.param(piece.getValueType(), piece.getValue());
			}
		}
		return builder.execute();
	}
}
