package jp.ill.photon.action;

import java.io.File;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import jp.ill.photon.util.JsonUtil;
import jp.ill.photon.util.ParamUtil;
import jp.ill.photon.util.StringUtil;

/**
 * アクションのパラメータを管理するクラス モジュール内では内容変更不可
 *
 * @author h_tanaka
 *
 */
public class ActionParamMap {

	private Map<String, String[]> params;

	private Map<String, FileParam[]> fileParams;

	/**
	 * 指定したキーの値を取得する。存在しない場合は空文字を返す。
	 *
	 * @param key
	 * @return
	 */
	public String get(String key) {
		Map<String, String[]> params = getParams();
		if (params.containsKey(key)) {
			String[] values = params.get(key);
			if (values != null && values.length > 0) {
				return values[0];
			}
		}

		return "";
	}

	/**
	 * 指定したキーの値を取得する。存在しない場合は空文字を返す。複数値ある場合は、カンマ区切りで返す
	 *
	 * @param key
	 * @return
	 */
	public String getJoined(String key) {
		Map<String, String[]> params = getParams();
		if (params.containsKey(key)) {
			String[] values = params.get(key);
			if (values != null && values.length > 0) {
				return String.join(",", values);
			}
		}

		return "";
	}

	/**
	 * 指定したキーのファイルパラメータを取得する。存在しない場合はnullを返す
	 *
	 * @param key
	 * @return
	 */
	public FileParam getFile(String key) {
		Map<String, FileParam[]> params = getFileParams();
		if (params.containsKey(key)) {
			FileParam[] values = params.get(key);
			if (values != null && values.length > 0) {
				return values[0];
			}
		}

		return null;
	}

	/**
	 * 指定したキーパスのファイルパラメータ情報を取得する
	 *
	 * @param key
	 * @return
	 */
	public String getFileValue(String key) {
		String value = "";
		if (key.contains(".")) {
			String[] chunks = key.split("\\.");
			String paramKey = chunks[0];
			String restKey = key.replaceFirst("[^\\.]*\\.", "");
			if (getFile(paramKey) == null) {
				value = null;
			} else {
				if ("json".equals(restKey)) {
					Map<String, Object> mp = new HashMap<String, Object>();
					mp.put("file_name", invokeFileParamValue(getFile(paramKey),
							"file_name"));
					mp.put("absolute_path", invokeFileParamValue(
							getFile(paramKey), "file.absolute_path"));
					value = JsonUtil.mapToJson(mp);
				} else {
					value = invokeFileParamValue(getFile(paramKey), restKey);
				}
			}
		} else {
			value = ((getFile(key) == null) ? null
					: getFile(key).getFile().getAbsolutePath());
		}

		return value;
	}

	/**
	 * パラメータをString型配列として取得する。
	 *
	 * @param key
	 * @return
	 */
	public String[] getValues(String key) {
		Map<String, String[]> params = getParams();
		return params.getOrDefault(key, new String[] {});
	}

	/**
	 * ファイルパラメータをFileParam型配列として取得する
	 *
	 * @param key
	 * @return
	 */
	public FileParam[] getFileParams(String key) {
		Map<String, FileParam[]> params = getFileParams();
		return params.getOrDefault(key, new FileParam[] {});
	}

	/**
	 * ファイルパラメータの情報をString型配列として取得する
	 *
	 * @param key
	 * @return
	 */
	public String[] getFileParamValues(String key) {
		List<String> values = new ArrayList<>();

		if (key.contains(".")) {
			String[] chunks = key.split("\\.");
			String paramKey = chunks[0];
			String restKey = key.replaceFirst("[^\\.]*\\.", "");

			FileParam[] fps = getFileParams(paramKey);
			for (FileParam fp : fps) {
				values.add(invokeFileParamValue(fp, restKey));
			}
		} else {
			FileParam[] fps = getFileParams(key);
			for (FileParam fp : fps) {
				values.add(fp.getFile().getAbsolutePath());
			}
		}

		return values.toArray(new String[] {});
	}

	/**
	 *
	 * 配列の添え字付でsubmitされたときの値を返却.<br />
	 * （2次元以降の配列は、本メソッド対象外）<br />
	 *
	 * 例）< input type="text" name="item_cd[0]" value="0000"><br />
	 * < input type="text" name="item_cd[1]" value="0001"><br />
	 *
	 * ↓<br />
	 *
	 * item_cd: ["0000", "0001"]<br />
	 *
	 * @param key
	 * @return リスト
	 *
	 */
	public List<String> getListValues(String key) {

		List<String> ret = null;

		Map<String, String[]> params = getParams();

		Map<Integer, Object> mp = new HashMap<Integer, Object>();

		for (Map.Entry<String, String[]> e : params.entrySet()) {

			String paramName = e.getKey();
			// パラメータ名が、<<keyの値>> + "[" + <<数値（0～）>> + "]"でおわる値のみを取得
			// （2次元以降の配列は、本メソッド対象外）
			Pattern p = Pattern.compile("^" + key + "\\[(0|[1-9][0-9]*)\\]$");
			Matcher m = p.matcher(paramName);
			if (m.find()) {

				// 添え字を数値化
				String idxStr = (m.groupCount() == 1) ? m.group(1) : "";
				if (idxStr != null && !idxStr.isEmpty()) {

					Object val = null;
					if (e.getValue() != null) {
						val = e.getValue()[0];
					}
					mp.put(Integer.parseInt(idxStr), val);

				}

			}

		}

		if (!mp.isEmpty()) {

			// 返却用配列の要素にセット
			ret = new ArrayList<>(Collections
					.nCopies(Collections.max(mp.keySet()) + 1, null)); // キーの個数分のnull要素をもつリストを作成
			for (Map.Entry<Integer, Object> e : mp.entrySet()) {
				ret.set(e.getKey(), (String) e.getValue());
			}

		}

		return ret;

	}

	/**
	 *
	 * 配列の添え字付でsubmitされたときの値を返却.<br />
	 * （2次元以降の配列は、本メソッド対象外）<br />
	 *
	 * 例）< input type="text" name="item_cd[0]" value="0000"><br />
	 * < input type="text" name="item_cd[1]" value="0001"><br />
	 *
	 * ↓<br />
	 *
	 * item_cd: ["0000", "0001"]<br />
	 *
	 * @param key
	 * @return リスト
	 *
	 */
	public List<String> getListFileValues(String key) {

		List<String> ret = null;

		Map<String, FileParam[]> params = getFileParams();
		// 「項目名.absolute_path」や「項目名.file_name」が来ることを想定
		String refKey = "";
		String suffix = "";
		if (key.contains(".")) {
			String[] arr = key.split("\\.");
			refKey = arr[0];
			StringJoiner sj = new StringJoiner(".");
			for (int i = 1; i < arr.length; i++) {
				sj.add(arr[i]);
			}
			suffix = sj.toString();
		} else {
			refKey = key;
		}

		Map<Integer, Object> mp = new HashMap<Integer, Object>();

		for (Map.Entry<String, FileParam[]> e : params.entrySet()) {

			String paramName = e.getKey();
			// パラメータ名が、<<keyの値>> + "[" + <<数値（0～）>> + "]"でおわる値のみを取得
			// （2次元以降の配列は、本メソッド対象外）
			Pattern p = Pattern
					.compile("^" + refKey + "\\[(0|[1-9][0-9]*)\\]$");
			Matcher m = p.matcher(paramName);
			if (m.find()) {

				// 添え字を数値化
				String idxStr = (m.groupCount() == 1) ? m.group(1) : "";
				if (idxStr != null && !idxStr.isEmpty()) {

					mp.put(Integer.parseInt(idxStr), e.getKey() + "." + suffix);

				}

			}

		}

		if (!mp.isEmpty()) {

			// 返却用配列の要素にセット
			ret = new ArrayList<>(Collections
					.nCopies(Collections.max(mp.keySet()) + 1, null)); // キーの個数分のnull要素をもつリストを作成
			for (Map.Entry<Integer, Object> e : mp.entrySet()) {
				ret.set(e.getKey(), (String) e.getValue());
			}

		}

		return ret;

	}

	/**
	 *
	 * 配列の添え字付でsubmitされたときの値を返却.<br />
	 * （2次元以降の配列は、本メソッド対象外）<br />
	 *
	 * 例）< input type="text" name="item_cd[0]" value="0000"><br />
	 * < input type="text" name="item_cd[1]" value="0001"><br />
	 *
	 * ↓<br />
	 *
	 * item_cd: ["0000", "0001"]<br />
	 *
	 * @param key
	 * @return リスト
	 *
	 */
	public List<String[]> getListMultiValues(String key) {

		List<String[]> ret = null;

		Map<String, String[]> params = getParams();

		Map<Integer, Object> mp = new HashMap<Integer, Object>();

		for (Map.Entry<String, String[]> e : params.entrySet()) {

			String paramName = e.getKey();
			// パラメータ名が、<<keyの値>> + "[" + <<数値（0～）>> + "]"でおわる値のみを取得
			// （2次元以降の配列は、本メソッド対象外）
			Pattern p = Pattern.compile("^" + key + "\\[(0|[1-9][0-9]*)\\]$");
			Matcher m = p.matcher(paramName);
			if (m.find()) {

				// 添え字を数値化
				String idxStr = (m.groupCount() == 1) ? m.group(1) : "";
				if (idxStr != null && !idxStr.isEmpty()) {

					mp.put(Integer.parseInt(idxStr), e.getValue());

				}

			}

		}

		if (!mp.isEmpty()) {

			// 返却用配列の要素にセット
			ret = new ArrayList<>(Collections
					.nCopies(Collections.max(mp.keySet()) + 1, null)); // キーの個数分のnull要素をもつリストを作成
			for (Map.Entry<Integer, Object> e : mp.entrySet()) {
				ret.set(e.getKey(), (String[]) e.getValue());
			}

		}

		return ret;

	}

	/**
	 *
	 * 配列の添え字付でsubmitされたときの値を返却.<br />
	 * （2次元以降の配列は、本メソッド対象外）<br />
	 *
	 * 例）< input type="text" name="item_cd[0]" value="0000"><br />
	 * < input type="text" name="item_cd[1]" value="0001"><br />
	 *
	 * ↓<br />
	 *
	 * item_cd: ["0000", "0001"]<br />
	 *
	 * @param key
	 * @return 文字列型の配列
	 *
	 */
	public String[] getArrayValues(String key) {

		List<String> list = getListValues(key);

		return (list != null && !list.isEmpty()) ? list.toArray(new String[0])
				: null;

	}

	/**
	 * FileParamオブジェクトから指定pathKeyのメソッドを呼び出して返り値を取得する
	 *
	 * @param fileParam
	 * @param pathKey
	 * @return
	 */
	protected String invokeFileParamValue(FileParam fileParam, String pathKey) {
		String value = "";
		String[] chunks = pathKey.split("\\.");

		String methodName = null;
		Method getterMethod = null;
		Object obj = fileParam;
		try {
			for (int i = 0; i < chunks.length; i++) {
				methodName = ParamUtil
						.getGetterName(ParamUtil.snakeToCamel(chunks[i]));
				getterMethod = obj.getClass().getMethod(methodName);
				obj = getterMethod.invoke(obj);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		if (obj != null) {
			value = StringUtil.defaultString(obj, "");
		}

		return value;
	}

	/**
	 *
	 * 配列の添え字付でsubmitされたときの値を返却.<br />
	 * （2次元以降の配列は、本メソッド対象外）<br />
	 *
	 * 例）< input type="text" name="item_cd[0]" value="0000"><br />
	 * < input type="text" name="item_cd[1]" value="0001"><br />
	 *
	 * ↓<br />
	 *
	 * item_cd: ["0000", "0001"]<br />
	 *
	 * @param key
	 * @return 文字列型の配列
	 *
	 */
	public String[][] getArrayMultiValues(String key) {

		List<String[]> list = getListMultiValues(key);

		return (list != null && !list.isEmpty())
				? list.toArray(new String[0][0]) : null;

	}

	/**
	 * 指定したキーパスのファイルパラメータ情報を取得する.<br />
	 * （2次元以降の配列は、本メソッド対象外）<br />
	 *
	 * 例）< input type="file" name="item_image[0]" /><br />
	 * < input type="file" name="item_image[1]" /><br />
	 *
	 * ↓<br />
	 *
	 * item_image: [パス, パス]<br />
	 *
	 * @param key
	 * @return
	 */
	public String[] getArrayFileValue(String key) {

		List<String> list = getListFileValues(key);
		List<String> ret = new ArrayList<String>();

		if (list != null) {
			for (String listKey : list) {
				ret.add(getFileValue((listKey == null) ? "" : listKey));
			}
		}

		return (ret != null && !ret.isEmpty()) ? ret.toArray(new String[0])
				: null;

	}

	/**
	 * パラメータのキー一覧を取得する
	 *
	 * @return
	 */
	public List<String> getKeys() {
		Map<String, String[]> params = getParams();
		return params.keySet().stream().collect(Collectors.toList());
	}

	public List<String> getFileKeys() {
		Map<String, FileParam[]> params = getFileParams();
		return params.keySet().stream().collect(Collectors.toList());
	}

	/**
	 * パラメータをセットする。
	 *
	 * @param key
	 * @param val
	 */
	public void put(String key, String val) {
		getParams().put(key, new String[] { val });
	}

	public void putFile(String key,
						File file,
						String contentType,
						String fileName) {
		FileParam fp = new FileParam(file, contentType, fileName);
		getFileParams().put(key, new FileParam[] { fp });
	}

	public Map<String, String[]> getParams() {
		if (params == null) {
			params = new HashMap<>();
		}
		return params;
	}

	public Map<String, String> getSingleValueParams() {
		List<String> keys = getKeys();
		Map<String, String> singleParams = new LinkedHashMap<>();
		for (String key : keys) {
			singleParams.put(key, get(key));
		}

		return singleParams;
	}

	public Map<String, String> getJoinedValueParams() {
		List<String> keys = getKeys();
		Map<String, String> singleParams = new LinkedHashMap<>();
		for (String key : keys) {
			singleParams.put(key, getJoined(key));
		}

		return singleParams;
	}

	public void setParams(Map<String, String[]> params) {
		this.params = params;
	}

	public Map<String, FileParam[]> getFileParams() {
		if (fileParams == null) {
			fileParams = new HashMap<>();
		}
		return fileParams;
	}

	public void setFileParams(Map<String, FileParam[]> fileParams) {
		this.fileParams = fileParams;
	}

}
